/* doscan - Denial Of Service Capable Auditing of Networks
 * Copyright (C) 2003 Florian Weimer
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/* This protocol module scans for hosts which are vulnerable to
 * CAN-2003-0715, CAN-2003-0528, CAN-2003-0605 (Microsoft Security
 * Bulletin MS03-039, "Buffer Overrun In RPCSS Service Could Allow
 * Code Execution").
 *
 * For more information on the vulnerability, see:
 *
 *   <http://www.microsoft.com/technet/security/bulletin/MS03-039.asp>
 *
 * This protocol module is based on a Nessus plugin by Tenable Network
 * Security:
 *
 *   <http://cgi.nessus.org/plugins/dump.php3?id=11835>
 *
 */

#include "config.h"
#include "engine_tcp.h"
#include "opt.h"
#include "proto.h"
#include "results.h"
#include "scan.h"

#include <cstdio>
#include <cstring>

typedef void (*receive_callback_t) (scan_host_t *, const char*, unsigned);

typedef unsigned char error_t[4];

struct closure {
  engine_tcp_t engine;
  receive_callback_t receive_callback;
  unsigned expected_length;

  char *bind_data;              /* for deferred send operations */
  unsigned bind_length;
  char *request_data;
  unsigned request_length;

  error_t error_1;
  error_t error_2;
  error_t error_3;
  error_t error_4;
  /* error_5 is SMB-based, not included here */
  error_t error_6;
  error_t error_7;
};

static bool rpc_start (subnets&);
static void rpc_open (scan_host_t *);
static void free_buffers (scan_host_t *);

static void rpc_bind(scan_host_t *, const char *, unsigned,
                     receive_callback_t);
static void rpc_bind_send (scan_host_t *);
static void rpc_receive_header (scan_host_t *);
static void rpc_receive_header_cmpl (scan_host_t *, const char *, unsigned );
static void rpc_receive_bind_cmpl (scan_host_t *, const char *, unsigned);
static void rpc_receive_reply (scan_host_t *);
static void rpc_receive_reply_cmpl (scan_host_t *, const char *, unsigned );
static void rpc_receive_request_cmpl (scan_host_t *, const char *, unsigned);

static void test_98me_start (scan_host_t *);
static void test_98me_result (scan_host_t *, const char *, unsigned);

static void test_1_start (scan_host_t *);
static void test_1_result (scan_host_t *, const char *, unsigned);

static void test_2_start (scan_host_t *);
static void test_2_result (scan_host_t *, const char *, unsigned);

static void test_3_start (scan_host_t *);
static void test_3_result (scan_host_t *, const char *, unsigned);

static void test_4_start (scan_host_t *);
static void test_4_result (scan_host_t *, const char *, unsigned);

static void test_6_start (scan_host_t *);
static void test_6_result (scan_host_t *, const char *, unsigned);

static void test_7_start (scan_host_t *);
static void test_7_result (scan_host_t *, const char *, unsigned);

static void interpret_errors (scan_host_t *);

static int show_all;
static int show_debug;

void
proto_ms_rpc_dcom_register (void)
{
  proto_register ("ms_rpc_dcom", rpc_start, rpc_open);
}

static bool
rpc_start (subnets&)
{
  if (opt_banner_size != 0) {
    fprintf (stderr, "%s: ms_rpc protocol module does not support --banner\n",
             opt_program);
    exit (EXIT_FAILURE);
  }

  if (opt_send && (opt_send[0] != '\0')) {
    fprintf (stderr, "%s: ms_rpc protocol module does not support --send\n",
             opt_program);
    exit (EXIT_FAILURE);
  }

  show_all = 0;
  show_debug = 0;
  if (opt_receive && (opt_receive[0] != '\0')) {
    if (strcmp (opt_receive, "all") == 0) {
      show_all = 1;
    } else if (strcmp (opt_receive, "debug") == 0) {
      show_all = 1;
      show_debug = 1;
    } else {
      fprintf (stderr, "%s: --receive argument '%s' not recognized\n",
             opt_program, opt_receive);
      exit (EXIT_FAILURE);
    }
  }

  return true;
}

static void
rpc_open (scan_host_t *s)
{
  struct closure *c;
  if (!s->state) {
    s->state = malloc (sizeof (struct closure));
  }

  c = static_cast<closure*>(s->state);
  c->bind_data = 0;
  c->bind_length = 0;
  c->request_data = 0;
  c->request_length = 0;

  test_98me_start (s);
}

static void
free_buffers (scan_host_t *s)
{
  struct closure *c = static_cast<closure*>(s->state);

  free (c->bind_data);
  c->bind_data = 0;
  free (c->request_data);
  c->request_data = 0;
}

static void
rpc_bind_and_request (scan_host_t *s, const char *bind, unsigned bind_length,
                      const char *request, unsigned request_length,
                      receive_callback_t callback)
{
  struct closure *c = static_cast<closure*>(s->state);

  c->bind_data = static_cast<char*>(realloc(c->bind_data, bind_length));
  c->bind_length = bind_length;
  memcpy(c->bind_data, bind, bind_length);


  c->request_data = static_cast<char*>(realloc(c->request_data,
                                              request_length));
  c->request_length = request_length;
  if (request) {
    memcpy(c->request_data, request, request_length);
  }

  c->receive_callback = callback;
  engine_tcp_open (s, rpc_bind_send, free_buffers);
}

static void
rpc_bind (scan_host_t *s, const char *data, unsigned length,
          receive_callback_t callback)
{
  rpc_bind_and_request (s, data, length, 0, 0, callback);
}


static void
rpc_bind_send (scan_host_t *s)
{
  struct closure *c = static_cast<closure*>(s->state);

  engine_tcp_send (s, c->bind_data, c->bind_length, ENGINE_TCP_NO_COPY,
                   rpc_receive_header);
}

static void
rpc_receive_header (scan_host_t *s)
{
  engine_tcp_receive_packet (s, 10, rpc_receive_header_cmpl);
}

static void
rpc_receive_header_cmpl (scan_host_t *s, const char *buffer, unsigned size)
{
  struct closure *c = static_cast<closure*>(s->state);
  unsigned data_length;

  if (size != 10) {
    results_add (ticks_get_cached (), s->host, RESULTS_ERROR_NOMATCH,
                 buffer, size);
    engine_tcp_close (s);
    return;
  }

  data_length = (unsigned char)buffer[8] + 256 * (unsigned char)buffer[9];
  c->expected_length = data_length - 10;
  engine_tcp_receive_packet (s, c->expected_length, rpc_receive_bind_cmpl);
}

static void
rpc_receive_bind_cmpl (scan_host_t *s, const char *buffer, unsigned size)
{
  struct closure *c = static_cast<closure*>(s->state);

  if (size != c->expected_length) {
    if (size > 10 && memcmp(buffer, "\000\000AVSS", 6) == 0) {
      if (show_all) {
        char buf[100];
        unsigned len;

        len = sprintf (buf, "AIX DCE Service");
        results_add (ticks_get_cached (), s->host, 0, buf, len);
      }

    } else {
      results_add (ticks_get_cached (), s->host, RESULTS_ERROR_NOMATCH,
                   buffer, size);
    }

    engine_tcp_close (s);
    return;
  }

  if (c->request_length) {
    engine_tcp_send (s, c->request_data, c->request_length,
                     ENGINE_TCP_NO_COPY, rpc_receive_reply);
  } else {
    c->receive_callback(s, buffer, size);
  }
}

static void
rpc_receive_reply (scan_host_t *s)
{
  engine_tcp_receive_packet (s, 10, rpc_receive_reply_cmpl);
}

static void
rpc_receive_reply_cmpl (scan_host_t *s, const char *buffer, unsigned size)
{
  struct closure *c = static_cast<closure*>(s->state);
  unsigned data_length;

  if (size != 10) {
    results_add (ticks_get_cached (), s->host, RESULTS_ERROR_NOMATCH,
                 buffer, size);
    engine_tcp_close (s);
    return;
  }

  data_length = (unsigned char)buffer[8] + 256 * (unsigned char)buffer[9];
  c->expected_length = data_length - 10;
  engine_tcp_receive_packet (s, c->expected_length, rpc_receive_request_cmpl);
}

static void
rpc_receive_request_cmpl (scan_host_t *s, const char *buffer, unsigned size)
{
  struct closure *c = static_cast<closure*>(s->state);

  if (size != c->expected_length) {
    results_add (ticks_get_cached (), s->host, RESULTS_ERROR_NOMATCH,
                 buffer, size);
    engine_tcp_close (s);
    return;
  }

  c->receive_callback(s, buffer, size);
}

static void
test_98me_start (scan_host_t *s)
{
  /* ME test */
  static const char bind[] = {
    0x05, 0x00, 0x0b, 0x03, 0x10, 0x00, 0x00, 0x00, 0x48, 0x00, 0x00,
    0x00, 0x53, 0x53, 0x56, 0x41, 0xd0, 0x16, 0xd0, 0x16, 0x00, 0x00,
    0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0xe6,
    0x73, 0x0c, 0xe6, 0xf9, 0x88, 0xcf, 0x11, 0x9a, 0xf1, 0x00, 0x20,
    0xaf, 0x6e, 0x72, 0xf4, 0x02, 0x00, 0x00, 0x00, 0x04, 0x5d, 0x88,
    0x8a, 0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10,
    0x48, 0x60, 0x02, 0x00, 0x00, 0x00,
  };
  unsigned length = sizeof (bind);

  rpc_bind (s, bind, length, test_98me_result);
}

static void
test_98me_result (scan_host_t *s, const char *reply, unsigned size)
{
  if (size >= 24
      && reply[size - 24] == 0x02
      && reply[size - 23] == 0x00
      && reply[size - 22] == 0x01
      && reply[size - 21] == 0x00) {
    if (show_all) {
      char buf[100];
      unsigned len;

      len = sprintf (buf, "DCOM Service on Windows 98/Me, not vulnerable");
      results_add (ticks_get_cached (), s->host, 0,
                   buf, len);
    }

    engine_tcp_close (s);
    return;
  }

  test_1_start (s);
}

static void
test_1_start (scan_host_t *s)
{
   static char bind [] = {
     0x05, 0x00, 0x0b, 0x03, 0x10, 0x00, 0x00, 0x00, 0x48, 0x00, 0x00,
     0x00, 0x01, 0x00, 0x00, 0x00, 0xd0, 0x16, 0xd0, 0x16, 0x00, 0x00,
     0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0xa0,
     0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x00, 0x00, 0x00,
     0x00, 0x00, 0x00, 0x46, 0x00, 0x00, 0x00, 0x00, 0x04, 0x5d, 0x88,
     0x8a, 0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10,
     0x48, 0x60, 0x02, 0x00, 0x00, 0x00,
   };
   static const unsigned bind_length = sizeof (bind);

   static char request[] = {
     0x05, 0x00, 0x00, 0x03, 0x10, 0x00, 0x00, 0x00, 0xb0, 0x03, 0x00,
     0x00, 0x01, 0x00, 0x00, 0x00, 0x98, 0x03, 0x00, 0x00, 0x00, 0x00,
     0x04, 0x00, 0x05, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
     0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
     0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
     0x00, 0x00, 0x00, 0x00, 0x00, 0x90, 0x05, 0x14, 0x00, 0x68, 0x03,
     0x00, 0x00, 0x68, 0x03, 0x00, 0x00, 0x4d, 0x45, 0x4f, 0x57, 0x04,
     0x00, 0x00, 0x00, 0xa2, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
     0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x46, 0x38, 0x03, 0x00,
     0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00,
     0x00, 0x46, 0x00, 0x00, 0x00, 0x00, 0x38, 0x03, 0x00, 0x00, 0x30,
     0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x10, 0x08, 0x00,
     0xcc, 0xcc, 0xcc, 0xcc, 0xc8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
     0x00, 0x30, 0x03, 0x00, 0x00, 0xd8, 0x00, 0x00, 0x00, 0x00, 0x00,
     0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x07, 0x00, 0x00, 0x00, 0x00,
     0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
     0x00, 0x00, 0x00, 0x00, 0x18, 0x01, 0x8d, 0x00, 0xb8, 0x01, 0x8d,
     0x00, 0x00, 0x00, 0x00, 0x00, 0x07, 0x00, 0x00, 0x00, 0xb9, 0x01,
     0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x00, 0x00, 0x00, 0x00,
     0x00, 0x00, 0x46, 0xab, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
     0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x46, 0xa5, 0x01, 0x00,
     0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00,
     0x00, 0x46, 0xa6, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0,
     0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x46, 0xa4, 0x01, 0x00, 0x00,
     0x00, 0x00, 0x00, 0x00, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
     0x46, 0xad, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x00,
     0x00, 0x00, 0x00, 0x00, 0x00, 0x46, 0xaa, 0x01, 0x00, 0x00, 0x00,
     0x00, 0x00, 0x00, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x46,
     0x07, 0x00, 0x00, 0x00, 0x60, 0x00, 0x00, 0x00, 0x58, 0x00, 0x00,
     0x00, 0x90, 0x00, 0x00, 0x00, 0x58, 0x00, 0x00, 0x00, 0x20, 0x00,
     0x00, 0x00, 0x68, 0x00, 0x00, 0x00, 0x30, 0x00, 0x00, 0x00, 0xc0,
     0x00, 0x00, 0x00, 0x01, 0x10, 0x08, 0x00, 0xcc, 0xcc, 0xcc, 0xcc,
     0x50, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff,
     0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
     0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
     0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
     0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
     0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
     0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
     0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
     0x01, 0x10, 0x08, 0x00, 0xcc, 0xcc, 0xcc, 0xcc, 0x48, 0x00, 0x00,
     0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x5d, 0x88, 0x9a, 0xeb, 0x1c,
     0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10, 0x48, 0x60, 0x10,
     0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
     0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xb8, 0x47, 0x0a,
     0x00, 0x58, 0x00, 0x00, 0x00, 0x05, 0x00, 0x06, 0x00, 0x01, 0x00,
     0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0,
     0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x46, 0xcc, 0xcc, 0xcc, 0xcc,
     0x01, 0x10, 0x08, 0x00, 0xcc, 0xcc, 0xcc, 0xcc, 0x80, 0x00, 0x00,
     0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
     0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x20,
     0xba, 0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x60, 0x00, 0x00, 0x00,
     0x60, 0x00, 0x00, 0x00, 0x4d, 0x45, 0x4f, 0x57, 0x04, 0x00, 0x00,
     0x00, 0xc0, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x00,
     0x00, 0x00, 0x00, 0x00, 0x00, 0x46, 0x3b, 0x03, 0x00, 0x00, 0x00,
     0x00, 0x00, 0x00, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x46,
     0x00, 0x00, 0x00, 0x00, 0x30, 0x00, 0x00, 0x00, 0x01, 0x00, 0x01,
     0x00, 0x67, 0x3c, 0x70, 0x94, 0x13, 0x33, 0xfd, 0x46, 0x87, 0x24,
     0x4d, 0x09, 0x39, 0x88, 0x93, 0x9d, 0x02, 0x00, 0x00, 0x00, 0x00,
     0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
     0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
     0x00, 0x01, 0x10, 0x08, 0x00, 0xcc, 0xcc, 0xcc, 0xcc, 0x48, 0x00,
     0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xb0,
     0x7e, 0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
     0xf0, 0x89, 0x0a, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
     0x00, 0x0d, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0d, 0x00,
     0x00, 0x00, 0x73, 0x00, 0x61, 0x00, 0x6a, 0x00, 0x69, 0x00, 0x61,
     0x00, 0x64, 0x00, 0x65, 0x00, 0x76, 0x00, 0x5f, 0x00, 0x78, 0x00,
     0x38, 0x00, 0x36, 0x00, 0x00, 0x00, 0x08, 0x00, 0xcc, 0xcc, 0xcc,
     0xcc, 0x01, 0x10, 0x08, 0x00, 0xcc, 0xcc, 0xcc, 0xcc, 0x10, 0x00,
     0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
     0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
     0x01, 0x10, 0x08, 0x00, 0xcc, 0xcc, 0xcc, 0xcc, 0x58, 0x00, 0x00,
     0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x5e, 0x0a, 0x00, 0x00, 0x00,
     0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1b,
     0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1b, 0x00, 0x00, 0x00,
     0x5c, 0x00, 0x5c, 0x00, 0x00, 0x00, 0x5c, 0x00, 0x6a, 0x00, 0x69,
     0x00, 0x61, 0x00, 0x64, 0x00, 0x65, 0x00, 0x76, 0x00, 0x5f, 0x00,
     0x78, 0x00, 0x00, 0x00, 0x36, 0x00, 0x5c, 0x00, 0x70, 0x00, 0x75,
     0x00, 0x62, 0x00, 0x6c, 0x00, 0x69, 0x00, 0x63, 0x00, 0x5c, 0x00,
     0x41, 0x00, 0x41, 0x00, 0x41, 0x00, 0x41, 0x00, 0x00, 0x00, 0x00,
     0x00, 0x01, 0x00, 0x15, 0x00, 0x01, 0x10, 0x08, 0x00, 0xcc, 0xcc,
     0xcc, 0xcc, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
     0x00, 0x00, 0x00, 0x90, 0x5b, 0x09, 0x00, 0x02, 0x00, 0x00, 0x00,
     0x01, 0x00, 0x6c, 0x00, 0xc0, 0xdf, 0x08, 0x00, 0x01, 0x00, 0x00,
     0x00, 0x07, 0x00, 0x55, 0x00, 0x00, 0x00, 0x00, 0x00,
   };
   static const unsigned request_length = sizeof (request);

   rpc_bind_and_request (s, bind, bind_length, request, request_length,
                         test_1_result);
}

static void
test_1_result (scan_host_t *s, const char *reply, unsigned size)
{
  struct closure *c = static_cast<closure*>(s->state);

  if (size < sizeof (error_t)) {
    results_add (ticks_get_cached (), s->host, RESULTS_ERROR_NOMATCH,
                 reply, size);
    engine_tcp_close (s);
    return;
  }

  memcpy (c->error_1, reply + size - sizeof (error_t), sizeof (error_t));
  test_2_start (s);
}

static void
test_2_start (scan_host_t *s)
{
   static char bind [] = {
     0x05, 0x00, 0x0b, 0x03, 0x10, 0x00, 0x00, 0x00, 0x48, 0x00, 0x00,
     0x00, 0x01, 0x00, 0x00, 0x00, 0xd0, 0x16, 0xd0, 0x16, 0x00, 0x00,
     0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0xa0,
     0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x00, 0x00, 0x00,
     0x00, 0x00, 0x00, 0x46, 0x00, 0x00, 0x00, 0x00, 0x04, 0x5d, 0x88,
     0x8a, 0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10,
     0x48, 0x60, 0x02, 0x00, 0x00, 0x00,
   };
   static const unsigned bind_length = sizeof (bind);

   static char request[] = {
     0x05, 0x00, 0x00, 0x03, 0x10, 0x00, 0x00, 0x00, 0xb0, 0x03, 0x00,
     0x00, 0x02, 0x00, 0x00, 0x00, 0x98, 0x03, 0x00, 0x00, 0x00, 0x00,
     0x04, 0x00, 0x05, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
     0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
     0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
     0x00, 0x00, 0x00, 0x00, 0x00, 0x90, 0x05, 0x14, 0x00, 0x68, 0x03,
     0x00, 0x00, 0x68, 0x03, 0x00, 0x00, 0x4d, 0x45, 0x4f, 0x57, 0x04,
     0x00, 0x00, 0x00, 0xa2, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
     0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x46, 0x38, 0x03, 0x00,
     0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00,
     0x00, 0x46, 0x00, 0x00, 0x00, 0x00, 0x38, 0x03, 0x00, 0x00, 0x30,
     0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x10, 0x08, 0x00,
     0xcc, 0xcc, 0xcc, 0xcc, 0xc8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
     0x00, 0x30, 0x03, 0x00, 0x00, 0xd8, 0x00, 0x00, 0x00, 0x00, 0x00,
     0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x07, 0x00, 0x00, 0x00, 0x00,
     0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
     0x00, 0x00, 0x00, 0x00, 0x18, 0x01, 0x8d, 0x00, 0xb8, 0x01, 0x8d,
     0x00, 0x00, 0x00, 0x00, 0x00, 0x07, 0x00, 0x00, 0x00, 0xb9, 0x01,
     0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x00, 0x00, 0x00, 0x00,
     0x00, 0x00, 0x46, 0xab, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
     0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x46, 0xa5, 0x01, 0x00,
     0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00,
     0x00, 0x46, 0xf6, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0,
     0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x46, 0xff, 0x01, 0x00, 0x00,
     0x00, 0x00, 0x00, 0x00, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
     0x46, 0xad, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x00,
     0x00, 0x00, 0x00, 0x00, 0x00, 0x46, 0xaa, 0x01, 0x00, 0x00, 0x00,
     0x00, 0x00, 0x00, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x46,
     0x07, 0x00, 0x00, 0x00, 0x60, 0x00, 0x00, 0x00, 0x58, 0x00, 0x00,
     0x00, 0x90, 0x00, 0x00, 0x00, 0x58, 0x00, 0x00, 0x00, 0x20, 0x00,
     0x00, 0x00, 0x68, 0x00, 0x00, 0x00, 0x30, 0x00, 0x00, 0x00, 0xc0,
     0x00, 0x00, 0x00, 0x01, 0x10, 0x08, 0x00, 0xcc, 0xcc, 0xcc, 0xcc,
     0x50, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff,
     0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
     0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
     0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
     0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
     0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
     0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
     0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
     0x01, 0x10, 0x08, 0x00, 0xcc, 0xcc, 0xcc, 0xcc, 0x48, 0x00, 0x00,
     0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x5d, 0x88, 0x9a, 0xeb, 0x1c,
     0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10, 0x48, 0x60, 0x10,
     0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
     0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xb8, 0x47, 0x0a,
     0x00, 0x58, 0x00, 0x00, 0x00, 0x05, 0x00, 0x06, 0x00, 0x01, 0x00,
     0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0,
     0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x46, 0xcc, 0xcc, 0xcc, 0xcc,
     0x01, 0x10, 0x08, 0x00, 0xcc, 0xcc, 0xcc, 0xcc, 0x80, 0x00, 0x00,
     0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
     0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x20,
     0xba, 0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x60, 0x00, 0x00, 0x00,
     0x60, 0x00, 0x00, 0x00, 0x4d, 0x45, 0x4f, 0x57, 0x04, 0x00, 0x00,
     0x00, 0xc0, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x00,
     0x00, 0x00, 0x00, 0x00, 0x00, 0x46, 0x3b, 0x03, 0x00, 0x00, 0x00,
     0x00, 0x00, 0x00, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x46,
     0x00, 0x00, 0x00, 0x00, 0x30, 0x00, 0x00, 0x00, 0x01, 0x00, 0x01,
     0x00, 0x67, 0x3c, 0x70, 0x94, 0x13, 0x33, 0xfd, 0x46, 0x87, 0x24,
     0x4d, 0x09, 0x39, 0x88, 0x93, 0x9d, 0x02, 0x00, 0x00, 0x00, 0x00,
     0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
     0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
     0x00, 0x01, 0x10, 0x08, 0x00, 0xcc, 0xcc, 0xcc, 0xcc, 0x48, 0x00,
     0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xb0,
     0x7e, 0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
     0xf0, 0x89, 0x0a, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
     0x00, 0x0d, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0d, 0x00,
     0x00, 0x00, 0x73, 0x00, 0x61, 0x00, 0x6a, 0x00, 0x69, 0x00, 0x61,
     0x00, 0x64, 0x00, 0x65, 0x00, 0x76, 0x00, 0x5f, 0x00, 0x78, 0x00,
     0x38, 0x00, 0x36, 0x00, 0x00, 0x00, 0x08, 0x00, 0xcc, 0xcc, 0xcc,
     0xcc, 0x01, 0x10, 0x08, 0x00, 0xcc, 0xcc, 0xcc, 0xcc, 0x10, 0x00,
     0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
     0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
     0x01, 0x10, 0x08, 0x00, 0xcc, 0xcc, 0xcc, 0xcc, 0x58, 0x00, 0x00,
     0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x5e, 0x0a, 0x00, 0x00, 0x00,
     0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1b,
     0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1b, 0x00, 0x00, 0x00,
     0x5c, 0x00, 0x5c, 0x00, 0x00, 0x00, 0x5c, 0x00, 0x6a, 0x00, 0x69,
     0x00, 0x61, 0x00, 0x64, 0x00, 0x65, 0x00, 0x76, 0x00, 0x5f, 0x00,
     0x78, 0x00, 0x00, 0x00, 0x36, 0x00, 0x5c, 0x00, 0x70, 0x00, 0x75,
     0x00, 0x62, 0x00, 0x6c, 0x00, 0x69, 0x00, 0x63, 0x00, 0x5c, 0x00,
     0x41, 0x00, 0x41, 0x00, 0x41, 0x00, 0x41, 0x00, 0x00, 0x00, 0x00,
     0x00, 0x01, 0x00, 0x15, 0x00, 0x01, 0x10, 0x08, 0x00, 0xcc, 0xcc,
     0xcc, 0xcc, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
     0x00, 0x00, 0x00, 0x90, 0x5b, 0x09, 0x00, 0x02, 0x00, 0x00, 0x00,
     0x01, 0x00, 0x6c, 0x00, 0xc0, 0xdf, 0x08, 0x00, 0x01, 0x00, 0x00,
     0x00, 0x07, 0x00, 0x55, 0x00, 0x00, 0x00, 0x00, 0x00,
   };
   static const unsigned request_length = sizeof (request);

   rpc_bind_and_request (s, bind, bind_length, request, request_length,
                         test_2_result);
}

static void
test_2_result (scan_host_t *s, const char *reply, unsigned size)
{
  struct closure *c = static_cast<closure*>(s->state);

  if (size < sizeof (error_t)) {
    results_add (ticks_get_cached (), s->host, RESULTS_ERROR_NOMATCH,
                 reply, size);
    engine_tcp_close (s);
    return;
  }

  memcpy (c->error_2, reply + size - sizeof (error_t), sizeof (error_t));
  test_3_start (s);
}

static void
test_3_start (scan_host_t *s)
{
   static char bind [] = {
     0x05, 0x00, 0x0b, 0x03, 0x10, 0x00, 0x00, 0x00, 0x48, 0x00, 0x00,
     0x00, 0x01, 0x00, 0x00, 0x00, 0xd0, 0x16, 0xd0, 0x16, 0x00, 0x00,
     0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0xa0,
     0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x00, 0x00, 0x00,
     0x00, 0x00, 0x00, 0x46, 0x00, 0x00, 0x00, 0x00, 0x04, 0x5d, 0x88,
     0x8a, 0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10,
     0x48, 0x60, 0x02, 0x00, 0x00, 0x00,
   };
   static const unsigned bind_length = sizeof (bind);

   static char request[] = {
     0x05, 0x00, 0x0e, 0x03, 0x10, 0x00, 0x00, 0x00, 0x48, 0x00, 0x00,
     0x00, 0x03, 0x00, 0x00, 0x00, 0xd0, 0x16, 0xd0, 0x16, 0x05, 0xaf,
     0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x01, 0x00, 0xb8,
     0x4a, 0x9f, 0x4d, 0x1c, 0x7d, 0xcf, 0x11, 0x86, 0x1e, 0x00, 0x20,
     0xaf, 0x6e, 0x7c, 0x57, 0x00, 0x00, 0x00, 0x00, 0x04, 0x5d, 0x88,
     0x8a, 0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10,
     0x48, 0x60, 0x02, 0x00, 0x00, 0x00,
   };

   static const unsigned request_length = sizeof (request);

   rpc_bind_and_request (s, bind, bind_length, request, request_length,
                         test_3_result);
};

static void
test_3_result (scan_host_t *s, const char *reply, unsigned size)
{
  struct closure *c = static_cast<closure*>(s->state);

  if (size < sizeof (error_t)) {
    results_add (ticks_get_cached (), s->host, RESULTS_ERROR_NOMATCH,
                 reply, size);
    engine_tcp_close (s);
    return;
  }

  memcpy (c->error_3, reply + size - sizeof (error_t), sizeof (error_t));
  test_4_start (s);
}

static void
test_4_start (scan_host_t *s)
{
  static char bind [] = {
    0x05, 0x00, 0x0b, 0x03, 0x10, 0x00, 0x00, 0x00, 0x48, 0x00, 0x00,
    0x00, 0x01, 0x00, 0x00, 0x00, 0xd0, 0x16, 0xd0, 0x16, 0x00, 0x00,
    0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0xa0,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x46, 0x00, 0x00, 0x00, 0x00, 0x04, 0x5d, 0x88,
    0x8a, 0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10,
    0x48, 0x60, 0x02, 0x00, 0x00, 0x00,
   };
   static const unsigned bind_length = sizeof (bind);

   static char request[] = {
     0x05, 0x00, 0x00, 0x03, 0x10, 0x00, 0x00, 0x00, 0x9a, 0x00, 0x00,
     0x00, 0x03, 0x00, 0x00, 0x00, 0x82, 0x00, 0x00, 0x00, 0x01, 0x00,
     0x00, 0x00, 0x05, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
     0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
     0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
     0x00, 0x95, 0x96, 0x95, 0x2a, 0x8c, 0xda, 0x6d, 0x4a, 0xb2, 0x36,
     0x19, 0xbc, 0xaf, 0x2c, 0x2d, 0xea, 0x34, 0xeb, 0x8f, 0x00, 0x07,
     0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x07, 0x00, 0x00, 0x00,
     0x5c, 0x00, 0x5c, 0x00, 0x4d, 0x00, 0x45, 0x00, 0x4f, 0x00, 0x57,
     0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x5c, 0x00,
     0x48, 0x00, 0x5c, 0x00, 0x48, 0x00, 0x01, 0x00, 0x00, 0x00, 0x58,
     0xe9, 0x8f, 0x00, 0x01, 0x00, 0x00, 0x00, 0x95, 0x96, 0x95, 0x2a,
     0x8c, 0xda, 0x6d, 0x4a, 0xb2, 0x36, 0x19, 0xbc, 0xaf, 0x2c, 0x2d,
     0xea, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x5c, 0x00,
   };

   static const unsigned request_length = sizeof (request);

   rpc_bind_and_request (s, bind, bind_length, request, request_length,
                         test_4_result);
};

static void
test_4_result (scan_host_t *s, const char *reply, unsigned size)
{
  struct closure *c = static_cast<closure*>(s->state);

  if (size < 8) {
    results_add (ticks_get_cached (), s->host, RESULTS_ERROR_NOMATCH,
                 reply, size);
    engine_tcp_close (s);
    return;
  }

  memcpy (c->error_4, reply + size - 8, sizeof (error_t));
  test_6_start (s);
}

static void
test_6_start (scan_host_t *s)
{
  static char bind [] = {
    0x05, 0x00, 0x0b, 0x03, 0x10, 0x00, 0x00, 0x00, 0x48, 0x00, 0x00,
    0x00, 0xde, 0xad, 0xbe, 0xef, 0xd0, 0x16, 0xd0, 0x16, 0x00, 0x00,
    0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0xb8,
    0x4a, 0x9f, 0x4d, 0x1c, 0x7d, 0xcf, 0x11, 0x86, 0x1e, 0x00, 0x20,
    0xaf, 0x6e, 0x7c, 0x57, 0x00, 0x00, 0x00, 0x00, 0x04, 0x5d, 0x88,
    0x8a, 0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10,
    0x48, 0x60, 0x02, 0x00, 0x00, 0x00,
  };
  static const unsigned bind_length = sizeof (bind);

  static char request[] = {
    0x05, 0x00, 0x00, 0x03, 0x10, 0x00, 0x00, 0x00, 0xc6, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0xae, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x05, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x5b, 0x4e, 0x65, 0x73, 0x73, 0x75, 0x73, 0x5d,
    0x5b, 0x4e, 0x65, 0x73, 0x73, 0x75, 0x73, 0x5d, 0x00, 0x00, 0x00,
    0x00, 0x4e, 0x4e, 0x4e, 0x4e, 0x4e, 0x4e, 0x4e, 0x4e, 0x4e, 0x4e,
    0x4e, 0x4e, 0x4e, 0x4e, 0x4e, 0x4e, 0x68, 0x0f, 0x0b, 0x00, 0x1e,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1e, 0x00, 0x00, 0x00,
    0x5c, 0x00, 0x5c, 0x00, 0x41, 0x00, 0x00, 0x00, 0x5c, 0x00, 0x00,
    0x00, 0x63, 0x00, 0x24, 0x00, 0x5c, 0x00, 0x74, 0x00, 0x65, 0x00,
    0x73, 0x00, 0x74, 0x00, 0x5f, 0x00, 0x32, 0x00, 0x30, 0x00, 0x30,
    0x00, 0x33, 0x00, 0x5f, 0x00, 0x6e, 0x00, 0x65, 0x00, 0x73, 0x00,
    0x73, 0x00, 0x75, 0x00, 0x73, 0x00, 0x2e, 0x00, 0x74, 0x00, 0x78,
    0x00, 0x74, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00,
    0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0xb8,
    0xeb, 0x0b, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x07, 0x00,
   };
   static const unsigned request_length = sizeof (request);

   rpc_bind_and_request (s, bind, bind_length, request, request_length,
                         test_6_result);
};

static void
test_6_result (scan_host_t *s, const char *reply, unsigned size)
{
  struct closure *c = static_cast<closure*>(s->state);

  if (size < 24) {
    memset(c->error_6, 0, sizeof (error_t));

  } else {
    memcpy (c->error_6, reply + size - 24, sizeof (error_t));
  }

  test_7_start (s);
}

static void
test_7_start (scan_host_t *s)
{
  static char bind [] = {
    0x05, 0x00, 0x0b, 0x03, 0x10, 0x00, 0x00, 0x00, 0x48, 0x00, 0x00,
    0x00, 0xde, 0xad, 0xbe, 0xef, 0xd0, 0x16, 0xd0, 0x16, 0x00, 0x00,
    0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0xb8,
    0x4a, 0x9f, 0x4d, 0x1c, 0x7d, 0xcf, 0x11, 0x86, 0x1e, 0x00, 0x20,
    0xaf, 0x6e, 0x7c, 0x57, 0x00, 0x00, 0x00, 0x00, 0x04, 0x5d, 0x88,
    0x8a, 0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10,
    0x48, 0x60, 0x02, 0x00, 0x00, 0x00,
  };
  static const unsigned bind_length = sizeof (bind);

  static char request[] = {
    0x05, 0x00, 0x00, 0x03, 0x10, 0x00, 0x00, 0x00, 0xc6, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0xae, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x05, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x5b, 0x4e, 0x65, 0x73, 0x73, 0x75, 0x73, 0x5d,
    0x5b, 0x4e, 0x65, 0x73, 0x73, 0x75, 0x73, 0x5d, 0x00, 0x00, 0x00,
    0x00, 0x48, 0x48, 0x48, 0x48, 0x48, 0x48, 0x48, 0x48, 0x48, 0x48,
    0x48, 0x48, 0x48, 0x48, 0x48, 0x48, 0x68, 0x0f, 0x0b, 0x00, 0x1e,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1e, 0x00, 0x00, 0x00,
    0x5c, 0x00, 0x5c, 0x00, 0x31, 0x00, 0x37, 0x00, 0x32, 0x00, 0x2e,
    0x00, 0x34, 0x00, 0x32, 0x00, 0x2e, 0x00, 0x31, 0x00, 0x34, 0x00,
    0x32, 0x00, 0x2e, 0x00, 0x31, 0x00, 0x34, 0x00, 0x32, 0x00, 0x00,
    0x00, 0x5c, 0x00, 0x74, 0x00, 0x65, 0x00, 0x73, 0x00, 0x74, 0x00,
    0x5f, 0x00, 0x4e, 0x00, 0x65, 0x00, 0x73, 0x00, 0x73, 0x00, 0x75,
    0x00, 0x73, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00,
    0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0xb8,
    0xeb, 0x0b, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x07, 0x00,
   };
   static const unsigned request_length = sizeof (request);

   rpc_bind_and_request (s, bind, bind_length, request, request_length,
                         test_7_result);
};

static void
test_7_result (scan_host_t *s, const char *reply, unsigned size)
{
  struct closure *c = static_cast<closure*>(s->state);

  if (size < 24) {
    memset(c->error_7, 0, sizeof (error_t));

  } else {
    memcpy (c->error_7, reply + size - 24, sizeof (error_t));
  }

  interpret_errors(s);
}

static void
print_error (scan_host_t *s, unsigned number, error_t error)
{
  char buf[100];
  unsigned len;

  len = sprintf (buf, "MS DCOM error code %u: %02X%02X%02X%02X",
                 number, (unsigned)error[0], (unsigned)error[1],
                 (unsigned)error[2], (unsigned)error[3]);
  results_add (ticks_get_cached (), s->host, 0, buf, len);
}

static void
interpret_errors (scan_host_t *s)
{
  struct closure *c = static_cast<closure*>(s->state);
  int err_6_and_7_good;

  if (show_debug) {
    print_error (s, 1, c->error_1);
    print_error (s, 2, c->error_2);
    print_error (s, 3, c->error_3);
    print_error (s, 4, c->error_4);
    print_error (s, 6, c->error_6);
    print_error (s, 7, c->error_7);
  }

  err_6_and_7_good
    = memcmp (c->error_6, "\x54\x01\x04\x80", sizeof (error_t)) == 0
    && memcmp (c->error_7, "\x04\x00\x08\x80", sizeof (error_t)) == 0;

  if (memcmp(c->error_1, "\x00\x00\x00\x00", sizeof (error_t)) == 0
      && memcmp(c->error_2, "\x00\x00\x00\x00", sizeof (error_t)) == 0
      && memcmp(c->error_4, "\x1c\x00\x00\x1c", sizeof (error_t)) == 0) {
    if (show_all) {
      char buf[100];
      unsigned len;

      len = sprintf (buf, "HP-UX dced");
      results_add (ticks_get_cached (), s->host, 0, buf, len);
    }

    engine_tcp_close (s);
    return;
  }
  else if (memcmp(c->error_1, "\x00\x00\x00\x00", sizeof (error_t)) == 0
      && memcmp(c->error_2, "\x00\x00\x00\x00", sizeof (error_t)) == 0
      && memcmp(c->error_4, "\x03\x00\x01\x1c", sizeof (error_t)) == 0) {
    if (show_all) {
      char buf[100];
      unsigned len;

      len = sprintf (buf, "Microsoft DCOM Service, "
                     "not vulnerable (Windows XP SP2)");
      results_add (ticks_get_cached (), s->host, 0, buf, len);
    }

    engine_tcp_close (s);
    return;
  }

  if (memcmp (c->error_1,  c->error_2, sizeof (error_t)) == 0) {
    if (memcmp (c->error_1, "\x05\x00\x07\x80", sizeof (error_t)) == 0) {
      if (show_all) {
        char buf[100];
        unsigned len;

        len = sprintf (buf, "Microsoft DCOM Service, disabled");
        results_add (ticks_get_cached (), s->host, 0, buf, len);
      }

      engine_tcp_close (s);
      return;
    }

    if (err_6_and_7_good) {
      if (show_all) {
        char buf[100];
        unsigned len;

        len = sprintf (buf, "Microsoft DCOM Service, not vulnerable");
        results_add (ticks_get_cached (), s->host, 0, buf, len);
      }

      engine_tcp_close (s);
      return;
    }

    if (memcmp(c->error_6, c->error_7, sizeof (error_t)) == 0
        && memcmp(c->error_6, "\x05\x00\x07\x80", sizeof (error_t)) == 0) {
      if (show_all) {
        char buf[100];
        unsigned len;

        len = sprintf (buf, "Microsoft DCOM Service, disabled");
        results_add (ticks_get_cached (), s->host, 0, buf, len);
      }

      engine_tcp_close (s);
      return;
    }


    {
      char buf[100];
      unsigned len;

      len = sprintf (buf, "Microsoft DCOM Service, vulnerable to MS03-039");
      results_add (ticks_get_cached (), s->host, 0, buf, len);

      engine_tcp_close (s);
      return;
    }

  } else {
    if (err_6_and_7_good) {
      if (show_all) {
        char buf[100];
        unsigned len;

        len = sprintf (buf, "Microsoft DCOM Service, not vulnerable");
        results_add (ticks_get_cached (), s->host, 0, buf, len);
      }

      engine_tcp_close (s);
      return;
    }

    if (show_all) {
      char buf[100];
      unsigned len;

      len = sprintf (buf, "??? Unknown reply, please report as bug");
      results_add (ticks_get_cached (), s->host, 0, buf, len);
    }

    engine_tcp_close (s);
    return;
  }
}

/* arch-tag: def1c409-8960-474c-bbed-647822033f22
 */
