"""
"""

__author__ = "Marie E. Rognes (meg@simula.no)"
__copyright__ = "Copyright (C) 2009 - 2010 Marie E. Rognes"
__license__  = "GNU GPL version 3 or any later version"

# Modified by Anders Logg, 2010.

# Last changed: 2010-06-22

from dolfin import info, info_blue, info_green, info_red
from dolfin.cpp import MeshFunction

def convert_domains_to_mf(sub_domains, mesh):

    if sub_domains is None:
        return (None, None, None)

    dim = mesh.topology().dim()
    mfs = {"cell": None, "exterior": None, "interior": None}
    dims = {"cell": dim, "exterior": dim - 1, "interior": dim - 1}

    # Initialize defaults
    for domain in mfs:
        if domain in sub_domains:
            mfs[domain] = MeshFunction("uint", mesh, dims[domain])
            default = max(sub_domains[domain].keys()) + 1
            mfs[domain].set_all(default)

    # Mark mesh_functions
    for sub_domain in sub_domains:
        for (tag, domain) in sub_domains[sub_domain].iteritems():
            domain.mark(mfs[sub_domain], tag)

    return (mfs["cell"], mfs["exterior"], mfs["interior"])

def convert_to_mesh_function(f):
    """
    Convert a piecewise constant function to a mesh function,
    for plotting purposes.
    """

    assert(f.function_space().ufl_element().degree() == 0)
    assert(f.function_space().ufl_element().value_shape() == ())
    assert(f.vector().size() == f.function_space().mesh().num_cells())

    mesh = f.function_space().mesh()
    mf = MeshFunction("double", mesh, mesh.topology().dim())
    cell_values = f.vector().array()
    for i in xrange(mesh.num_cells()):
        mf[i] = cell_values[i]
    return mf

def print_value(name, value):
    if not value is None:
        info(name + ": " + str(value))

def print_value_green(name, value):
    if not value is None:
        info_green(name + ": " + str(value))

def print_value_red(name, value):
    if not value is None:
        info_red(name + ": " + str(value))

def print_iteration(number):
    info("\n" + "-"*80)
    info_blue("Adaptive iteration %d\n" % number)
    info("-"*80)

def print_stage(string):
    info_blue("\n" + string)
    info("-" * len(string))

def domain_to_mf(facet_domains, mesh):
    # FIXME: Highly specialized.

    if not facet_domains:
        return None

    # Construct markers
    markers = MeshFunction("uint", mesh, mesh.topology().dim() - 1)
    markers.set_all(1)
    facet_domains.mark(markers, 0)
    return markers

def extract_data(data):
    """ data should be a list of Adaptive Data"""

    dofs = []
    estimates = []
    errors = []
    indices = []
    for datum in data:

        if datum.reference is None:
            print "Cannot handle adaptive data without reference value"
            return

        dofs += [datum.num_dofs]
        estimates += [datum.error_estimate]
        actual_error = abs(datum.reference - datum.functional_value)
        errors += [actual_error]
        indices += [datum.error_estimate/actual_error]

    return (dofs, errors, estimates, indices)

def save_data(data, filename):

    (dofs, errors, estimates, indices) = extract_data(data)
    sum_indicators = [datum.sum_indicators for datum in data]
    functional_values = [datum.functional_value for datum in data]

    file = open(filename, 'w')
    file.write("dofs = %s\n" % dofs)
    file.write("errors = %s\n" % errors)
    file.write("estimates = %s\n" % estimates)
    file.write("sum_indicators = %s\n" % sum_indicators)
    file.write("indices = %s\n" % indices)
    file.write("functional_values = %s\n" % functional_values)
    file.close()

