"""
Module containing functionality for computing residuals and residual
representations
"""

__author__ = "Marie E. Rognes (meg@simula.no)"
__date__ = "2009-11-04"
__copyright__ = "Copyright (C) 2009 - Marie E. Rognes"
__license__  = "GNU GPL version 3 or any later version"

# Modified by Anders Logg, 2009.
# Last changed: 2010-05-09

from ufl.algorithms import preprocess

from dolfin import lhs, rhs, dx, ds, dS, avg, inner, replace, action
from dolfin import info, assemble, plot
from dolfin import VariationalProblem, TrialFunction, TestFunction, Function

from dolfin.cpp import solve

from dolfin.adaptivity.specialfunctions import *
from dolfin.adaptivity.formmanipulation import tear, increase_order

import numpy

__all__ = ["residual", "compute_residual_representation", "bank_weiser",
           "compute_cell_residual", "compute_facet_residual"]

def residual(F, u_h):
    """
    r = lambda w: action(F, w) doesn't quite work
    """
    processed = preprocess(F)
    arguments = processed.form_data().original_arguments

    # If bilinear, replace TrialFunction by given u_h
    if len(arguments) == 2:
        u = arguments[1]
        F = replace(F, {u: u_h})

    v = arguments[0]
    return lambda w: replace(F, {v:w})

def compute_residual_representation(F, u_h):
    """
    For a given problem F = 0 and a given approximation u_h,
    compute the 'L^2' residual representation (R_T, R_dT).
    """

    info("Computing residual representation")

    # Extract residual: r(.) = F(u_h; .)
    r = residual(F, u_h)

    # Compute cell residual representation
    V_h = u_h.function_space()
    R_T = compute_cell_residual(V_h, r)

    # Compute facet residual representation
    R_dT = compute_facet_residual(V_h, r, R_T)

    return (R_T, R_dT)

def compute_cell_residual(V_h, r):
    """
    For a given residual r = 0 and a function_space V_h,
    compute the 'L^2' cell residual representation R_T.
    """

    # Note: We could do this locally
    info("Computing cell residual")

    # Construct cell bubble b_T on each cell
    b_T = Bubble(V_h.mesh())

    # Define variational problem for cell residual:
    W_h = tear(V_h)
    u = TrialFunction(W_h)
    v = TestFunction(W_h)

    v_T = b_T*v
    a = inner(v_T, u)*dx
    l = r(v_T)

    # Solve for cell residual
    pde = VariationalProblem(a, l)
    pde.parameters["linear_solver"] = "iterative"
    R_T = pde.solve()
    return R_T

def compute_facet_residual(V_h, r, R_T):
    """

    For a given residual r, a function_space V_h, and R_T, compute the
    'L^2' facet residual representation R_dT.

    For each cell T, the facet residuals R_e are determined by the
    equations

    \int_e b_e|T (v|T, R_e|T) = r(b_e|T v|_T) - b_e*inner(v|_T, R_T)*dx
    """

    info("Computing facet residuals")

    W_h = tear(V_h)
    u = TrialFunction(W_h)
    v = TestFunction(W_h)

    # Functions for R_e (one for each local facet)
    mesh = W_h.mesh()
    num_facets = mesh.topology().dim() + 1
    R_e = [Function(W_h) for i in range(num_facets)]

    # Compute each R_e
    for e in range(num_facets):

        # Initialize bubble on edge number e
        b_e = FacetBubble(mesh, e)

        # Define forms corresponding to above equations
        v_e = b_e*v
        a = (inner(v_e('+'), u('+')) + inner(v_e('-'), u('-')))*dS \
            + inner(v_e, u)*ds
        l = r(v_e) - inner(v_e, R_T)*dx

        # Assemble matrices
        A_e = assemble(a)
        L_e = assemble(l)

        # Non-singularize:
        _tmp_constrain_spaces(A_e, L_e)

        # Solve
        solve(A_e, R_e[e].vector(), L_e, "gmres")

    # Combine R_e to form R_dT (Fascinating hack inspired by
    # FacetNormal)

    R_dT = SpecialFacetFunction(element=W_h.ufl_element())
    R_dT.set(R_e)

    return R_dT

def bank_weiser(F, R_T, R_dT):

    # Create element space of increased polynomial degree
    W_h = increase_order(R_T.function_space())

    # Define test and trial functions
    u = TrialFunction(W_h)
    v = TestFunction(W_h)

    # Define the forms for the local problems
    a = lhs(F) # FIXME: Linear only

    preprocessed = preprocess(F)
    (v0, u0) = preprocessed.form_data().original_arguments

    a_local = replace(a, {v0: v, u0: u})
    l_local = inner(v, R_T)*dx + inner(v, R_dT)*ds \
              + inner(v('+'), avg(R_dT))*dS + inner(v('-'), avg(R_dT))*dS

    # Assemble matrices
    A = assemble(a_local)
    b = assemble(l_local)

    # Remove the basis functions corresponding to P_1
    _tmp_modulo_P1(A, b, W_h.mesh().cells())

    # Solve linear system
    R_BW = Function(W_h)
    solve(A, R_BW.vector(), b)

    # Return Bank-Weiser representation
    return R_BW


def _tmp_modulo_P1(A, b, cells):
    P2s = [i*6 + j for j in [0, 2, 5] for i in range(len(cells))]
    dofs = numpy.array(P2s, dtype="I")
    A.ident(dofs)
    b[dofs] = 0.0

def _tmp_constrain_spaces(A, b):
    """
    Super-hack
    """
    dofs = [i for i in range(A.size(0)) if A[i, i] < 1.e-10]
    dofs = numpy.array(dofs, dtype="I")
    A.ident(dofs)
    b[dofs] = 0.0

def spy(A):
    for i in range(A.size(0)):
        for j in range(A.size(1)):
            if abs(A[i, j]) > 1e-10:
                print "x",
            else:
                print " ",
        print

