# Copyright (C) 2008 Anders Logg
#
# This file is part of DOLFIN.
#
# DOLFIN is free software: you can redistribute it and/or modify
# it under the terms of the GNU Lesser General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# DOLFIN is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with DOLFIN.  If not, see <http://www.gnu.org/licenses/>.
#
# Modified by Johan Hake, 2009
#
# First added:  2008-09-22
# Last changed: 2009-07-31

__all__ = ["errornorm"]

from ufl import FiniteElement, VectorElement

import dolfin.cpp as cpp

from dolfin.cpp import error, Vector
from dolfin.fem.norm import norm
from dolfin.fem.interpolate import interpolate
from dolfin.functions.function import Function
from dolfin.functions.functionspace import FunctionSpace, VectorFunctionSpace

dim_to_shape = {1: "interval", 2: "triangle", 3: "tetrahedron"}

def errornorm(u, uh, norm_type="l2", degree=3, mesh=None):
    """Compute the error e = u - uh in the given norm. The parameter k
    denotes the degree of accuracy (degree of piecewise polynomials
    approximating u an uh).

    In simple cases, one may just define

      e = u - uh

    and evalute for example the square of the error in the L2 norm by

      e = u - uh
      assemble(e*e*dx, mesh)

    However, this is not stable w.r.t. round-off errors considering that
    the form compiler may expand the expression above to

      u*u*dx + uh*uh*dx - 2*u*uh*dx

    and this might get further expanded into thousands of terms for
    higher order elements. Thus, the error will be evaluated by adding
    a large number of terms which should sum up to something close to
    zero (if the error is small).

    This module computes the error by first interpolating both u and uh
    to a common space (of high accuracy), then subtracting the two fields
    (which is easy since they are expressed in the same basis) and then
    evaluating the integral."""

    # Check argument
    if not isinstance(u, cpp.GenericFunction):
        cpp.error("Unable to compute error norm, not a Function or Expression: " + str(u))
    if not isinstance(uh, cpp.GenericFunction):
        cpp.error("Unable to compute error norm, not a Function or Expression: " + str(u))

    # Get mesh
    if isinstance(u, Function) and mesh is None:
        mesh = u.function_space().mesh()
    if isinstance(uh, Function) and mesh is None:
        mesh = uh.function_space().mesh()
    if mesh is None:
        error("Unable to compute error norm, missing mesh.")

    # Get rank
    if not u.value_rank() == uh.value_rank():
        error("Unable to compute error norm, value ranks don't match.")
    rank = u.value_rank()

    # Create function space
    if rank == 0:
        V = FunctionSpace(mesh, "Discontinuous Lagrange", degree)
    elif rank == 1:
        V = VectorFunctionSpace(mesh, "Discontinuous Lagrange", degree)
    else:
        error("Unable to compute error norm, can't handle elements of rank %d." % rank)

    # Interpolate functions into finite element space
    pi_u = interpolate(u, V)
    pi_uh = interpolate(uh, V)

    # Compute the difference
    e = Function(V)
    e.assign(pi_u)
    e.vector().axpy(-1.0, pi_uh.vector())

    # Compute norm
    return norm(e, norm_type=norm_type, mesh=mesh)
