#include <stdio.h>
#include <stdlib.h>

#include "hash.h"

int hash_keyequal(hashkey_t k1, hashkey_t k2) {
	return (k1 == k2);
}

int
hash_hashkey(struct hashtable *hashtable, hashkey_t key)
{
	/* Well, at least this means characters in the same block
	   are put in different buckets. */
	return ((unsigned int)key) % hashtable->size;
}

void
hash_destructor(hashkey_t key, hashvalue_t value)
{
	free(value);
}


void *mallocc(size_t size) {
	void *p = malloc(size);
	if(!p) {
		fprintf(stderr, "malloc(): out of memory\n");
		abort();
	}
}

struct hashtable
*hash_new(int size)
{
	int i;
	struct hashtable *hashtable;
	hashtable = mallocc(sizeof(struct hashtable));

	hashtable->size = size;
	hashtable->num_elements = 0;
	hashtable->table = mallocc(size * sizeof(struct hashentry));

	for(i = 0; i<size; i++)
		hashtable->table[i] = NULL;
	
	return hashtable;
}

void
hash_delete(struct hashtable *hashtable)
{
	int i;
	struct hashentry *pos, *oldpos;

	for(i = 0; i<hashtable->size; i++) {
		for(pos = hashtable->table[i]; pos; ) {
			hash_destructor(pos->key, pos->value);
			oldpos = pos;
			pos = pos->next;
			free(oldpos);
		}
	}

	free(hashtable);
}

int
hash_put(struct hashtable *hashtable, hashkey_t key, hashvalue_t value)
{
	int hash;
	struct hashentry *pos;
	
	hash = hash_hashkey(hashtable, key);
	pos = hashtable->table[hash];

	if(!pos) {
		/* New bucket */
		struct hashentry *newentry = mallocc(sizeof(struct hashentry));
		newentry->next = NULL;
		newentry->key = key;
		newentry->value = value;
		hashtable->table[hash] = newentry;
		hashtable->num_elements++;
		return 1;
	}

	for(;; pos = pos->next) {
		if(hash_keyequal(key, pos->key)) {
			hash_destructor(pos->key, pos->value);
			pos->key = key;
			pos->value = value; 
			return 2;
		}

		if(!pos->next) {
			/* The last element in the bucket */
			struct hashentry *newentry = mallocc(sizeof(struct hashentry));
			newentry->next = NULL;
			newentry->key = key;
			newentry->value = value;
			pos->next = newentry;
			hashtable->num_elements++;
			return 1;
		}
	}

	return 0;
}

int
hash_get(struct hashtable *hashtable, hashkey_t key, hashvalue_t *valueptr)
{
	int hash;
	struct hashentry *entry;
	hash = hash_hashkey(hashtable, key);
	
	for (entry = hashtable->table[hash]; entry; entry = entry->next)
	{
		if(hash_keyequal(key, entry->key)) {
			if(valueptr)
				*valueptr = entry->value;
			return 1;
		}
	}

	return 0;
}

int
hash_remove(struct hashtable *hashtable, hashkey_t key)
{
	int hash;
	struct hashentry *pos;
	
	hash = hash_hashkey(hashtable, key);
	pos = hashtable->table[hash];

	if(!pos)
		return 0;

	if(hash_keyequal(key, pos->key)) {
		hash_destructor(pos->key, pos->value);
		hashtable->table[hash] = pos->next;
		hashtable->num_elements--;
		free(pos);
		return 1;
	}
		
	for(; pos->next; pos = pos->next) {
		if(hash_keyequal(key, pos->next->key)) {
			hash_destructor(pos->next->key, pos->next->value);
			pos->next = pos->next->next;
			hashtable->num_elements--;
			free(pos->next);
			return 1;
		}
	}

	return 0;
}


