/* vim: sta et sw=4 
 */

/*
 * $Id: main.c,v 1.19 2003/06/01 20:33:03 stevecheng Exp $
 *
 * (C) 2001 Steve Cheng <stevecheng@users.sourceforge.net>
 *
 * This source compiles to db2x_xsltproc.
 *
 * See ../COPYING for the copyright status of this software.
 *
 * Derived from xsltproc.c from Daniel Veillard's libxslt.
 *
 */

#include "ext.h"

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <libxml/globals.h>
#include <libxml/parser.h>
#include <libxml/xmlIO.h>

#include <libxslt/xsltconfig.h>
#include <libxslt/xslt.h>
#include <libxslt/xsltInternals.h>
#include <libxslt/transform.h>
#include <libxslt/xsltutils.h>

#include <libexslt/exslt.h>

#ifdef LIBXML_DOCB_ENABLED
#include <libxml/DOCBparser.h>
#endif
#ifdef LIBXML_XINCLUDE_ENABLED
#include <libxml/xinclude.h>
#endif
#ifdef LIBXML_CATALOG_ENABLED
#include <libxml/catalog.h>
#endif


#define URN_PUBID_TEXI "urn:publicid:-:Steve+Cheng:DOCUMENT+DocBook+Texinfo+Stylesheet:EN"
#define URN_PUBID_MAN "urn:publicid:-:Steve+Cheng:DOCUMENT+DocBook+Man-page+Stylesheet:EN"

/* Prototypes */
void load_default_catalogs(void);
static void xslt_setup(void);
static void print_usage(void);
static void print_version(void);
static void parse_options(int argc, char *argv[]);

#ifdef HAVE_GETOPT_H
#include <getopt.h>

/* Long-option specification */
struct option long_options[] =
{
    { "version", 0, 0, 'v' },
    { "help", 0, 0, 'h' },
    
    { "output", 1, 0, 'o' },
    { "xinclude", 0, 0, 'I' },
    { "sgml-docbook", 0, 0, 'S' },
    { "network", 0, 0, 'N' },
    
    { "nesting-limit", 1, 0, 'n' },
    { "param", 1, 0, 'p' },
    { "stringparam", 1, 0, 'g' },
    { "string-param", 1, 0, 'g' },
    { "stylesheet", 1, 0, 's' },

    { "debug", 0, 0, 'd' },
    { "profile", 0, 0, 'P' },

    { 0, 0, 0, 0 }
};
#endif

/* Options */
const char *prog_name;
const char *input_file = NULL;
const char *output_file = NULL;
const char *more_catalogs = NULL;
#ifdef LIBXML_DOCB_ENABLED
int sgml_docbook = 0;
#endif
#ifdef LIBXML_XINCLUDE_ENABLED
int xinclude = 0;
#endif
int profile = 0;
const char *stylesheet_file = NULL;
const char **params;
int nbparams = 0;
int maxnbparams = 16;
int network = 0;

static
void xslt_setup(void)
{
    xmlLineNumbersDefault(1);
#ifdef LIBXML_XINCLUDE_ENABLED
    if(xinclude)
        xsltSetXIncludeDefault(1);
#endif

    /* Replace entities with their content. */
    xmlSubstituteEntitiesDefault(1);
    
    if(!network)
        xmlSetExternalEntityLoader(xmlNoNetExternalEntityLoader);

    /* For DocBook, generally, there will be ID references,
     * so we always load the external DTD. */
    xmlLoadExtDtdDefaultValue = XML_DETECT_IDS | XML_COMPLETE_ATTRS;

#ifdef LIBXML_CATALOG_ENABLED
    load_default_catalogs();

    /* Load additional catalogs specified from command line */
    if(more_catalogs)
        xmlLoadCatalogs(more_catalogs);

    /* The catalog for Texinfo and man page stylesheets
     * are hard-coded in the binary.  
     * Ideally it should be entered into the system catalog 
     * /etc/xml/catalog instead. */
    xmlLoadCatalogs(STYLESHEETS_CATALOG);

#endif
    
    /* Load docbook2X hash extensions */
    hashext_registermodule();

    /* Load the EXSLT and test modules; 
     * somebody might want to use them */
    exsltRegisterAll();
    xsltRegisterTestModule();
    
    /* Disable CDATA from being built in the document tree */
    xmlDefaultSAXHandlerInit();
    xmlDefaultSAXHandler.cdataBlock = NULL;
}

int
main(int argc, char **argv)
{
    xsltStylesheetPtr cur = NULL;
    xmlDocPtr doc = NULL;
    xmlDocPtr res = NULL;
    int err;

    prog_name = argv[0];
    
    LIBXML_TEST_VERSION

    parse_options(argc, argv);

    xslt_setup();

    /* Parse stylesheet */
    if(stylesheet_file) {
        cur = xsltParseStylesheetFile((const xmlChar *)stylesheet_file);
        if(!cur) {
            fprintf(stderr, "%s: cannot parse stylesheet\n", prog_name);
            return 2;
        }
    }

    /* Parse input document */
#ifdef LIBXML_DOCB_ENABLED
    if(sgml_docbook)
        doc = docbParseFile(input_file, NULL);
    else 
#endif
    {
        doc = xmlParseFile(input_file);
#ifdef LIBXML_XINCLUDE_ENABLED
        if(xinclude)
            xmlXIncludeProcess(doc);
#endif
    }
    
    if(!doc) {
        fprintf(stderr, "%s: cannot parse input document\n", prog_name);
        if(cur) xsltFreeStylesheet(cur);
        return 3;
    }

    /* User didn't specify stylesheet on command-line, so may be inside document */
    if(!cur) {
        cur = xsltLoadStylesheetPI(doc);
        if(!cur) {
            fprintf(stderr, "%s: no stylesheet\n", prog_name);
            xmlFreeDoc(doc);
            return 1;
        }
    }

    /* Apply stylesheet */
    if(profile)
        res = xsltProfileStylesheet(cur, doc, params, stderr);
    else 
        res = xsltApplyStylesheet(cur, doc, params);

    if(!res) {
        fprintf(stderr, "%s: applying stylesheet failed\n", prog_name);
        xsltFreeStylesheet(cur);
        xmlFreeDoc(doc);
        return 4;
    }

    /* Output result */
    if(output_file)
        err = xsltSaveResultToFilename(output_file, res, cur, 0);
    else
        err = xsltSaveResultToFile(stdout, res, cur);

    if(err <= -1) {
        fprintf(stderr, "%s: failed to output result\n", prog_name);
        xsltFreeStylesheet(cur);
        xmlFreeDoc(res);
        xmlFreeDoc(doc);
        return 5;
    }
    
    xsltFreeStylesheet(cur);
    xmlFreeDoc(res);
    xmlFreeDoc(doc);

    return 0;
}

static void
print_version(void)
{
    puts("db2x_xsltproc (part of docbook2X " 
#ifdef HAVE_CONFIG_H
    VERSION
#endif
        ")");

    puts("$Revision: 1.19 $ $Date: 2003/06/01 20:33:03 $");
    printf("Using libxml %s, libxslt %s\n",
        xmlParserVersion, xsltEngineVersion);
    puts("<URL:http://docbook2x.sourceforge.net/>\n");
    
    puts("Copyright (C) 2000-2001 Steve Cheng\n"
         "This is free software; see the source for copying conditions.\n"
         "There is NO warranty; not even for MERCHANTABILITY or FITNESS FOR\n"
         "A PARTICULAR PURPOSE.");
}

static void
print_usage()
{
    printf("Usage: %s [options] xml-document\n", prog_name);
    puts("Convert DocBook to Texi-XML (for Texinfo)\n");
    
    puts("  -v, --version           display version information and exit\n"
         "  -h, --help              display this usage information\n"
         "  -C, --catalogs FILES    use additional catalogs\n"
         "  -o, --output FILE       send output to file instead of stdout\n"
         "  -I, --xinclude          do XInclude processing\n"
         "  -S, --sgml-docbook      input document is SGML DocBook\n"
         "  -N, --network           allow fetching resources over network\n"
         "  -n, --nesting-limit     change maximum nesting depth of templates\n"
         "  -p, --param NAME=VALUE  add or modify a parameter to stylesheet\n"
         "                          VALUE is an XPath expression\n"
         "  -g, --string-param NAME=VALUE\n"
         "                          same as -p, but VALUE is treated as a string\n"
         "  -s, --stylesheet FILE   specify different stylesheet to use\n"
         "  -d, --debug             display log of transformation\n"
         "  -P, --profile           display profiling information\n");
}
    
static void
parse_options(int argc, char *argv[])
{
    int optc;

    params = malloc((maxnbparams+1)*sizeof(char*));
    if(!params) abort();
    
#ifdef HAVE_GETOPT_H
    while((optc = getopt_long(argc, argv, "vhC:o:ISNn:p:g:s:dP", 
                long_options, NULL)) != -1)
#else
    while((optc = getopt(argc, argv, "vhC:o:ISNn:p:g:s:dP")) != -1)
#endif
    {
        switch(optc) {

        /* --version */
        case 'v':
            print_version();
            exit(0);
            
        /* --help */
        case 'h':
            print_usage();
            exit(0);
                
#ifdef LIBXML_CATALOG_ENABLED
        /* --catalogs {file} */
        case 'C':
            more_catalogs = optarg;
            break;
#else
        case 'C':
            fprintf(stderr, "%s: No catalog support in this build\n", prog_name);
            exit(1);
#endif
            
        /* --output {file} */
        case 'o':
            output_file = optarg;
            break;

#ifdef LIBXML_XINCLUDE_ENABLED
        /* --xinclude */
        case 'I':
            xinclude = 1;
            break;
#else
        case 'I':
            fprintf(stderr, "%s: XInclude not supported in this build\n", 
                    prog_name);
            exit(1);
#endif
        
#ifdef LIBXML_DOCB_ENABLED
        /* --sgml-docbook */
        case 'S':
            sgml_docbook = 1;
            break;
#else
        case 'S':
            fprintf(stderr, "%s: SGML DocBook not supported in this build\n", 
                    prog_name);
            exit(1);
#endif

        /* --network */
        case 'N':
            network = 1;
            break;
        
        /* --nesting-limit {n} */
        case 'n': {
            int value = 0;
            if(sscanf(optarg, "%d", &value) == 1 && value>0)
                xsltMaxDepth = value;
            else {
                fprintf(stderr, "%s: invalid nesting-limit\n", prog_name);
                exit(1);
            }
            break;
        }

        /* --param {name=value} */
        case 'p': {
            char *name;
            char *value;
            int i;
            
            for(i = 0; optarg[i]; i++) 
                if(optarg[i] == '=')
                    break;

            if(!optarg[i]) {
                fprintf(stderr, "%s: no value for parameter\n", prog_name);
                exit(1);
            }
            
            /* I'm not sure if you can modify the argv
             * string; just to be safe */
            name = malloc(i+1);
            strncpy(name, optarg, i);
            name[i] = '\0';

            value = malloc(strlen(optarg+i));
            strcpy(value, optarg+i+1);

            if(nbparams+2 > maxnbparams) {
                maxnbparams += 16;
                params = realloc(params, (maxnbparams+1)*sizeof(const char*));
                if(!params) abort();
            }

            params[nbparams++] = name;
            params[nbparams++] = value;
            break;
        }

        /* --string-param {name=value} */
        case 'g': {
            char *name;
            char *string;
            xmlChar *value;
            int i;
            int len;
            
            for(i = 0; optarg[i]; i++) 
                if(optarg[i] == '=')
                    break;

            if(!optarg[i]) {
                fprintf(stderr, "%s: no value for parameter\n", prog_name);
                exit(1);
            }
            
            /* I'm not sure if you can modify the argv
             * string; just to be safe */
            name = malloc(i+1);
            strncpy(name, optarg, i);
            name[i] = '\0';

            string = optarg+i+1;

            len = xmlStrlen(string);
            if (xmlStrchr(string, '"')) {
                if (xmlStrchr(string, '\'')) {
/* Stupid, but even xsltproc has to do this too */
                    fprintf(stderr,
                    "Sorry, string-param cannot contain both quote and double-quotes.\n");
                    exit(8);
                    }

                value = xmlStrdup((const xmlChar *)"'");
                value = xmlStrcat(value, string);
                value = xmlStrcat(value, (const xmlChar *)"'");
            } else {
                value = xmlStrdup((const xmlChar *)"\"");
                value = xmlStrcat(value, string);
                value = xmlStrcat(value, (const xmlChar *)"\"");
            }

            if(nbparams+2 > maxnbparams) {
                maxnbparams += 16;
                params = realloc(params, (maxnbparams+1)*sizeof(const char*));
                if(!params) abort();
            }

            params[nbparams++] = name;
            params[nbparams++] = value;
            break;
        }

        /* --stylesheet {file} */    
        case 's':
            if(strcmp(optarg, "texi")==0)
                stylesheet_file = URN_PUBID_TEXI;
            else if(strcmp(optarg, "man")==0)
                stylesheet_file = URN_PUBID_MAN;
            else
                stylesheet_file = optarg;
            break;

        /* --debug */
        case 'd':
            xsltSetGenericDebugFunc(stderr, NULL);
            break;
        
        /* --profile */
        case 'P':
            profile = 1;
            break;

        case '?':
        default:
            exit(1);
        }
    }

    if(optind != argc-1) {
        fprintf(stderr, "%s: must specify one input document\n", prog_name);
        exit(1);
    }

    input_file = argv[optind];

    params[nbparams] = NULL;
}
