/* dirmngr.c - LDAP access
 *	Copyright (C) 2002 Klarälvdalens Datakonsult AB
 *      Copyright (C) 2003, 2004, 2006 g10 Code GmbH
 *
 * This file is part of DirMngr.
 *
 * DirMngr is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * DirMngr is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include <config.h>

#include <stdio.h>
#include <stdlib.h>
#include <stddef.h>
#include <stdarg.h>
#include <string.h>
#include <errno.h>
#include <assert.h>
#include <time.h>
#include <fcntl.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <sys/un.h>
#include <unistd.h>
#include <signal.h>
#include <pth.h>

#include <gcrypt.h>
#include <ksba.h>
#include <assuan.h> /* Needed for the malloc hooks */

#define JNLIB_NEED_LOG_LOGV
#include "dirmngr.h"
#include "certcache.h"
#include "crlcache.h"
#include "crlfetch.h"
#include "misc.h"

enum cmd_and_opt_values 
{ aNull = 0,
  oCsh		  = 'c',
  oQuiet	  = 'q',
  oSh		  = 's',
  oVerbose	  = 'v',
  oNoVerbose = 500,
  
  aServer,
  aDaemon,
  aListCRLs,
  aLoadCRL,
  aFetchCRL,
  aShutdown,
  aFlush,
  aGPGConfList,

  oOptions,
  oDebug,
  oDebugAll,
  oDebugWait,
  oDebugLevel,
  oNoGreeting,
  oNoOptions,
  oHomedir,
  oNoDetach,
  oLogFile,
  oBatch,
  oDisableHTTP,
  oDisableLDAP,
  oIgnoreLDAPDP,
  oIgnoreHTTPDP,
  oIgnoreOCSPSvcUrl,
  oHonorHTTPProxy,
  oHTTPProxy,
  oLDAPProxy,
  oOnlyLDAPProxy,
  oLDAPFile,
  oLDAPTimeout,
  oLDAPAddServers,
  oOCSPResponder,
  oOCSPSigner,
  oOCSPMaxClockSkew,
  oOCSPCurrentPeriod,
  oMaxReplies,
  oFakedSystemTime,
  oForce,
  oAllowOCSP,
  oSocketName,
  oLDAPWrapperProgram,
  oHTTPWrapperProgram,
aTest };



static ARGPARSE_OPTS opts[] = {
  
  { 300, NULL, 0, N_("@Commands:\n ") },

  { aServer,   "server",    256, N_("run in server mode (foreground)") },
  { aDaemon,   "daemon",    256, N_("run in daemon mode (background)") },
  { aListCRLs, "list-crls", 256, N_("list the contents of the CRL cache")},
  { aLoadCRL,  "load-crl",  256, N_("|FILE|load CRL from FILE into cache")},
  { aFetchCRL, "fetch-crl", 256, N_("|URL|fetch a CRL from URL")},
  { aShutdown, "shutdown",  256, N_("shutdown the dirmngr")},
  { aFlush,    "flush",     256, N_("flush the cache")},
  { aGPGConfList, "gpgconf-list", 256, "@" },

  { 301, NULL, 0, N_("@\nOptions:\n ") },

  { oVerbose,  "verbose",   0, N_("verbose") },
  { oQuiet,    "quiet",     0, N_("be somewhat more quiet") },
  { oSh,       "sh",        0, N_("sh-style command output") },
  { oCsh,      "csh",       0, N_("csh-style command output") },
  { oOptions,  "options"  , 2, N_("|FILE|read options from FILE")},
  { oDebugLevel, "debug-level",2,
                               N_("|LEVEL|set the debugging level to LEVEL")},
  { oNoDetach, "no-detach" ,0, N_("do not detach from the console")},
  { oLogFile,  "log-file"  ,2, N_("|FILE|write server mode logs to FILE")},
  { oBatch   , "batch"     ,0, N_("run without asking a user")},
  { oForce,    "force"     ,0, N_("force loading of outdated CRLs")},
  { oAllowOCSP, "allow-ocsp",0,N_("allow sending OCSP requests")},
  { oDisableHTTP, "disable-http", 0, N_("inhibit the use of HTTP")},
  { oDisableLDAP, "disable-ldap", 0, N_("inhibit the use of LDAP")},
  { oIgnoreHTTPDP,"ignore-http-dp", 0,
    N_("ignore HTTP CRL distribution points")},
  { oIgnoreLDAPDP,"ignore-ldap-dp", 0,
    N_("ignore LDAP CRL distribution points")},
  { oIgnoreOCSPSvcUrl, "ignore-ocsp-service-url", 0,
    N_("ignore certificate contained OCSP service URLs")},
      /* Note: The next one is to fix a typo in gpgconf - should be
         removed eventually. */
  { oIgnoreOCSPSvcUrl, "ignore-ocsp-servic-url", 0, "@"},

  { oHTTPProxy,  "http-proxy", 2,
    N_("|URL|redirect all HTTP requests to URL")},
  { oLDAPProxy,  "ldap-proxy", 2,
    N_("|HOST|use HOST for LDAP queries")},
  { oOnlyLDAPProxy, "only-ldap-proxy", 0, 
    N_("do not use fallback hosts with --ldap-proxy")},

  { oLDAPFile, "ldapserverlist-file", 2,
    N_("|FILE|read LDAP server list from FILE")},
  { oLDAPAddServers, "add-servers", 0,
    N_("add new servers discovered in CRL distribution points to serverlist")},
  { oLDAPTimeout, "ldaptimeout", 1,
    N_("|N|set LDAP timeout to N seconds")},

  { oOCSPResponder, "ocsp-responder", 2, N_("|URL|use OCSP responder at URL")},
  { oOCSPSigner, "ocsp-signer", 2, N_("|FPR|OCSP response signed by FPR")}, 
  { oOCSPMaxClockSkew, "ocsp-max-clock-skew", 1, "@" },
  { oOCSPCurrentPeriod, "ocsp-current-period", 1, "@" },

  { oMaxReplies, "max-replies", 1,
    N_("|N|do not return more than N items in one query")},

  { oSocketName, "socket-name", 2, N_("|FILE|listen on socket FILE") },

  { oFakedSystemTime, "faked-system-time", 4, "@" }, /* (epoch time) */
  { oDebug,    "debug"     ,4|16, "@"},
  { oDebugAll, "debug-all" ,0,    "@"},
  { oDebugWait, "debug-wait", 1, "@"},
  { oNoGreeting, "no-greeting", 0, "@"},
  { oHomedir, "homedir", 2, "@" },  
  { oLDAPWrapperProgram, "ldap-wrapper-program", 2, "@"},
  { oHTTPWrapperProgram, "http-wrapper-program", 2, "@"},
  { oHonorHTTPProxy,     "honor-http-proxy", 0, "@" },

  { 302, NULL, 0, N_(
  "@\n(See the \"info\" manual for a complete listing of all commands and options)\n"
                    )},

  {0}
};

#define DEFAULT_MAX_REPLIES 10
#define DEFAULT_LDAP_TIMEOUT 100 /* arbitrary large timeout */
#define DEFAULT_SOCKET_NAME "/var/run/dirmngr/socket"

/* For the cleanup handler we need to keep track of the socket's name. */
static const char *socket_name;
/* Only if this flag has been set we will remove the socket file.  */
static int cleanup_socket;
/* Keep track of the current log file so that we can avoid updating
   the log file after a SIGHUP if it didn't changed. Malloced. */
static char *current_logfile;
/* Name of a config file, which will be reread on a HUP if it is not NULL. */
static char *config_filename;
/* Helper to implement --debug-level. */
static const char *debug_level;
/* Flag indicating that a shutdown has been requested.  */
static int shutdown_pending;
/* Counter for the active connections.  */
static int active_connections;

/* Prototypes. */
static void cleanup (void);
static ldap_server_t parse_ldapserver_file (const char* filename);
static void free_ldapservers_list (ldap_server_t servers);
static void handle_connections (int listen_fd);

/* Pth wrapper function definitions. */
GCRY_THREAD_OPTION_PTH_IMPL;


static const char *
my_strusage( int level )
{
    const char *p;
    switch( level ) {
      case 11: p = "dirmngr";
	break;
      case 13: p = VERSION; break;
      case 14: p = "Copyright (C) 2006 g10 Code GmbH"; break;
      case 17: p = PRINTABLE_OS_NAME; break;
      case 19: p =
	    _("Please report bugs to <gpa-dev@gnupg.org>.\n");
	break;
      case 1:
      case 40:	p =
	    _("Usage: dirmngr [options] (-h for help)");
	break;
      case 41:	p =
	    _("Syntax: dirmngr [options] [command [args]]\n"
	      "LDAP and OCSP access for GnuPG\n");
	break;

      default:	p = NULL;
    }
    return p;
}


/* Used by gcry for logging. */
static void
my_gcry_logger (void *dummy, int level, const char *fmt, va_list arg_ptr)
{
  /* translate the log levels */
  switch (level)
    {
    case GCRY_LOG_CONT: level = JNLIB_LOG_CONT; break;
    case GCRY_LOG_INFO: level = JNLIB_LOG_INFO; break;
    case GCRY_LOG_WARN: level = JNLIB_LOG_WARN; break;
    case GCRY_LOG_ERROR:level = JNLIB_LOG_ERROR; break;
    case GCRY_LOG_FATAL:level = JNLIB_LOG_FATAL; break;
    case GCRY_LOG_BUG:  level = JNLIB_LOG_BUG; break;
    case GCRY_LOG_DEBUG:level = JNLIB_LOG_DEBUG; break;
    default:            level = JNLIB_LOG_ERROR; break;  
    }
  log_logv (level, fmt, arg_ptr);
}


/* Callback from libksba to hash a provided buffer.  Our current
   implementation does only allow SHA-1 for hashing. This may be
   extended by mapping the name, testing for algorithm availibility
   and adjust the length checks accordingly. */
static gpg_error_t 
my_ksba_hash_buffer (void *arg, const char *oid,
                     const void *buffer, size_t length, size_t resultsize,
                     unsigned char *result, size_t *resultlen)
{
  if (oid && strcmp (oid, "1.3.14.3.2.26")) 
    return gpg_error (GPG_ERR_NOT_SUPPORTED); 
  if (resultsize < 20)
    return gpg_error (GPG_ERR_BUFFER_TOO_SHORT);
  gcry_md_hash_buffer (2, result, buffer, length); 
  *resultlen = 20;
  return 0;
}


/* Setup the debugging.  With a LEVEL of NULL only the active debug
   flags are propagated to the subsystems.  With LEVEL set, a specific
   set of debug flags is set; thus overriding all flags already
   set. */
static void
set_debug (void)
{
  if (!debug_level)
    ;
  else if (!strcmp (debug_level, "none"))
    opt.debug = 0;
  else if (!strcmp (debug_level, "basic"))
    opt.debug = DBG_ASSUAN_VALUE;
  else if (!strcmp (debug_level, "advanced"))
    opt.debug = (DBG_ASSUAN_VALUE|DBG_X509_VALUE|DBG_LOOKUP_VALUE);
  else if (!strcmp (debug_level, "expert"))
    opt.debug = (DBG_ASSUAN_VALUE|DBG_X509_VALUE|DBG_LOOKUP_VALUE
                 |DBG_CACHE_VALUE|DBG_CRYPTO_VALUE);
  else if (!strcmp (debug_level, "guru"))
    opt.debug = ~0;
  else
    {
      log_error (_("invalid debug-level `%s' given\n"), debug_level);
      log_info (_("valid debug levels are: %s\n"),
                "none, basic, advanced, expert, guru");
      opt.debug = 0; /* Reset debugging, so that prior debug
                        statements won't have an undesired effect. */
    }


  if (opt.debug && !opt.verbose)
    {
      opt.verbose = 1;
      gcry_control (GCRYCTL_SET_VERBOSITY, (int)opt.verbose);
    }
  if (opt.debug && opt.quiet)
    opt.quiet = 0;

  if (opt.debug & DBG_CRYPTO_VALUE )
    gcry_control (GCRYCTL_SET_DEBUG_FLAGS, 1);
}
 

static void
i18n_init(void)
{
#ifdef USE_SIMPLE_GETTEXT
  set_gettext_file (PACKAGE);
#else
# ifdef ENABLE_NLS
  setlocale (LC_ALL, "" );
  bindtextdomain (PACKAGE, LOCALEDIR);
  textdomain (PACKAGE);
# endif
#endif
}


static void
wrong_args (const char *text)
{
  fputs (_("usage: dirmngr [options] "), stderr);
  fputs (text, stderr);
  putc ('\n', stderr);
  dirmngr_exit (2);
}


/* Helper to start the reaper thread for the ldap wrapper.  */
static void
launch_reaper_thread (void)
{
  static int done;
  pth_attr_t tattr;

  if (done)
    return;
  done = 1;

  tattr = pth_attr_new();
  pth_attr_set (tattr, PTH_ATTR_JOINABLE, 0);
  pth_attr_set (tattr, PTH_ATTR_STACK_SIZE, 256*1024);
  pth_attr_set (tattr, PTH_ATTR_NAME, "ldap-reaper");

  if (!pth_spawn (tattr, ldap_wrapper_thread, NULL))
    {
      log_error (_("error spawning ldap wrapper reaper thread: %s\n"),
                 strerror (errno) );
      dirmngr_exit (1);
    }
}

/* Helper to stop the reaper thread for the ldap wrapper.  */
static void
shutdown_reaper (void)
{
  ldap_wrapper_wait_connections ();
}

/* Handle options which are allowed to be reset after program start.
   Return true if the current option in PARGS could be handled and
   false if not.  As a special feature, passing a value of NULL for
   PARGS, resets the options to the default.  REREAD should be set
   true if it is not the initial option parsing. */
static int
parse_rereadable_options (ARGPARSE_ARGS *pargs, int reread)
{
  if (!pargs)
    { /* Reset mode. */
      opt.quiet = 0;
      opt.verbose = 0;
      opt.debug = 0;
      opt.ldap_wrapper_program = NULL;
      opt.disable_http = 0;
      opt.disable_ldap = 0; 
      opt.honor_http_proxy = 0; 
      opt.http_proxy = NULL; 
      opt.ldap_proxy = NULL; 
      opt.only_ldap_proxy = 0;
      opt.ignore_http_dp = 0;
      opt.ignore_ldap_dp = 0;
      opt.ignore_ocsp_service_url = 0;
      opt.allow_ocsp = 0;
      opt.ocsp_responder = NULL;
      opt.ocsp_signer = NULL; 
      opt.ocsp_max_clock_skew = 10 * 60;      /* 10 minutes.  */
      opt.ocsp_current_period = 3 * 60 * 60;  /* 3 hours. */
      opt.max_replies = DEFAULT_MAX_REPLIES;
      return 1;
    }

  switch (pargs->r_opt)
    {
    case oQuiet:   opt.quiet = 1; break;
    case oVerbose: opt.verbose++; break;
    case oDebug:   opt.debug |= pargs->r.ret_ulong; break;
    case oDebugAll: opt.debug = ~0; break;
    case oDebugLevel: debug_level = pargs->r.ret_str; break;

    case oLogFile:
      if (!reread)
        return 0; /* Not handled. */
      if (!current_logfile || !pargs->r.ret_str
          || strcmp (current_logfile, pargs->r.ret_str))
        {
          log_set_file (pargs->r.ret_str);
          xfree (current_logfile);
          current_logfile = xtrystrdup (pargs->r.ret_str);
        }
      break;

    case oLDAPWrapperProgram:
      opt.ldap_wrapper_program = pargs->r.ret_str;
      break;
    case oHTTPWrapperProgram:
      opt.http_wrapper_program = pargs->r.ret_str;
      break;

    case oDisableHTTP: opt.disable_http = 1; break;
    case oDisableLDAP: opt.disable_ldap = 1; break;
    case oHonorHTTPProxy: opt.honor_http_proxy = 1; break;
    case oHTTPProxy: opt.http_proxy = pargs->r.ret_str; break;
    case oLDAPProxy: opt.ldap_proxy = pargs->r.ret_str; break;
    case oOnlyLDAPProxy: opt.only_ldap_proxy = 1; break;
    case oIgnoreHTTPDP: opt.ignore_http_dp = 1; break;
    case oIgnoreLDAPDP: opt.ignore_ldap_dp = 1; break;
    case oIgnoreOCSPSvcUrl: opt.ignore_ocsp_service_url = 1; break;

    case oAllowOCSP: opt.allow_ocsp = 1; break;
    case oOCSPResponder: opt.ocsp_responder = pargs->r.ret_str; break;
    case oOCSPSigner:    opt.ocsp_signer = pargs->r.ret_str; break;
    case oOCSPMaxClockSkew: opt.ocsp_max_clock_skew = pargs->r.ret_int; break;
    case oOCSPCurrentPeriod: opt.ocsp_current_period = pargs->r.ret_int; break;

    case oMaxReplies: opt.max_replies = pargs->r.ret_int; break;

    default:
      return 0; /* Not handled. */
    }

  return 1; /* Handled. */
}

int
main (int argc, char **argv )
{
  enum cmd_and_opt_values cmd = 0;
  ARGPARSE_ARGS pargs;
  int orig_argc;
  char **orig_argv;
  FILE *configfp = NULL;
  char *configname = NULL;
  const char *shell;
  unsigned configlineno;
  int parse_debug = 0;
  int default_config =1;
  int greeting = 0;
  int nogreeting = 0;
  int nodetach = 0;
  int csh_style = 0;
  char *logfile = NULL;
  char *ldapfile = NULL;
  int debug_wait = 0;
  int rc;
  int homedir_seen = 0;

  set_strusage (my_strusage);
  log_set_prefix ("dirmngr", 1|4); 
  
  /* Check that the libraries are suitable.  Do it here because
     the option parsing may need services of the libraries. */

  /* Libgcrypt requires us to register the threading model first.
     Note that this will also do the pth_init. */

  /* Init Libgcrypt. */
  rc = gcry_control (GCRYCTL_SET_THREAD_CBS, &gcry_threads_pth);
  if (rc)
    {
      log_fatal ("can't register GNU Pth with Libgcrypt: %s\n",
                 gpg_strerror (rc));
    }
  gcry_control (GCRYCTL_DISABLE_SECMEM, 0);
  if (!gcry_check_version (NEED_LIBGCRYPT_VERSION) )
    {
      log_fatal (_("%s is too old (need %s, have %s)\n"),
                 "libgcrypt",
                 NEED_LIBGCRYPT_VERSION, gcry_check_version (NULL) );
    }
  gcry_set_log_handler (my_gcry_logger, NULL);
  gcry_control (GCRYCTL_INITIALIZATION_FINISHED);

  /* Init KSBA.  */
  if (!ksba_check_version (NEED_KSBA_VERSION) )
    {
      log_fatal( _("%s is too old (need %s, have %s)\n"),
                 "libksba",
                 NEED_KSBA_VERSION, ksba_check_version (NULL) );
    }
  ksba_set_malloc_hooks (gcry_malloc, gcry_realloc, gcry_free );
  ksba_set_hash_buffer_function (my_ksba_hash_buffer, NULL);


  /* Init Assuan. */
  assuan_set_malloc_hooks (gcry_malloc, gcry_realloc, gcry_free);
  assuan_set_assuan_log_stream (log_get_stream ());
  assuan_set_assuan_log_prefix (log_get_prefix (NULL));

  /* Setup I18N. */
  i18n_init();

  /* Setup defaults. */
  shell = getenv ("SHELL");
  if (shell && strlen (shell) >= 3 && !strcmp (shell+strlen (shell)-3, "csh") )
    csh_style = 1;
  
  opt.homedir = getenv("GNUPGHOME");
  if (!opt.homedir || !*opt.homedir)
    {
#ifdef HAVE_DRIVE_LETTERS
      opt.homedir = "c:/gnupg";
#else
      opt.homedir = "~/.gnupg";
#endif
    }

  /* Reset rereadable options to default values. */
  parse_rereadable_options (NULL, 0); 

  /* LDAP defaults.  */
  opt.add_new_ldapservers = 0;
  opt.ldaptimeout = DEFAULT_LDAP_TIMEOUT;

  /* Other defaults.  */
  socket_name = DEFAULT_SOCKET_NAME;


  /* Check whether we have a config file given on the commandline */
  orig_argc = argc;
  orig_argv = argv;
  pargs.argc = &argc;
  pargs.argv = &argv;
  pargs.flags= 1|(1<<6);  /* do not remove the args, ignore version */
  while (arg_parse( &pargs, opts))
    {
      if (pargs.r_opt == oDebug || pargs.r_opt == oDebugAll)
        parse_debug++;
      else if (pargs.r_opt == oOptions)
        { /* Yes there is one, so we do not try the default one, but
	     read the option file when it is encountered at the
	     commandline */
          default_config = 0;
	}
      else if (pargs.r_opt == oNoOptions)
        default_config = 0; /* --no-options */
      else if (pargs.r_opt == oHomedir)
        {
          opt.homedir = pargs.r.ret_str;
          homedir_seen = 1;
        }
      else if (pargs.r_opt == aDaemon)
        opt.system_daemon = 1;
    }

  /* If --daemon has been given on the command line but not --homedir,
     we switch to /etc/dirmngr as default home directory.  Note, that
     this also overrides the GNUPGHOME environment variable.  */
  if (opt.system_daemon && !homedir_seen)
    {
      opt.homedir = DIRMNGR_SYSCONFDIR;
      opt.homedir_data = DIRMNGR_DATADIR;
      opt.homedir_cache = DIRMNGR_CACHEDIR;
    }

  if (default_config)
    configname = make_filename (opt.homedir, "dirmngr.conf", NULL );
  
  argc = orig_argc;
  argv = orig_argv;
  pargs.argc = &argc;
  pargs.argv = &argv;
  pargs.flags= 1;  /* do not remove the args */
 next_pass:
  if (configname)
    {
      configlineno = 0;
      configfp = fopen (configname, "r");
      if (!configfp)
        {
          if (default_config)
            {
              if( parse_debug )
                log_info (_("NOTE: no default option file `%s'\n"),
                          configname );
	    }
          else
            {
              log_error (_("option file `%s': %s\n"),
                         configname, strerror(errno) );
              exit(2);
	    }
          xfree (configname); 
          configname = NULL;
	}
      if (parse_debug && configname )
        log_info (_("reading options from `%s'\n"), configname );
      default_config = 0;
    }

  while (optfile_parse( configfp, configname, &configlineno, &pargs, opts) )
    {
      if (parse_rereadable_options (&pargs, 0))
        continue; /* Already handled */
      switch (pargs.r_opt)
        {
        case aServer: 
        case aDaemon:
        case aShutdown: 
        case aFlush: 
	case aListCRLs: 
	case aLoadCRL: 
        case aFetchCRL:
	case aGPGConfList:
          cmd = pargs.r_opt;
          break;

        case oQuiet: opt.quiet = 1; break;
        case oVerbose: opt.verbose++; break;
        case oBatch: opt.batch=1; break;

        case oDebug: opt.debug |= pargs.r.ret_ulong; break;
        case oDebugAll: opt.debug = ~0; break;
        case oDebugLevel: debug_level = pargs.r.ret_str; break;
        case oDebugWait: debug_wait = pargs.r.ret_int; break;

        case oOptions:
          /* Config files may not be nested (silently ignore them) */
          if (!configfp)
            {
		xfree(configname);
		configname = xstrdup(pargs.r.ret_str);
		goto next_pass;
	    }
          break;
        case oNoGreeting: nogreeting = 1; break;
        case oNoVerbose: opt.verbose = 0; break;
        case oNoOptions: break; /* no-options */
        case oHomedir: /* Ignore this option here. */; break;
        case oNoDetach: nodetach = 1; break;
        case oLogFile: logfile = pargs.r.ret_str; break;
        case oCsh: csh_style = 1; break;
        case oSh: csh_style = 0; break;
	case oLDAPFile: ldapfile = pargs.r.ret_str; break;
	case oLDAPAddServers: opt.add_new_ldapservers = 1; break;
	case oLDAPTimeout: 
	  opt.ldaptimeout = pargs.r.ret_int; 
	  break;

        case oFakedSystemTime:
          set_time ( (time_t)pargs.r.ret_ulong, 0);
          break;

        case oForce: opt.force = 1; break;

        case oSocketName: socket_name = pargs.r.ret_str; break;

        default : pargs.err = configfp? 1:2; break;
	}
    }
  if (configfp)
    {
      fclose( configfp );
      configfp = NULL;
      /* Keep a copy of the name so that it can be read on SIGHUP. */
      config_filename = configname;
      configname = NULL;
      goto next_pass;
    }
  xfree (configname);
  configname = NULL;
  if (log_get_errorcount(0))
    exit(2);
  if (nogreeting )
    greeting = 0;

  if (!opt.homedir_data)
    opt.homedir_data = opt.homedir;
  if (!opt.homedir_cache)
    opt.homedir_cache = opt.homedir;

  if (greeting)
    {
      fprintf (stderr, "%s %s; %s\n",
               strusage(11), strusage(13), strusage(14) );
      fprintf (stderr, "%s\n", strusage(15) );
    }

#ifdef IS_DEVELOPMENT_VERSION
  log_info ("NOTE: this is a development version!\n");
#endif

  if (faked_time_p ())
    {
      dirmngr_isotime_t tbuf;
      get_isotime (tbuf);
      log_info (_("WARNING: running with faked system time %s\n"), tbuf);
    }

  set_debug ();

  /* Get LDAP server list from file. */
  if (!ldapfile) 
    {
      ldapfile = make_filename (opt.homedir,
                                opt.system_daemon?
                                "ldapservers.conf":"dirmngr_ldapservers.conf",
                                NULL);
      opt.ldapservers = parse_ldapserver_file (ldapfile);
      xfree (ldapfile);
    }
  else
      opt.ldapservers = parse_ldapserver_file (ldapfile);

  /* We need to ignore the PIPE signal because the we might log to a
     socket and that code handles EPIPE properly.  The ldap wrapper
     also requires us to ignore this silly signal. Assuan would set
     this signal to ignore anyway.*/
  signal (SIGPIPE, SIG_IGN);


  /* Ready.  Now to our duties. */
  if (!cmd)
    cmd = aServer;
  rc = 0;

  if (cmd == aServer)
    {
      if (argc)
        wrong_args ("--server");

      if (logfile)
        {
          log_set_file (logfile);
          log_set_prefix (NULL, 2|4);
        }
       
      if (debug_wait)
        {
          log_debug ("waiting for debugger - my pid is %u .....\n",
                     (unsigned int)getpid());
          sleep (debug_wait);
          log_debug ("... okay\n");
        }

      launch_reaper_thread ();
      cert_cache_init ();
      crl_cache_init ();
      start_command_handler (-1);
      shutdown_reaper ();
    }
  else if (cmd == aDaemon)
    {
      int fd;
      pid_t pid;
      int len;
      struct sockaddr_un serv_addr;

      if (argc)
        wrong_args ("--daemon");
      
      /* Now start with logging to a file if this is desired. */
      if (logfile)
        {
          log_set_file (logfile);
          log_set_prefix (NULL, (JNLIB_LOG_WITH_PREFIX
                                 |JNLIB_LOG_WITH_TIME
                                 |JNLIB_LOG_WITH_PID));
          current_logfile = xstrdup (logfile);
        }

      if (strchr (socket_name, ':') )
        {
          log_error (_("colons are not allowed in the socket name\n"));
          dirmngr_exit (1);
        }
      if (strlen (socket_name)+1 >= sizeof serv_addr.sun_path ) 
        {
          log_error (_("name of socket too long\n"));
          dirmngr_exit (1);
        }
   
      fd = socket (AF_UNIX, SOCK_STREAM, 0);
      if (fd == -1)
        {
          log_error (_("can't create socket: %s\n"), strerror(errno) );
          cleanup ();
          dirmngr_exit (1);
        }

      memset (&serv_addr, 0, sizeof serv_addr);
      serv_addr.sun_family = AF_UNIX;
      strcpy (serv_addr.sun_path, socket_name);
      len = (offsetof (struct sockaddr_un, sun_path)
             + strlen(serv_addr.sun_path) + 1);

      rc = bind (fd, (struct sockaddr*)&serv_addr, len);
      if (rc == -1 && errno == EADDRINUSE)
        {
          remove (socket_name);
          rc = bind (fd, (struct sockaddr*)&serv_addr, len);
        }
      if (rc == -1)
        {
          log_error (_("error binding socket to `%s': %s\n"),
                     serv_addr.sun_path, strerror (errno) );
          close (fd);
          dirmngr_exit (1);
        }
      cleanup_socket = 1;
  
      if (listen (fd, 5 ) == -1)
        {
          log_error (_("listen() failed: %s\n"), strerror (errno));
          close (fd);
          dirmngr_exit (1);
        }

      if (opt.verbose)
        log_info (_("listening on socket `%s'\n"), socket_name );

      fflush (NULL);
      pid = pth_fork ();
      if (pid == (pid_t)-1) 
        {
          log_fatal (_("fork failed: %s\n"), strerror (errno) );
          dirmngr_exit (1);
        }

      if (pid) 
        { /* We are the parent */
          char *infostr;
          
          /* Don't let cleanup() remove the socket - the child is
             responsible for doing that.  */
          cleanup_socket = 0;

          close (fd);
          
          /* Create the info string: <name>:<pid>:<protocol_version> */
          if (asprintf (&infostr, "DIRMNGR_INFO=%s:%lu:1",
                        socket_name, (ulong)pid ) < 0)
            {
              log_error (_("out of core\n"));
              kill (pid, SIGTERM);
              dirmngr_exit (1);
            }
          /* Print the environment string, so that the caller can use
             shell's eval to set it */
          if (csh_style)
            {
              *strchr (infostr, '=') = ' ';
              printf ( "setenv %s\n", infostr);
            }
          else
            {
              printf ( "%s; export DIRMNGR_INFO;\n", infostr);
            }
          free (infostr);
          exit (0); 
          /*NEVER REACHED*/
        } /* end parent */
      
      
      /* 
         This is the child
       */

      /* Detach from tty and put process into a new session */
      if (!nodetach )
        { 
          int i;
          unsigned int oldflags;

          /* Close stdin, stdout and stderr unless it is the log stream */
          for (i=0; i <= 2; i++)
            {
              if (!log_test_fd (i) && i != fd )
                close (i);
            }
          if (setsid() == -1)
            {
              log_error (_("setsid() failed: %s\n"), strerror(errno) );
              dirmngr_exit (1);
            }

          log_get_prefix (&oldflags);
          log_set_prefix (NULL, oldflags | JNLIB_LOG_RUN_DETACHED);
          opt.running_detached = 1;

          if (chdir("/"))
            {
              log_error (_("chdir to / failed: %s\n"), strerror (errno));
              dirmngr_exit (1);
            }
        }

      launch_reaper_thread ();
      cert_cache_init ();
      crl_cache_init ();
      handle_connections (fd);
      close (fd);
      shutdown_reaper ();
    }
  else if (cmd == aListCRLs)
    {
      /* Just list the CRL cache and exit. */
      if (argc)
        wrong_args ("--list-crls");
      launch_reaper_thread ();
      crl_cache_init ();
      crl_cache_list (stdout);
    }
  else if (cmd == aLoadCRL)
    {
      struct server_control_s ctrlbuf;

      memset (&ctrlbuf, 0, sizeof ctrlbuf);
      dirmngr_init_default_ctrl (&ctrlbuf);

      launch_reaper_thread ();
      cert_cache_init ();
      crl_cache_init ();
      if (!argc)
        rc = crl_cache_load (&ctrlbuf, NULL);
      else
        {
          for (; !rc && argc; argc--, argv++)
            rc = crl_cache_load (&ctrlbuf, *argv);
        }
    }
  else if (cmd == aFetchCRL)
    {
      ksba_reader_t reader;
      struct server_control_s ctrlbuf;

      if (argc != 1)
        wrong_args ("--fetch-crl URL");

      memset (&ctrlbuf, 0, sizeof ctrlbuf);
      dirmngr_init_default_ctrl (&ctrlbuf);

      launch_reaper_thread ();
      cert_cache_init ();
      crl_cache_init ();
      rc = crl_fetch (&ctrlbuf, argv[0], &reader);
      if (rc)
        log_error (_("fetching CRL from `%s' failed: %s\n"),
                     argv[0], gpg_strerror (rc));
      else
        {
          rc = crl_cache_insert (&ctrlbuf, argv[0], reader); 
          if (rc)
            log_error (_("processing CRL from `%s' failed: %s\n"),
                       argv[0], gpg_strerror (rc));
          crl_close_reader (reader);
        }
    }
  else if (cmd == aFlush)
    {
      /* Delete cache and exit. */
      if (argc)
        wrong_args ("--flush");
      rc = crl_cache_flush();
    }
  else if (cmd == aGPGConfList)
    {
      char *filename;
      /* List options and default values in the GPG Conf format.  */

/* The following list is taken from gnupg/tools/gpgconf-comp.c.  */
/* Option flags.  YOU MUST NOT CHANGE THE NUMBERS OF THE EXISTING
   FLAGS, AS THEY ARE PART OF THE EXTERNAL INTERFACE.  */
#define GC_OPT_FLAG_NONE	0UL
/* The DEFAULT flag for an option indicates that the option has a
   default value.  */
#define GC_OPT_FLAG_DEFAULT	(1UL << 4)
/* The DEF_DESC flag for an option indicates that the option has a
   default, which is described by the value of the default field.  */
#define GC_OPT_FLAG_DEF_DESC	(1UL << 5)
/* The NO_ARG_DESC flag for an option indicates that the argument has
   a default, which is described by the value of the ARGDEF field.  */
#define GC_OPT_FLAG_NO_ARG_DESC	(1UL << 6)

      /* First the configuration file.  This is not an option, but it
	 is vital information for GPG Conf.  */
      if (!config_filename)
        config_filename = make_filename (opt.homedir, "dirmngr.conf", NULL );

      printf ("gpgconf-dirmngr.conf:%lu:\"%s\n",
              GC_OPT_FLAG_DEFAULT, config_filename);

      printf ("verbose:%lu:\n", GC_OPT_FLAG_NONE);
      printf ("quiet:%lu:\n", GC_OPT_FLAG_NONE);
      printf ("debug-level:%lu:\"none\n", GC_OPT_FLAG_DEFAULT);
      printf ("log-file:%lu:\n", GC_OPT_FLAG_NONE);
      printf ("force:%lu:\n", GC_OPT_FLAG_NONE);

      /* --csh and --sh are mutually exclusive, something we can not
         express in GPG Conf.  --options is only usable from the
         command line, really.  --debug-all interacts with --debug,
         and having both of them is thus problematic.  --no-detach is
         also only usable on the command line.  --batch is unused.  */

      filename = make_filename (opt.homedir, 
                                opt.system_daemon?
                                "ldapservers.conf":"dirmngr_ldapservers.conf",
                                NULL);
      printf ("ldapserverlist-file:%lu:\"%s\n", GC_OPT_FLAG_DEFAULT, filename);
      xfree (filename);

      printf ("ldaptimeout:%lu:%u\n",
              GC_OPT_FLAG_DEFAULT, DEFAULT_LDAP_TIMEOUT);
      printf ("max-replies:%lu:%u\n",
              GC_OPT_FLAG_DEFAULT, DEFAULT_MAX_REPLIES);
      printf ("allow-ocsp:%lu:\n", GC_OPT_FLAG_NONE);
      printf ("ocsp-responder:%lu:\n", GC_OPT_FLAG_NONE);
      printf ("ocsp-signer:%lu:\n", GC_OPT_FLAG_NONE);

      printf ("faked-system-time:%lu:\n", GC_OPT_FLAG_NONE);
      printf ("no-greeting:%lu:\n", GC_OPT_FLAG_NONE);

      printf ("disable-http:%lu:\n", GC_OPT_FLAG_NONE);
      printf ("disable-ldap:%lu:\n", GC_OPT_FLAG_NONE);
      printf ("honor-http-proxy:%lu\n", GC_OPT_FLAG_NONE);
      printf ("http-proxy:%lu:\n", GC_OPT_FLAG_NONE);
      printf ("ldap-proxy:%lu:\n", GC_OPT_FLAG_NONE);
      printf ("only-ldap-proxy:%lu:\n", GC_OPT_FLAG_NONE);
      printf ("ignore-ldap-dp:%lu:\n", GC_OPT_FLAG_NONE);
      printf ("ignore-http-dp:%lu:\n", GC_OPT_FLAG_NONE);
      printf ("ignore-ocsp-service-url:%lu:\n", GC_OPT_FLAG_NONE);
      /* Note: The next one is to fix a typo in gpgconf - should be
         removed eventually. */
      printf ("ignore-ocsp-servic-url:%lu:\n", GC_OPT_FLAG_NONE);
    }
  cleanup ();
  return !!rc;
}


static void
cleanup (void)
{
  crl_cache_deinit ();
  cert_cache_deinit (1);

  free_ldapservers_list (opt.ldapservers);
  opt.ldapservers = NULL;

  if (cleanup_socket)
    {
      cleanup_socket = 0;
      if (socket_name && *socket_name)
        remove (socket_name);
    }
}


void 
dirmngr_exit (int rc)
{
  cleanup ();
  exit (rc);
}


void
dirmngr_init_default_ctrl (ctrl_t ctrl)
{
  /* Nothing for now. */
}


/* Create a list of LDAP servers from the file FILENAME. Returns the
   list or NULL in case of errors. 

   The format fo such a file is line oriented where empty lines and
   lines starting with a hash mark are ignored.  All other lines are
   assumed to be colon seprated with these fields:

   1. field: Hostname
   2. field: Portnumber
   3. field: Username 
   4. field: Password
   5. field: Base DN

*/
static ldap_server_t
parse_ldapserver_file (const char* filename)
{
  char buffer[1024];
  char *p, *endp;
  ldap_server_t server, serverstart, *serverend;
  int c;
  unsigned int lineno = 0;
  FILE *fp;
  int fieldno;

  fp = fopen (filename, "r");
  if (!fp)
    {
      log_error (_("error opening `%s': %s\n"), filename, strerror (errno));
      return NULL;
    }

  serverstart = NULL;
  serverend = &serverstart;
  while (fgets (buffer, sizeof buffer, fp))
    {
      int fail = 0;

      lineno++;
      if (!*buffer || buffer[strlen(buffer)-1] != '\n')
        {
          if (*buffer && feof (fp))
            ; /* Last line not terminated - continue. */
          else
            {
              log_error (_("%s:%u: line too long - skipped\n"),
                         filename, lineno);
              while ( (c=fgetc (fp)) != EOF && c != '\n')
                ; /* Skip until end of line. */
              continue;
            }
        }
      /* Skip empty and comment lines.*/
      for (p=buffer; spacep (p); p++)
        ;
      if (!*p || *p == '\n' || *p == '#')
        continue;

      /* Parse the colon separated fields. */
      server = xcalloc (1, sizeof *server);
      for (fieldno=1, p = buffer; p; p = endp, fieldno++ )
        {
          endp = strchr (p, ':');
          if (endp)
            *endp++ = '\0';
          trim_spaces (p);
          switch (fieldno)
            {
            case 1:
              if (*p)
                server->host = xstrdup (p);
              else
                {
                  log_error (_("%s:%u: no hostname given\n"),
                             filename, lineno);
                  fail = 1;
                }
              break;
                
            case 2:
              if (*p)
                server->port = atoi (p);
              break;

            case 3:
              if (*p)
                server->user = xstrdup (p);
              break;

            case 4:
              if (*p && !server->user)
                {
                  log_error (_("%s:%u: password given without user\n"), 
                             filename, lineno);
                  fail = 1;
                }
              else if (*p)
                server->pass = xstrdup (p);
              break;

            case 5:
              if (*p)
                server->base = xstrdup (p);
              break;

            default:
              /* (We silently ignore extra fields.) */
              break;
            }
        }
          
      if (fail)
        {
          log_info (_("%s:%u: skipping this line\n"), filename, lineno);
          free_ldapservers_list (server);
        }
      else
        {
          *serverend = server;
          serverend = &server->next;
        }
    } 
  
  if (ferror (fp))
    log_error (_("error reading `%s': %s\n"), filename, strerror (errno));
  fclose (fp);

  return serverstart;
}


/* Release the list of SERVERS. As usual it is okay to call this
   fucntion with SERVERS passed as NULL. */
static void 
free_ldapservers_list (ldap_server_t servers)
{
  while (servers)
    {
      ldap_server_t tmp = servers->next;
      xfree (servers->host);
      xfree (servers->user);
      if (servers->pass)
        memset (servers->pass, 0, strlen (servers->pass));
      xfree (servers->pass);
      xfree (servers->base);
      xfree (servers);
      servers = tmp;
    }
}



/*
   Stuff used in daemon mode.  
 */



/* Reread parts of the configuration.  Note, that this function is
   obviously not thread-safe and should only be called from the PTH
   signal handler. 

   Fixme: Due to the way the argument parsing works, we create a
   memory leak here for all string type arguments.  There is currently
   no clean way to tell whether the memory for the argument has been
   allocated or points into the process' original arguments.  Unless
   we have a mechanism to tell this, we need to live on with this. */
static void
reread_configuration (void)
{
  ARGPARSE_ARGS pargs;
  FILE *fp;
  unsigned int configlineno = 0;
  int dummy;

  if (!config_filename)
    return; /* No config file. */

  fp = fopen (config_filename, "r");
  if (!fp)
    {
      log_error (_("option file `%s': %s\n"),
                 config_filename, strerror(errno) );
      return;
    }

  parse_rereadable_options (NULL, 1); /* Start from the default values. */

  memset (&pargs, 0, sizeof pargs);
  dummy = 0;
  pargs.argc = &dummy;
  pargs.flags = 1;  /* do not remove the args */
  while (optfile_parse (fp, config_filename, &configlineno, &pargs, opts) )
    {
      if (pargs.r_opt < -1)
        pargs.err = 1; /* Print a warning. */
      else /* Try to parse this option - ignore unchangeable ones. */
        parse_rereadable_options (&pargs, 1);
    }
  fclose (fp);

  set_debug ();
}



/* The signal handler. */
static void
handle_signal (int signo)
{
  switch (signo)
    {
    case SIGHUP:
      log_info (_("SIGHUP received - "
                  "re-reading configuration and flushing caches\n"));
      reread_configuration ();
      cert_cache_deinit (0);
      crl_cache_deinit ();
      cert_cache_init ();
      crl_cache_init ();
      break;
      
    case SIGUSR1:
      cert_cache_print_stats ();
      break;
      
    case SIGUSR2:
      log_info (_("SIGUSR2 received - no action defined\n"));
      break;

    case SIGTERM:
      if (!shutdown_pending)
        log_info (_("SIGTERM received - shutting down ...\n"));
      else
        log_info (_("SIGTERM received - still %d active connections\n"),
                  active_connections);
      shutdown_pending++;
      if (shutdown_pending > 2)
        {
          log_info (_("shutdown forced\n"));
          log_info ("%s %s stopped\n", strusage(11), strusage(13) );
          cleanup ();
          dirmngr_exit (0);
	}
      break;
        
    case SIGINT:
      log_info (_("SIGINT received - immediate shutdown\n"));
      log_info( "%s %s stopped\n", strusage(11), strusage(13));
      cleanup ();
      dirmngr_exit (0);
      break;

    default:
      log_info (_("signal %d received - no action defined\n"), signo);
    }
}



/* Helper to call a connection's main fucntion. */
static void *
start_connection_thread (void *arg)
{
  int fd = (int)arg;

  active_connections++;
  if (opt.verbose)
    log_info (_("handler for fd %d started\n"), fd);

  start_command_handler (fd);

  if (opt.verbose)
    log_info (_("handler for fd %d terminated\n"), fd);
  active_connections--;
  
  return NULL;
}


/* Main loop in daemon mode. */
static void
handle_connections (int listen_fd)
{
  pth_attr_t tattr;
  pth_event_t ev;
  sigset_t sigs;
  int signo;
  struct sockaddr_un paddr;
  socklen_t plen = sizeof( paddr );
  int fd;

  tattr = pth_attr_new();
  pth_attr_set (tattr, PTH_ATTR_JOINABLE, 0);
  pth_attr_set (tattr, PTH_ATTR_STACK_SIZE, 1024*1024);
  pth_attr_set (tattr, PTH_ATTR_NAME, "dirmngr");

  sigemptyset (&sigs );
  sigaddset (&sigs, SIGHUP);
  sigaddset (&sigs, SIGUSR1);
  sigaddset (&sigs, SIGUSR2);
  sigaddset (&sigs, SIGINT);
  sigaddset (&sigs, SIGTERM);
  ev = pth_event (PTH_EVENT_SIGS, &sigs, &signo);

  for (;;)
    {
      if (shutdown_pending)
        {
          if (!active_connections)
            break; /* ready */

          /* Do not accept anymore connections but wait for existing
             connections to terminate.  */
          signo = 0;
          pth_wait (ev);
          if (pth_event_occurred (ev) && signo)
            handle_signal (signo);
          continue;
	}

      fd = pth_accept_ev (listen_fd, (struct sockaddr *)&paddr, &plen, ev);
      if (fd == -1)
        {
          if (pth_event_occurred (ev))
            {
              handle_signal (signo);
              continue;
	    }
          log_error (_("accept failed: %s - waiting 1s\n"), strerror (errno));
          pth_sleep (1);
          continue;
	}

      if (!pth_spawn (tattr, start_connection_thread, (void*)fd))
        {
          log_error (_("error spawning connection handler: %s\n"),
                     strerror (errno) );
          close (fd);
	}
    }
  
  pth_event_free (ev, PTH_FREE_ALL);
  cleanup ();
  log_info ("%s %s stopped\n", strusage(11), strusage(13));
}
