/* ============================================================
 *
 * This file is a part of digiKam project
 * https://www.digikam.org
 *
 * Date        : 2009-02-04
 * Description : a command line tool to test qt PGF interface
 *
 * SPDX-FileCopyrightText: 2009-2022 by Gilles Caulier <caulier dot gilles at gmail dot com>
 *
 * SPDX-License-Identifier: GPL-2.0-or-later
 *
 * ============================================================ */

// C++ includes

#include <ctime>

// Qt includes

#include <QBuffer>
#include <QByteArray>
#include <QDataStream>

#include <QFile>
#include <QIODevice>
#include <QImage>

// Local includes

#include "digikam_debug.h"
#include "pgfutils.h"

using namespace Digikam;

int main(int /*argc*/, char** /*argv*/)
{
    clock_t    start, end;
    QImage     img;
    QByteArray pgfData, jpgData, pngData;

    // QImage => PGF conversion

    qCDebug(DIGIKAM_TESTS_LOG) << "Using LibPGF version: " << PGFUtils::libPGFVersion();

    img.load(QLatin1String("test.png"));

    qCDebug(DIGIKAM_TESTS_LOG) << "Generate PGF file using file stream";

    // First, write QImage as PGF file using file stream
    if (!PGFUtils::writePGFImageFile(img, QLatin1String("test-filestream.pgf"), 0, true))
    {
        qCDebug(DIGIKAM_TESTS_LOG) << "writePGFImageData failed...";
        return -1;
    }

    qCDebug(DIGIKAM_TESTS_LOG) << "Generate PGF file using data stream";

    start = clock();

    // Second, write QImage as PGF file using data stream
    if (!PGFUtils::writePGFImageData(img, pgfData, 0, true))
    {
        qCDebug(DIGIKAM_TESTS_LOG) << "writePGFImageData failed...";
        return -1;
    }

    end   = clock();

    qCDebug(DIGIKAM_TESTS_LOG) << "PGF Encoding time: " << double(end - start)/CLOCKS_PER_SEC << " s";

    // Write PGF file.

    QFile file(QLatin1String("test-datastream.pgf"));

    if (!file.open(QIODevice::WriteOnly))
    {
        qCDebug(DIGIKAM_TESTS_LOG) << "Cannot open PGF file to write...";
        return -1;
    }

    QDataStream stream(&file);
    stream.writeRawData(pgfData.data(), pgfData.size());
    file.close();

    // PGF => QImage conversion

    qCDebug(DIGIKAM_TESTS_LOG) << "Load PGF file generated by data stream";

    start = clock();

    if (!PGFUtils::readPGFImageData(pgfData, img, true))
    {
        qCDebug(DIGIKAM_TESTS_LOG) << "readPGFImageData failed...";
        return -1;
    }

    end   = clock();

    img.save(QLatin1String("test2.png"), "PNG");

    qCDebug(DIGIKAM_TESTS_LOG) << "PGF Decoding time: " << double(end - start)/CLOCKS_PER_SEC << " s";

    // JPEG tests for comparisons.

    img.load(QLatin1String("test.png"));

    qCDebug(DIGIKAM_TESTS_LOG) << "Generate JPG file to compare performances";

    start = clock();

    QBuffer buffer(&jpgData);
    buffer.open(QIODevice::WriteOnly);
    img.save(&buffer, "JPEG", 85);  // Here we will use JPEG quality = 85 to reduce artifacts.

    if (jpgData.isNull())
    {
        qCDebug(DIGIKAM_TESTS_LOG) << "Save JPG image data to byte array failed...";
        return -1;
    }

    end   = clock();

    qCDebug(DIGIKAM_TESTS_LOG) << "JPG Encoding time: " << double(end - start)/CLOCKS_PER_SEC << " s";

    start = clock();

    buffer.open(QIODevice::ReadOnly);
    img.load(&buffer, "JPEG");

    if (jpgData.isNull())
    {
        qCDebug(DIGIKAM_TESTS_LOG) << "Load JPG image data from byte array failed...";
        return -1;
    }

    end   = clock();

    qCDebug(DIGIKAM_TESTS_LOG) << "JPG Decoding time: " << double(end - start)/CLOCKS_PER_SEC << " s";

    // PNG tests for comparisons.

    img.load(QLatin1String("test.png"));

    qCDebug(DIGIKAM_TESTS_LOG) << "Generate PNG file to compare performances";

    start = clock();

    QBuffer buffer2(&pngData);
    buffer2.open(QIODevice::WriteOnly);
    img.save(&buffer2, "PNG", 100);

    if (pngData.isNull())
    {
        qCDebug(DIGIKAM_TESTS_LOG) << "Save PNG image data to byte array failed...";
        return -1;
    }

    end   = clock();

    qCDebug(DIGIKAM_TESTS_LOG) << "PNG Encoding time: " << double(end - start)/CLOCKS_PER_SEC << " s";

    start = clock();

    buffer2.open(QIODevice::ReadOnly);
    img.load(&buffer2, "PNG");

    if (pngData.isNull())
    {
        qCDebug(DIGIKAM_TESTS_LOG) << "Load PNG image data from byte array failed...";
        return -1;
    }

    end   = clock();

    qCDebug(DIGIKAM_TESTS_LOG) << "PNG Decoding time: " << double(end - start)/CLOCKS_PER_SEC << " s";

    return 0;
}
