/*
 *   This file is part of Dianara
 *   Copyright 2012-2013  JanKusanagi <janjabber@gmail.com>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA .
 */

#include "commenterblock.h"

CommenterBlock::CommenterBlock(PumpController *pumpController, QWidget *parent) : QWidget(parent)
{
    this->pController = pumpController;


    this->showAllCommentsLinkLabel = new QLabel("<a href=\"showall://\">"
                                                + tr("Show All Comments")
                                                + "</a>");
    showAllCommentsLinkLabel->setContextMenuPolicy(Qt::NoContextMenu);
    QFont showAllFont;
    showAllFont.setPointSize(showAllFont.pointSize() - 3);
    showAllCommentsLinkLabel->setFont(showAllFont);
    connect(showAllCommentsLinkLabel, SIGNAL(linkActivated(QString)),
            this, SLOT(requestAllComments()));


    commentsLayout = new QVBoxLayout();
    commentsWidget = new QWidget();
    commentsWidget->setLayout(commentsLayout);

    this->commentsScrollArea = new QScrollArea();
    commentsScrollArea->setWidget(commentsWidget);
    commentsScrollArea->setWidgetResizable(true);


    // Hide these until setComments() is called, if there are any comments
    showAllCommentsLinkLabel->hide();
    commentsScrollArea->hide();

    scrollToBottomTimer = new QTimer(this);
    scrollToBottomTimer->setSingleShot(true);
    connect(scrollToBottomTimer, SIGNAL(timeout()),
            this, SLOT(scrollCommentsToBottom()));



    this->commentComposer = new Composer(false); // forPublisher = false
    this->commentComposer->setVerticalScrollBarPolicy(Qt::ScrollBarAsNeeded);
    this->commentComposer->setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
    this->commentComposer->setSizePolicy(QSizePolicy::Minimum,
                                         QSizePolicy::Minimum);

    connect(commentComposer, SIGNAL(editingFinished()),  this, SLOT(sendComment()));
    connect(commentComposer, SIGNAL(editingCancelled()), this, SLOT(setMinimumMode()));


    // Info label about sending status
    this->statusInfoLabel = new QLabel();
    statusInfoLabel->setAlignment(Qt::AlignCenter);
    statusInfoLabel->setWordWrap(true);
    showAllFont.setPointSize(showAllFont.pointSize() + 1);
    statusInfoLabel->setFont(showAllFont);


    this->commentButton = new QPushButton(QIcon::fromTheme("mail-message-new"),
                                          tr("&Comment"));
    connect(commentButton, SIGNAL(clicked()), this, SLOT(sendComment()));

    this->cancelButton = new QPushButton(QIcon::fromTheme("dialog-cancel"),
                                         tr("C&ancel"));
    connect(cancelButton, SIGNAL(clicked()), commentComposer, SLOT(cancelPost()));


    mainLayout = new QGridLayout();
    mainLayout->addWidget(showAllCommentsLinkLabel, 0, 0, 1, 4, Qt::AlignRight | Qt::AlignTop);
    mainLayout->addWidget(commentsScrollArea,       1, 0, 1, 4);
    mainLayout->addWidget(commentComposer,          2, 0, 3, 3);
    mainLayout->addWidget(statusInfoLabel,          2, 3, 1, 1, Qt::AlignCenter);
    mainLayout->addWidget(commentButton,            3, 3, 1, 1);
    mainLayout->addWidget(cancelButton,             4, 3, 1, 1);
    this->setLayout(mainLayout);


    this->setMinimumMode();

    qDebug() << "Commenter created";
}


CommenterBlock::~CommenterBlock()
{
    qDebug() << "Commenter destroyed";
}



void CommenterBlock::clearComments()
{
    foreach (Comment *comment, commentsInBlock)
    {
        comment->deleteLater();
    }

    commentsInBlock.clear();
}



void CommenterBlock::setComments(QVariantList commentsList)
{
    this->clearComments();

    if (commentsList.size() > 0)
    {
        foreach (QVariant commentVariant, commentsList)
        {
            Comment *comment = new Comment(this->pController,
                                           commentVariant.toMap(),
                                           this);
            this->commentsInBlock.append(comment);

            // Add the comment on top, since the list comes in reverse
            this->commentsLayout->insertWidget(0, comment);
        }


        this->showAllCommentsLinkLabel->show();
        this->commentsScrollArea->show();

        // Move scrollbar to the bottom
        this->scrollCommentsToBottom(); // First, try
        // Then, 50 msec later, try again, in case there was no scrollbar before,
        // and the first try didn't work
        // Trying immediately first avoids a flicker-like effect sometimes
        scrollToBottomTimer->start(50);
    }
}




/*******************************************************************/
/****************************** SLOTS ******************************/
/*******************************************************************/



void CommenterBlock::setMinimumMode()
{
    this->commentComposer->hide();
    statusInfoLabel->clear();
    this->statusInfoLabel->hide();
    this->commentButton->hide();
    this->cancelButton->hide();

    // Clear formatting options like bold or italic
    this->commentComposer->setCurrentCharFormat(QTextCharFormat());

    // Unblock updates to timeline, since we're done commenting
    this->pController->setUpdatesToTimelineBlocked(false);
}



void CommenterBlock::setFullMode()
{
    this->commentComposer->show();
    this->statusInfoLabel->show();
    this->commentButton->show();
    this->cancelButton->show();

    this->commentComposer->setFocus();

    // Block updates to timeline while we're commenting...
    this->pController->setUpdatesToTimelineBlocked(true);

    // FIXME: this fails if user clicks "comment" in 2 posts, and then
    // cancels or sends one of them
}


void CommenterBlock::requestAllComments()
{
    emit allCommentsRequested();
}


/*
 * Called when commentPosted() is emmited by PumpController,
 * which means comment posted successfully.
 *
 */
void CommenterBlock::onPostingCommentOk()
{
    this->requestAllComments();


    // Clear info message
    this->statusInfoLabel->clear();

    // Comment was added successfully, so we can re-enable things
    this->setEnabled(true);

    // Erase the text from the comment box...
    this->commentComposer->erase();

    // and since we're done posting the comment, hide this
    setMinimumMode();


    disconnect(pController, SIGNAL(commentPosted()),
               this, SLOT(onPostingCommentOk()));
}


/*
 * Executed when commentPostingFailed() signal is received from PumpController
 *
 */
void CommenterBlock::onPostingCommentFailed()
{
    qDebug() << "Posting the comment failed, re-enabling Commenter";

    // Alert about the error
    this->statusInfoLabel->setText(tr("Posting comment failed.\nTry again."));

    // Re-enable things, so user can try again
    this->setEnabled(true);
    this->commentComposer->setFocus();

    disconnect(pController, SIGNAL(commentPostingFailed()),
               this, SLOT(onPostingCommentFailed()));
}




void CommenterBlock::sendComment()
{
    qDebug() << "Commenter character count:" << commentComposer->textCursor().document()->characterCount();

    // If there's some text in the comment, send it
    if (commentComposer->textCursor().document()->characterCount() > 1)
    {
        emit commentSent(commentComposer->toHtml());

        connect(pController, SIGNAL(commentPosted()),
                this, SLOT(onPostingCommentOk()));
        connect(pController, SIGNAL(commentPostingFailed()),
                this, SLOT(onPostingCommentFailed()));

        this->statusInfoLabel->setText(tr("Sending comment..."));
        this->setDisabled(true);
    }
    else
    {
        qDebug() << "Can't post, comment is empty";
    }
}

/*
 * Called by the QTimer
 *
 */
void CommenterBlock::scrollCommentsToBottom()
{
    commentsScrollArea->verticalScrollBar()->triggerAction(QScrollBar::SliderToMaximum);
}
