/*
 * Diagnostics - a unified framework for code annotation, logging,
 * program monitoring, and unit-testing.
 *
 * Copyright (C) 2002-2005 Christian Schallhart
 *               2006-2007 model.in.tum.de group
 *  
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */


/**
 * @file diagnostics/util/marshaling.cpp
 *
 * $Id$
 *
 * @author Christian Schallhart
 *
 * @brief implementation of read/write methods
 */

#include <diagnostics/util/marshaling.hpp>

DIAGNOSTICS_NAMESPACE_BEGIN;
INTERNAL_NAMESPACE_BEGIN;

bool read(::std::istream & stream, unsigned long & target)
{
    unsigned char target_buf[4];
    stream.read(reinterpret_cast<char*>(target_buf),4);

    target
	=static_cast<unsigned long>(target_buf[0])
	+static_cast<unsigned long>(target_buf[1])*256
	+static_cast<unsigned long>(target_buf[2])*256*256
	+static_cast<unsigned long>(target_buf[3])*256*256*256;
    
    return stream.good();
}

void write(::std::ostream & stream, unsigned long source)
{
    unsigned char source_buf[4];
    source_buf[0]=static_cast<unsigned char>(source & 255); source>>=8;
    source_buf[1]=static_cast<unsigned char>(source & 255); source>>=8;
    source_buf[2]=static_cast<unsigned char>(source & 255); source>>=8;
    source_buf[3]=static_cast<unsigned char>(source);

    stream.write(reinterpret_cast<char*>(source_buf),4);
}

bool read(::std::istream & stream, ::std::string & target)
{
    // read length
    unsigned long length;
    if(!read(stream,length)) return false;

    // read data
    char * const target_buffer(new char[length]);
    stream.read(target_buffer,length);
    if(!stream.good()) {
	delete target_buffer;
	return false;
    }
    
    // convert
    target= ::std::string(target_buffer,length);
    delete target_buffer;
    return true;
}


void write(::std::ostream & stream, ::std::string const & source)
{
    write(stream,static_cast<unsigned long>(source.size()));
    stream.write(source.data(),static_cast< ::std::streamsize>(source.size()));
}


bool read(::std::istream & stream, ::std::map< ::std::string, ::std::string> & target)
{
    unsigned long size;
    if(!read(stream,size)) return false;
    unsigned long i;
    for(i=0;i<size;++i){
	::std::pair< ::std::string, ::std::string> id_value_pair;
	if(!read(stream,id_value_pair.first)) return false;
	if(!read(stream,id_value_pair.second)) return false;
	target.insert(id_value_pair);
    }
    return true;
}

void write(::std::ostream & stream, ::std::map< ::std::string, ::std::string> const & source)
{
    typedef ::std::map< ::std::string, ::std::string> data_t;
    data_t::const_iterator current(source.begin());
    data_t::const_iterator const end(source.end());
    
    write(stream,static_cast<unsigned long>(source.size()));

    for(;current!=end;++current){
	write(stream,current->first);
	write(stream,current->second);
    }
}

INTERNAL_NAMESPACE_END;
DIAGNOSTICS_NAMESPACE_END;
// vim:ts=4:sw=4
