/*
 * Diagnostics - a unified framework for code annotation, logging,
 * program monitoring, and unit-testing.
 *
 * Copyright (C) 2002-2005 Christian Schallhart
 *               2006-2007 model.in.tum.de group
 *  
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */


/**
 * @file diagnostics/frame/logging_config.t.cpp
 *
 * @brief [LEVEL: beta] testing @ref diagnostics::Logging_Config
 *
 * $Id: logging_config.t.cpp,v 1.16 2005/06/23 09:54:19 esdentem Exp $
 *
 * @author Christian Schallhart
 *
 * @attention If an error occurs in this test-suite, the whole thing
 * might crash away, or provoced errors (since the logging is in
 * disray) may arise.
 */
#include <diagnostics/unittest.hpp>

#include <diagnostics/frame/logging_config.hpp>

// used components
#include <diagnostics/unittest/test_system_exception.hpp>
#include <diagnostics/frame/logging_config_exception.hpp>
#include <diagnostics/frame/record.hpp>
// furthermore used components -- it is circular -- however, the system is testing itself
#include <diagnostics/logger/file_logger.hpp>
#include <diagnostics/unittest/test_system/stream_test_system.hpp>

// test support
#include <diagnostics/util/dummy_logger.ts.hpp>

#include <vector>
#include <fstream>

#define TEST_COMPONENT_NAME Logging_Config
#define TEST_COMPONENT_NAMESPACE diagnostics

static ::diagnostics::testing::Dummy_Logger * static_dummy_logger=NULL;

DIAGNOSTICS_NAMESPACE_BEGIN;
TEST_NAMESPACE_BEGIN;
TEST_COMPONENT_TEST_NAMESPACE_BEGIN;
using namespace unittest;

/**
 * @brief registering and unregistering a @ref Logger
 */
void register_unregister(Test_Data & test_data)
{
    Dummy_Logger * dummy_logger(new Dummy_Logger);
    TEST_ASSERT(dummy_logger->is_closed()==0);
    TEST_ASSERT(dummy_logger->records().size()==0);

    Logging_Config::register_logger(dummy_logger);
    TEST_ASSERT(dummy_logger->is_closed()==0);
    TEST_ASSERT(dummy_logger->records().size()==1);
    TEST_ASSERT(dummy_logger->records()[0].type()==TYPE_LOG_OPEN);
    TEST_ASSERT(dummy_logger->records()[0].level()==LEVEL_SYSTEM);
    // line == 1 iff logger is an initial one 
    TEST_ASSERT(dummy_logger->records()[0].line()!=1);
    
    Logging_Config::unregister_logger(dummy_logger);
    TEST_ASSERT(dummy_logger->is_closed()==0);
    TEST_ASSERT(dummy_logger->records().size()==2);
    TEST_ASSERT(dummy_logger->records()[1].type()==TYPE_LOG_CLOSE);
    TEST_ASSERT(dummy_logger->records()[1].level()==LEVEL_SYSTEM);
    // line == 1 iff logger is an final one 
    TEST_ASSERT(dummy_logger->records()[0].line()!=1);
}

/**
 * @brief registering and unregistering a @ref Logger twice,
 * unregistering without registering, handing over NULL
 */
void invalid_register_unregister(Test_Data & test_data)
{
    Dummy_Logger * dummy_logger(new Dummy_Logger);
    TEST_ASSERT(dummy_logger->is_closed()==0);
    TEST_ASSERT(dummy_logger->records().size()==0);

    Logging_Config::register_logger(dummy_logger);
    TEST_ASSERT(dummy_logger->is_closed()==0);
    TEST_ASSERT(dummy_logger->records().size()==1);
    TEST_ASSERT(dummy_logger->records()[0].type()==TYPE_LOG_OPEN);
    TEST_ASSERT(dummy_logger->records()[0].level()==LEVEL_SYSTEM);
    // line == 1 iff logger is an initial one 
    TEST_ASSERT(dummy_logger->records()[0].line()!=1);

    // same logger registered twice
    TEST_THROWING_BLOCK_ENTER;
    Logging_Config::register_logger(dummy_logger);    
    TEST_THROWING_BLOCK_EXIT(Logging_Config_Exception);
    // logger unchanged
    TEST_ASSERT(dummy_logger->is_closed()==0);
    TEST_ASSERT(dummy_logger->records().size()==1);

    // unregistering... ok
    Logging_Config::unregister_logger(dummy_logger);
    TEST_ASSERT(dummy_logger->is_closed()==0);
    TEST_ASSERT(dummy_logger->records().size()==2);
    TEST_ASSERT(dummy_logger->records()[1].type()==TYPE_LOG_CLOSE);
    TEST_ASSERT(dummy_logger->records()[1].level()==LEVEL_SYSTEM);
    // line == 1 iff logger is an final one 
    TEST_ASSERT(dummy_logger->records()[0].line()!=1);

    // unregistering a second time
    TEST_THROWING_BLOCK_ENTER;
    Logging_Config::unregister_logger(dummy_logger);    
    TEST_THROWING_BLOCK_EXIT(Logging_Config_Exception);
    // logger unchanged
    TEST_ASSERT(dummy_logger->is_closed()==0);
    TEST_ASSERT(dummy_logger->records().size()==2);

    // registering a NULL pointer
    TEST_THROWING_BLOCK_ENTER;
    Logging_Config::register_logger(NULL);    
    TEST_THROWING_BLOCK_EXIT(Logging_Config_Exception);

    // unregistering a NULL pointer
    TEST_THROWING_BLOCK_ENTER;
    Logging_Config::unregister_logger(NULL);    
    TEST_THROWING_BLOCK_EXIT(Logging_Config_Exception);
}

/**
 * @brief checking for the registering a Dummy_Logger with @ref
 * Logging_Config::init
 *
 * @note The converse cannot be checked. Even if we would have an
 * explicit shut-down method, we could not shutdown, since we would
 * kick out the ::diagnostics::unittest::Dummy_Logger (not the
 * Dummy_Logger) -- it has to receive close(), which will
 * cause it to drop the Test_Case at hand. 
 */
void static_init(Test_Data & test_data)
{
    TEST_ASSERT(static_dummy_logger->is_closed()==0);
    TEST_ASSERT(static_dummy_logger->records().size()==2);

    // initial message
    TEST_ASSERT(static_dummy_logger->records()[0].type()==TYPE_LOG_OPEN);
    TEST_ASSERT(static_dummy_logger->records()[0].level()==LEVEL_SYSTEM);
    // line == 1 iff logger is an initial one 
    TEST_ASSERT(static_dummy_logger->records()[0].line()==1);
    // testcase opens...
    TEST_ASSERT(static_dummy_logger->records()[1].type()==TYPE_TESTCASE_ENTER);
    TEST_ASSERT(static_dummy_logger->records()[1].level()==LEVEL_TEST);
}


TEST_COMPONENT_TEST_NAMESPACE_END;
TEST_NAMESPACE_END;
DIAGNOSTICS_NAMESPACE_END;


TEST_SUITE_BEGIN;
// must be first
TEST_NORMAL_CASE1(&static_init,LEVEL_PROD,3,false);

TEST_NORMAL_CASE(&register_unregister,LEVEL_PROD);
TEST_ABNORMAL_CASE(&invalid_register_unregister,LEVEL_PROD);
TEST_SUITE_END;


DIAGNOSTICS_NAMESPACE_BEGIN; 
void set_initial_loggers(::std::vector<Logger *> & loggers) 
{ 
  loggers.push_back(new File_Logger("test/diagnostic.log")); 
  loggers.push_back(static_dummy_logger=new ::diagnostics::testing::Dummy_Logger); 
} 
DIAGNOSTICS_NAMESPACE_END;


/**
 * @brief manual start -- we mess around with the initial logger: We
 * do NOT call @ref Logging_Config::init.
 */
int main(int argc,char ** argv) 
try 
   { 
	   ::diagnostics::unittest::Stream_Test_System 
			 test_system(static_cast< ::diagnostics::Level_t>(DEBUG__LEVEL__), 
						 argc,argv,"test/test.dat", 
						 "test/result.log",
						 ::std::cin, ::std::cout);
	   test_system.add(::diagnostics::unittest::build_test_suite()); 
	   test_system.run(); 
	   return 0; 
   } 
catch(::diagnostics::unittest::Test_System_Exception & e) 
{ 
	::std::cerr << "An Error occurred -- caught Test_System_Exception with what '" 
				<< e.what() 
				<< "'" << ::std::endl; 
}
catch(...) 
{ 
    ::std::cerr << "An Error occurred -- caught an unknown exception" << ::std::endl;
}

// vim:ts=4:sw=4
