/*
 * Unit tests for DiaCanvasItem.
 */

#include "unit-test.h"
#include "test-item.c"
#include <diacanvas/dia-canvas.h>
#include <diacanvas/dia-canvas-group.h>
#include <diacanvas/dia-canvas-line.h>
#include <diacanvas/dia-canvas-box.h>

DiaCanvas *canvas;
DiaCanvasItem *item;

#define DIA_FLAGS(i) (((DiaCanvasItem*) (i))->flags)

void
test_dia_canvas_item_setup (void)
{
	canvas = dia_canvas_new ();
	item = dia_canvas_item_create (TYPE_TEST_ITEM,
				       "parent", canvas->root, NULL);
	//g_object_ref (item);

	TEST (item != NULL);
	TEST (DIA_IS_CANVAS_ITEM (item));
	TEST ((DIA_FLAGS (item) & DIA_NEED_UPDATE) != 0);
	TEST (item->canvas == canvas);
	TEST (item->parent == canvas->root);
	TEST (item->bounds.left == 0.0);
	TEST (item->bounds.right == 0.0);
	TEST (item->bounds.top == 0.0);
	TEST (item->bounds.bottom == 0.0);
	TEST (item->handles == NULL);
}

void
test_dia_canvas_item_teardown (void)
{
	TEST (G_OBJECT (item)->ref_count == 2);
	if (canvas)
		g_object_unref (canvas);
	canvas = NULL;
	TEST (G_OBJECT (item)->ref_count == 1);
	if (item)
		g_object_unref (item);
	item = NULL;
}

TEST_BEGIN (DiaCanvasItem, test_dia_canvas_item_setup, test_dia_canvas_item_teardown)


TEST_NEW (dia_canvas_item_create)
{
	DiaCanvasItem *ti1 = NULL;
	DiaCanvasItem *ti2 = NULL;

	ti1 = dia_canvas_item_create (TYPE_TEST_ITEM, NULL);
	TEST (ti1->parent == NULL);
	TEST (ti1->canvas == NULL);
	TEST (G_OBJECT (ti1)->ref_count == 1);
	
	ti2 = dia_canvas_item_create (TYPE_TEST_ITEM, "parent", ti1, NULL);
	TEST (ti2->parent == ti1);
	TEST (TEST_ITEM (ti1)->children->data == (gpointer) ti2);

	TEST (G_OBJECT (ti1)->ref_count == 1);
	TEST (G_OBJECT (ti2)->ref_count == 2);

	g_object_unref (ti1);
}

TEST_NEW (notify::parent)
{
	DiaCanvasItem *ti1 = NULL;
	DiaCanvasItem *ti2 = NULL;

	ti1 = dia_canvas_item_create (TYPE_TEST_ITEM, NULL);
	ti2 = dia_canvas_item_create (TYPE_TEST_ITEM, "parent", ti1, NULL);

	g_object_set (ti1, "parent", canvas->root, NULL);
	TEST (DIA_CANVAS_ITEM (ti1)->parent == canvas->root);
	TEST (DIA_CANVAS_ITEM (ti2)->parent == ti1);
	TEST (DIA_CANVAS_ITEM (ti1)->canvas == canvas);
	g_message("%p %p %p", canvas, DIA_CANVAS_ITEM (ti1)->canvas, DIA_CANVAS_ITEM (ti2)->canvas);
	TEST (DIA_CANVAS_ITEM (ti2)->canvas == canvas);
}

TEST_NEW (dia_canvas_item_update_now)
{
	dia_canvas_item_update_now (item);
	TEST ((DIA_FLAGS (item) & DIA_NEED_UPDATE) == 0);
	TEST ((DIA_FLAGS ((DiaCanvasItem*) item->parent) & DIA_NEED_UPDATE) != 0);
	dia_canvas_item_update_now (canvas->root);
	TEST ((DIA_FLAGS ((DiaCanvasItem*) item->parent) & DIA_NEED_UPDATE) == 0);
}

TEST_NEW (dia_canvas_item_request_update)
{
	dia_canvas_item_update_now (canvas->root);
	TEST ((DIA_CANVAS_ITEM_FLAGS (canvas->root) & DIA_NEED_UPDATE) == 0);
	TEST ((DIA_CANVAS_ITEM_FLAGS (item) & DIA_NEED_UPDATE) == 0);
	TEST ((DIA_CANVAS_ITEM_FLAGS (item->parent) & DIA_NEED_UPDATE) == 0);

	dia_canvas_item_request_update (item);
	TEST ((DIA_CANVAS_ITEM_FLAGS (canvas->root) & DIA_NEED_UPDATE) != 0);
	TEST ((DIA_CANVAS_ITEM_FLAGS (item) & DIA_NEED_UPDATE) != 0);
	TEST ((DIA_CANVAS_ITEM_FLAGS (item->parent) & DIA_NEED_UPDATE) != 0);
}

TEST_NEW (DIA_COMPOSITE)
{
	DiaCanvasItem *ti1 = NULL, *ti2 = NULL;

	ti1 = dia_canvas_item_create (TYPE_TEST_ITEM, "parent", canvas->root, NULL);
	ti2 = dia_canvas_item_create (TYPE_TEST_ITEM, "parent", ti1, NULL);
	DIA_CANVAS_ITEM_FLAGS(ti2) |= DIA_COMPOSITE;

	
}

TEST_NEW (set_property)
{
	gboolean n;
	DiaCanvasItem *ti1;

	ti1 = dia_canvas_item_create (TYPE_TEST_ITEM, NULL);

	TEST_PROPERTY_NOTIFY (ti1, "parent", n);
	TEST_PROPERTY_NOTIFY (ti1, "visible", n);

	g_object_set (ti1, "parent", canvas->root, NULL);
	TEST (n == TRUE);

	n = FALSE;
	g_object_set (ti1, "visible", FALSE, NULL);
	TEST (n == TRUE);

	n = FALSE;
	g_object_set (ti1, "parent", NULL, NULL);
	TEST (n == TRUE);

	g_object_unref (ti1);
}

TEST_NEW(groupable_add/remove)
{
	DiaCanvasItem *ti1 = item;
	DiaCanvasItem *ti2 = dia_canvas_item_create (TYPE_TEST_ITEM,
						     "parent", canvas->root,
						     NULL);
	DiaCanvasItem *ti3 = dia_canvas_item_create (TYPE_TEST_ITEM,
						     "parent", canvas->root,
						     NULL);

	TEST (g_list_length (DIA_CANVAS_GROUP(canvas->root)->children) == 3);

	g_object_set (canvas, "allow_undo", TRUE, NULL);
	TEST (G_OBJECT (ti1)->ref_count == 2);
	TEST (G_OBJECT (ti2)->ref_count == 2);
	TEST (G_OBJECT (ti3)->ref_count == 2);

	dia_canvas_groupable_remove (DIA_CANVAS_GROUPABLE (canvas->root), ti1);
	/* One for ti1, one for the undo stack */
	TEST (G_OBJECT (ti1)->ref_count == 1); 

	dia_canvas_groupable_remove (DIA_CANVAS_GROUPABLE (canvas->root), ti2);
	TEST (G_OBJECT (ti2)->ref_count == 1); 
	dia_canvas_groupable_remove (DIA_CANVAS_GROUPABLE (canvas->root), ti3);
	TEST (G_OBJECT (ti3)->ref_count == 1); 

	TEST (g_list_length (DIA_CANVAS_GROUP(canvas->root)->children) == 0);

	/* Now add them in reverse order: */
	dia_canvas_groupable_add (DIA_CANVAS_GROUPABLE (canvas->root), ti3);
	dia_canvas_groupable_add (DIA_CANVAS_GROUPABLE (canvas->root), ti2);
	dia_canvas_groupable_add (DIA_CANVAS_GROUPABLE (canvas->root), ti1);

	TEST (g_list_length (DIA_CANVAS_GROUP(canvas->root)->children) == 3);
	// TODO: fix group adding!
	//TEST (DIA_CANVAS_GROUP(canvas->root)->children->data == ti3);
	//TEST (DIA_CANVAS_GROUP(canvas->root)->children->next->data == ti2);
	//TEST (DIA_CANVAS_GROUP(canvas->root)->children->next->next->data == ti1);

	g_object_set (canvas, "allow_undo", FALSE, NULL);
	dia_canvas_clear_undo (canvas);
	dia_canvas_clear_redo (canvas);
	g_object_unref (ti2);
	g_object_unref (ti3);
}

TEST_NEW(dia_canvas_line_connect)
{
	DiaCanvasItem *item, *connectable;
	DiaHandle *handle;

	item = dia_canvas_item_create (DIA_TYPE_CANVAS_LINE, NULL);
	connectable = dia_canvas_item_create (DIA_TYPE_CANVAS_LINE, NULL);
	handle = dia_handle_new (connectable);

	g_object_set(G_OBJECT (handle), "connectable", TRUE, NULL);
	g_print ("Connect: %d\n", dia_canvas_item_connect (item, handle));
}

TEST_NEW(test_handle_property)
{
	DiaCanvasItem *item;
	DiaHandle *h1, *h2, *h3;
	GList *handles = NULL;

	item = dia_canvas_item_create (TYPE_TEST_ITEM, NULL);
	h1 = dia_handle_new (item);
	h2 = dia_handle_new (item);
	h3 = dia_handle_new (item);

	TEST (G_OBJECT (h1)->ref_count == 1); 
	TEST (G_OBJECT (h2)->ref_count == 1); 
	TEST (G_OBJECT (h3)->ref_count == 1); 

	g_object_get (G_OBJECT (item), "handles", &handles, NULL);
	TEST (g_list_length (handles) == 3);
	//g_print ("*** before: 0x%x, %d\n", handles, g_list_length (handles));
	TEST (item->handles != handles);
	TEST (G_OBJECT (h1)->ref_count == 2); 
	TEST (G_OBJECT (h2)->ref_count == 2); 
	TEST (G_OBJECT (h3)->ref_count == 2); 

	//handles = g_list_append (handles, handle);
	//g_object_set (G_OBJECT (item), "handles", handles, NULL);

	g_object_get (G_OBJECT (item), "handles", &handles, NULL);
	TEST (g_list_length (handles) == 3);
	//g_print ("*** after: 0x%x, %d\n", handles, g_list_length (handles));

	TEST (G_OBJECT (h1)->ref_count == 3); 
	TEST (G_OBJECT (h2)->ref_count == 3); 
	TEST (G_OBJECT (h3)->ref_count == 3); 

	g_object_unref (item);

	TEST (G_OBJECT (h1)->ref_count == 2); 
	TEST (G_OBJECT (h2)->ref_count == 2); 
	TEST (G_OBJECT (h3)->ref_count == 2); 
}

TEST_NEW(undo)
{
	DiaCanvasItem *line, *box;
	DiaHandle *handle;

	g_object_set (G_OBJECT (canvas), "allow-undo", TRUE, NULL);

	box = dia_canvas_item_create (DIA_TYPE_CANVAS_BOX, 
				       "parent", canvas->root, NULL);
	TEST (G_OBJECT (box)->ref_count == 2); 
	TEST (box->parent == canvas->root); 

	line = dia_canvas_item_create (DIA_TYPE_CANVAS_LINE, 
				       "parent", canvas->root, NULL);
	TEST (G_OBJECT (line)->ref_count == 2); 
	TEST (line->parent == canvas->root); 

	handle = line->handles->data;

	TEST (DIA_IS_HANDLE (handle));

	dia_canvas_item_connect (box, handle);

	TESTEQ (G_OBJECT (box)->ref_count, 3); 
	TEST (handle->connected_to == box);
	TEST (handle->constraints != NULL);

	TESTEQ (g_list_length (box->connected_handles), 1);
	TEST (!dia_undo_manager_can_undo (dia_canvas_get_undo_manager (canvas)));

	g_message("Begin undoable transaction");
	dia_undo_manager_begin_transaction (dia_canvas_get_undo_manager (canvas));

	TEST (dia_undo_manager_in_transaction (dia_canvas_get_undo_manager (canvas)));
	//dia_canvas_groupable_remove (DIA_CANVAS_GROUPABLE (box->parent), box);
	dia_canvas_item_set_parent (box, NULL);
	TESTEQ (G_OBJECT (box)->ref_count, 3); 
	TEST (box->parent == NULL); 
	TEST (handle->constraints == NULL);
	TEST (handle->connected_to == NULL);

	g_message("Commit undoable transaction");
	dia_undo_manager_commit_transaction (dia_canvas_get_undo_manager (canvas));

	//dia_canvas_box_set_parent (line, NULL);

	TESTEQ (G_OBJECT (line)->ref_count, 3); 
	TEST (dia_undo_manager_can_undo (dia_canvas_get_undo_manager (canvas)));

	dia_undo_manager_undo_transaction (dia_canvas_get_undo_manager (canvas));

	//g_message("%p %p", handle->connected_to, box);
	TESTEQ (G_OBJECT (box)->ref_count, 5); 
	TEST (box->parent == canvas->root); 
	TEST (handle->constraints != NULL);
	TEST (handle->connected_to == box);

	g_object_unref (line);
	g_object_unref (box);
}

TEST_END ();
