#!/bin/sh
### BEGIN INIT INFO
# Provides:		dhcp-probe
# Required-Start:    $local_fs $remote_fs $syslog $network $all
# Required-Stop:     $local_fs $remote_fs $syslog $network 
# Default-Start:     2 3 4 5
# Default-Stop:      0 1 6
# Short-Description: dhcp-probe daemon to survey DHCP/BootP server on LAN
### END INIT INFO

set -e

PATH=/usr/local/sbin:/usr/local/bin:/sbin:/bin:/usr/sbin:/usr/bin
DAEMON=/usr/sbin/dhcp_probe
LABEL=${DAEMON##*/}
NAME=dhcp-probe
PIDFILE=dhcp_probe.pid
BASE_RUNNING=/var/run
BASEPIDDIR=$BASE_RUNNING/$NAME
INITPIDFILE=$BASEPIDDIR/$PIDFILE
DESC="$NAME daemon"
ETC=/etc
LOGDIR=/var/log/$NAME.log
DODTIME=2   

test -x $DAEMON || exit 0

if [ -r /lib/lsb/init-functions ]; then 
	. /lib/lsb/init-functions
fi

sourceable() {
	[ -r $1 ] || return 1
  # Just to temporary files created by VIM
	case $1 in
		*\~)
			return 1
			;;
	esac
}

#
# Debian Maintainer note :
#   This function has to be delete when the new one will
#   be fully operational
running_pid() {
	# Check if a given process pid's cmdline matches a given name
	pid=$1
	name=$2
	[ "$pid" ] || return 1
	[ -d /proc/$pid ] ||  return 1
	read cmd <<-EOF
		$(cat /proc/$pid/cmdline | tr '\0' '\n')
	EOF
	# Is this the expected child?
	[ "$cmd" = "$name" ] ||  return 1
}


running() {
  INTERFACE=$1
  PIDFILE=$INITPIDFILE.$INTERFACE
	_max_repeat=10
	_repeat=1
	while [ ! -r $PIDFILE ] && [ $_repeat -le $_max_repeat ]; do
		echo "Waiting for pid file round: $_repeat" >&2
		sleep 1
		_repeat=$(($_repeat + 1))
	done
	echo "Waited ${_repeat}s to find a $INTERFACE pid file" >&2
  if [ $_repeat -le $_max_repeat ]; then
    read pid < $PIDFILE || return 1
  fi
  for i in `pidof dhcp_probe`; do
    if [ "$i" = "$pid" ]; then
      return 0
    fi
  done
  return 1
}

start_daemon() {
	# Start one daemon for each network interface
	[ -d $BASEPIDDIR ] || mkdir -p $BASEPIDDIR || {
		echo "Failed to create missing pid file directory $BASEPIDDIR!"
		return 1
	}
	sts=0
	for config_file in $ETC/$NAME/*; do
		sourceable $config_file || continue
		. $config_file
		PIDFILE=$INITPIDFILE.$INTERFACE
		DAEMON_OPTS="-T -p $PIDFILE $INTERFACE"
		echo -n "Starting $DESC on interface $INTERFACE: "

		max_repeat=10
		repeat=1
		while ip=$(sleep 1
			echo -en "\nstart_daemon - Waiting for an ip on iface $INTERFACE, round: \
      $repeat" >&2
			while read kw snet dummy; do
				case $kw in
					(inet)
						echo ${snet%/*}
						sleep 1
						break
						;;
				esac
			done <<-EOF
				$(ip addr show dev $INTERFACE)
EOF
			) && [ -z "$ip" ] && [ $repeat -le $max_repeat ]; do
			repeat=$(($repeat + 1))
		done
		echo -e "\nwaited ${repeat}s to get an ip on $INTERFACE" >&2

		start-stop-daemon --start --quiet --pidfile $PIDFILE \
			--exec $DAEMON -- $DAEMON_OPTS
		if running $INTERFACE; then
			echo "Done."
		else
			echo "Failed!"
			sts=1
		fi
	done
	return $sts
}

stop_daemon() {
# Stop all existing dhcp_probe daemon
	for config_file in $ETC/$NAME/*; do
		sourceable $config_file || continue
		. $config_file
		PIDFILE=$INITPIDFILE.$INTERFACE
		if [ -r $PIDFILE ]; then
			echo -n "Stopping $DESC on interface $INTERFACE: "
			start-stop-daemon --stop --quiet --pidfile $PIDFILE 
			echo "$NAME."
			rm -f $PIDFILE
    else
      force_stop
		fi
	done
}

force_stop() {
	# Forcefully kill the process
	for config_file in $ETC/$NAME/*; do
		sourceable $config_file || continue
		. $config_file
		PIDFILE=$INITPIDFILE.$INTERFACE
		if [ -r "$PIDFILE" ]; then
      if running $INTERFACE; then
        start-stop-daemon -K 15 --quiet --pidfile $PIDFILE \
          --exec $DAEMON -- $DAEMON_OPTS
              # Is it really dead?
        [ -z "$DODTIME" ] || sleep "$DODTIME"
        if running $INTERFACE; then
          start-stop-daemon -K 9 --quiet --pidfile $PIDFILE \
            --exec $DAEMON -- $DAEMON_OPTS
          [ "$DODTIME" ] || sleep "$DODTIME"
          if running $INTERFACE; then
            echo "Cannot kill $LABEL (pid=$pid)!"
            exit 1
          fi
        fi
      fi
    else
      killall `basename $DAEMON`
    fi
		rm -f $PIDFILE
	done
}


case $1 in
	start)
		start_daemon
		;;

	stop)
		stop_daemon
		;;

	force-stop)
		for config_file in $ETC/$NAME/*; do
			sourceable $config_file || continue
			. $config_file
			PIDFILE=$INITPIDFILE.$INTERFACE
			echo -n "Forcefully stopping $DESC: "
			force_stop
			if ! running $INTERFACE; then
				echo "$NAME on interface $INTERFACE."
			else
				echo "Error !"
			fi
		done
		;;

	force-reload)	# Need to be improved
		echo "Reload operation is not supported -- use restart."
		exit 1
		;;

	restart)
		echo -n "Restarting $DESC: "
		stop_daemon
		[ -z "$DODTIME" ] || sleep $DODTIME
		start_daemon
		echo "$NAME."
		;;

	status)
		for config_file in $ETC/$NAME/*; do
			sourceable $config_file || continue
			. $config_file
			PIDFILE=$INITPIDFILE.$INTERFACE
			echo -n "$LABEL is "
			if running $INTERFACE;  then
				echo "running on interface $INTERFACE"
			else
				echo "not running."
				exit 1
			fi
		done
		;;

	*)
		N=/etc/init.d/$NAME
		echo "Usage: $N {start|stop|restart|status|force-stop}" >&2
		exit 1
		;;
esac

exit 0
