#!/usr/bin/python
#
# deluge.py
# Copyright (C) Zach Tibbitts 2006 <zach@collegegeek.org>
# Copyright (C) Alon Zakai    2006 <kripkensteiner@gmail.com>
# 
# Deluge is free software.
# 
# You may redistribute this file and/or modify it under the terms of the
# GNU General Public License, as published by the Free Software
# Foundation; either version 2 of the License, or (at your option)
# any later version.
# 
# This file is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
# See the GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this file.  If not, write to:
# 	The Free Software Foundation, Inc.,
# 	51 Franklin Street, Fifth Floor
# 	Boston, MA  02110-1301, USA.

import sys
import os
import torrent
import gtk

from torrentclass import Torrent, TorrentError

import delugecommon
dc = delugecommon

import gobject


class TorrentHandler:
	def __init__(self, parent):
		self.bool_continue = True
		self.parent = parent
		self.torrent_list = []
		self.auto_seed_ratio = -1

		# Call the update function every so often to show changes to our torrents
		# We use a Python generator function, not a thread

	def startUpdateTick(self):
		gobject.timeout_add(dc.UPDATE_INTERVAL, self.update().next)

	def update(self):
		while self.bool_continue:
			self.parent.UpdateButtons()

			self.parent.UpdateSessionInfo()

			self.parent.PrepareStatusBar()

			# This is here, because if done during the window's Destroy event, the data is false
			self.parent.savedSize     = self.parent.window.get_size()
			self.parent.savedPosition = self.parent.window.get_position()

			maxDownloads = self.parent.preferences.get_max_num_dl()

			# Update all torrents for the upper summary list
			my_iter = self.parent.torrentList.get_iter_first()

			while my_iter is not None:
				uniqueID = self.parent.torrentList.get_value(my_iter, 0)
				t = self.get_torrent_from_uniqueID(uniqueID)

				t.update()
				
				##
				## Code to check if a torrent is done seeding, then
				## pause it if it is.
				##
				if(self.auto_seed_ratio != -1 and not t.userPause):
					dc.debugmsg("Autoseeding code executing")
					isSeeding = t.get_is_seeding()
					
					targetRatio = self.auto_seed_ratio
					
					up = float(t.status['totalUpload'] + t.uploadedMemory)
					down = float(t.status["totalDone"])
					
					try:
						cr = float(up/down)
					except:
						cr = -1
					dc.debugmsg("Target Ratio: " + str(targetRatio))
					dc.debugmsg("UL/DL: " + str(up) + "/" + str(down))
					dc.debugmsg("Ratio: " + str(cr))
					if(isSeeding and cr >= targetRatio):
						dc.debugmsg("The target ratio was reached, now downqueuing torrent")
						self.parent.internalQueueTorrentAllDown(uniqueID)
						continue # we MUST continue; the previous command (queue to bottom
									# has reordered the iters, on which we are iterating...

#						t.userPause = True
#						t.pause()
						
				##
				##
				##

				curr_list = t.getList()
				queuePos = self.parent.torrentList.get_value(my_iter, 2)
				curr_list.insert(2, queuePos) # Add the queue position
				c = 0
				for val in curr_list:
					self.parent.torrentList.set_value(my_iter,c,curr_list[c])
					c = c + 1

				# Keep only the right torrents active; pause the rest
				if t.isPaused() and not t.isForcePaused() and (queuePos <= maxDownloads or maxDownloads is 0):
					t.start()

				if not t.isPaused() and (queuePos > maxDownloads and maxDownloads is not 0):
					t.pause()

				my_iter = self.parent.torrentList.iter_next(my_iter)

			# Update details of selected torrent
			if self.parent.torrentView is not None:
				(temp, selected_iter) = self.parent.torrentView.get_selection().get_selected()
				if selected_iter is not None:
					self.parent.updateTorrentDetails(selected_iter)

			self.handleEvents()

			self.parent.pluginManager.update()

			# Done after everything else, so we can show all relevant data
			self.parent.UpdateStatusBar()

			if not self.parent.finalInitDone:
				self.parent.finalInitialization()

			yield True

		yield False
		
	def update_all(self):
		my_iter = self.parent.torrentList.get_iter_first()
		while my_iter is not None:
			uniqueID = self.parent.torrentList.get_value(my_iter, 0)
			t = self.get_torrent_from_uniqueID(uniqueID)
			t.update()
			t.updateTracker() ### ??? What is this ???

			my_iter = self.parent.torrentList.iter_next(my_iter)

	def handleEvents(self):
		event = torrent.popEvent()

		while event is not None:
#			self.parent.addMessage(str(event))

			if event['eventType'] is torrent.constants()['EVENT_FINISHED']:
				# If we are autoseeding, then seed. Otherwise, push back in queue, we are done
				if(self.auto_seed_ratio == -1):
					uniqueID = event['uniqueID']
					self.parent.internalQueueTorrentAllDown(uniqueID)

			elif event['eventType'] is torrent.constants()['EVENT_PEER_ERROR']:
#				self.parent.addMessage(_("Peer Error") + ": " + str(event), "I") # Debug only!
				pass
			elif event['eventType'] is torrent.constants()['EVENT_INVALID_REQUEST']:
				self.parent.addMessage(_("Invalid request") + ": " + str(event), "W") # Maybe "I"?
			elif event['eventType'] is torrent.constants()['EVENT_FILE_ERROR']:
				dc.debugmsg("File error! " + str(event))
				self.parent.addMessage(_("File error") + "! " + str(event), "F")
			elif event['eventType'] is torrent.constants()['EVENT_HASH_FAILED_ERROR']:
				self.parent.addMessage(_("Hash failed") + ": " + str(event), "I")
			elif event['eventType'] is torrent.constants()['EVENT_PEER_BAN_ERROR']:
				self.parent.addMessage(_("Peer banned") + ": " + str(event), "I")
			elif event['eventType'] is torrent.constants()['EVENT_FASTRESUME_REJECTED_ERROR']:
				dc.debugmsg("Fastresume rejected: " + str(event))
				self.parent.addMessage(_("Fastresume rejected") + ": " + str(event), "W")
			elif event['eventType'] is torrent.constants()['EVENT_TRACKER']:
				index = self.get_index_from_uniqueID(event['uniqueID'])
				self.torrent_list[index].setTrackerMessage(event['trackerStatus'], event['message'])
			elif event['eventType'] is torrent.constants()['EVENT_OTHER']:
				self.parent.addMessage(_("Event") + ": " + str(event), "W")
			else:
				dc.debugmsg("Internal error, undefined event type")
				dc.debugmsg("No such event error. Raw data: " + str(event))
				self.parent.addMessage(_("Event") + ": " + str(event), "C")

			event = torrent.popEvent()

	def stop(self):
		self.bool_continue = False
		
	def setAutoSeedRatio(self, ratio):
		self.auto_seed_ratio = ratio

# Name Size Progress Seeders Peers D/L U/L ETA %%
	def add_torrent(self, queuePos, torrent_path, save_dir, compact_allocation):
		try:
			new_torrent = Torrent(torrent_path, save_dir, compact_allocation)
		except TorrentError, E:
			dc.ShowPopupWarning(self.parent.window, E.message)
			return None

		self.torrent_list.append(new_torrent)
		data = new_torrent.getList()
		data.insert(2, queuePos) # Add the queue position
		for x in data:
			dc.debugmsg(x)
		self.parent.torrentList.append(data)
		return new_torrent

	def get_index_from_uniqueID(self, uniqueID):
		for i in range(len(self.torrent_list)):
			if self.torrent_list[i].getUniqueID() == uniqueID:
				return i

		dc.debugmsg("INTERNAL ERROR: cannot find uniqueID "+ str(uniqueID))
		return -1

	def get_torrent_from_uniqueID(self, uniqueID):
		return self.torrent_list[self.get_index_from_uniqueID(uniqueID)]

	def remove_torrent(self, uniqueID):
		index = self.get_index_from_uniqueID(uniqueID)

		self.torrent_list.pop(index)
		torrent.removeTorrent(uniqueID)
	def clear_completed(self):
		dc.debugmsg("Clearing finished torrents")
		# Update all torrents for the upper summary list
		my_iter = self.parent.torrentList.get_iter_first()

		while (my_iter != None):
			uniqueID = self.parent.torrentList.get_value(my_iter, 0)
			t = self.get_torrent_from_uniqueID(uniqueID)
			t.update()
			progress = t.getProgressFloat()
			dc.debugmsg("Torrent Progress:" + str(progress))
			if (progress == 100):
				dc.debugmsg( "Chicken's Done!")
				index = self.get_index_from_uniqueID(uniqueID)
				self.torrent_list.pop(index)
				torrent.removeTorrent(uniqueID)
				self.parent.torrentList.remove(my_iter)
				my_iter = self.parent.torrentList.get_iter_first()
			else:
				my_iter = self.parent.torrentList.iter_next(my_iter)
			dc.debugmsg( my_iter)
		dc.debugmsg( "finished clearing completed")
			
	def getNumActiveTorrents(self):
		ret = 0

		for torrent in self.torrent_list:
			if torrent.inProgress():
				ret = ret + 1

		return ret
