/*
 * the Decibel Realtime Communication Framework
 * Copyright (C) 2006 by basyskom GmbH
 *  @author Tobias Hunger <info@basyskom.de>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifndef DECIBEL_DAEMON_PROTOCOLMANAGER_H
#define DECIBEL_DAEMON_PROTOCOLMANAGER_H

#include <QtCore/QObject>
#include <QtCore/QStringList>

#include <QtDBus/QDBusContext>

class ProtocolManagerPrivate;

namespace QtTapioca
{
    class ConnectionManager;
} // namespace

/**
 * @brief The ProtocolManager encapsulates the Connection Managers of Telepathy.
 *
 * The ProtocolManager encapsulates the Connection Managers of Telepathy,
 * storing a preferred Connection Manager for each protocol supported.
 *
 * Application developers are encouraged to use the preferred Connection
 * Manager instead of grabbing a random one.
 *
 * This class is accessible over DBus so that a control center module
 * can set up preferred connection managers.
 *
 * @author Tobias Hunger <info@basyskom.de>
 */
class ProtocolManager : public QObject, protected QDBusContext
{
    Q_OBJECT

public:
    /**
     * @brief Constructor
     */
    explicit ProtocolManager(QObject * parent = 0);
    /**
     * @brief Destructor
     */
    ~ProtocolManager();

    /**
     * @brief Get a Connection Manager by name.
     * @param cm_name Name of the connection manager to get.
     *
     * This method is *not* exported to DBus!
     */
    QtTapioca::ConnectionManager* connectionManagerByName(const QString & cm_name);

public Q_SLOTS:
    /**
     * @brief Get a list of installed Connection Managers.
     * @return A list of connection managers.
     *
     * Get a list containing the names of all connection managers installed
     * on this system.
     */
    QStringList allConnectionManagers() const;
    /**
     * @brief Get a list of supported protocols.
     * @return A list of names of supported protocols.
     *
     * Get a list of protocols supported by the installed connection managers.
     */
    QStringList supportedProtocols() const;
    /**
     * @brief Get a list of Connection Managers supporting a given protocol.
     * @param protocol The requested protocol
     * @return A list of connection manager names supporting the given protocol.
     *         This list is empty if no connection manager supports the
     *         protocol.
     *
     * Get a list containing the names of all connection managers speaking the
     * requested protocol.
     */
    QStringList connectionManagersForProtocol(const QString & protocol) const;

    /**
     * @brief Sets the preferred Connection Manager for a given protocol.
     * @param protocol The protocol to use the connction manager for
     * @param cm_name The connection manager to use
     * @return true on success and false otherwise
     *
     * Set the preferred connection manager for the given protocol. This
     * method returns true if the connection manager was successfully set
     * as the default connection manager for this protocol and false
     * otherwise.
     */
    bool setDefaultConnectionManagerForProtocol(const QString & protocol,
                                                const QString & cm_name);

    /**
     * @brief Get the preferred Connection Manager for a given protocol.
     * @param protocol The name of the protocol to request the connection
     *        manager for.
     * @return The name of the default connection manager for the given
     *         protocol. A empty string is returned if no connection manager
     *         supports the requested protocol.
     *
     * Get the connection manager to use for the given protocol. This is
     * the one the user set up earlier (using setDefaultConnectionManager). If
     * the user has not selected a default handler for a protocol than a
     * random connection manager is selected from the list of those that
     * supported the requested protocol.
     */
    QString defaultConnectionManagerForProtocol(const QString & protocol) const;

private:
    ProtocolManagerPrivate * const d;
};

#endif
