/*
 * the Decibel Realtime Communication Framework
 * Copyright (C) 2006 by basyskom GmbH
 *  @author Tobias Hunger <info@basyskom.de>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include <QtCore/QDebug>

#include <QtCore/QHash>
#include <QtCore/QSettings>
#include <QtCore/QDir>

#include <QtTapioca/Channel>
#include <QtTapioca/Handle>

#include "componentmanager.h"
#include "componentmanageradaptor.h"

#include "config.h"

namespace
{
    static const QString fallback_profile_name("fallback");
    static const QString component_search_path("~/.local/share/Decibel/components;"
                                               "/usr/local/share/Decibel/components;"
                                               "/usr/share/Decibel/components;"
                                               COMPONENT_SEARCH_DIR);

    static const QString component_group("Component");
    static const QString componentmanager_group("ComponentManager");

    static const QString search_path_key("Search Path");

    static const QString protocols_list_key("Protocols");
    static const QString types_list_key("Types");
    static const QString target_types_list_key("Targets");
    static const QString service_name_key("Service Name");
    static const QString object_path_key("Object Path");

    static const QString profiles_array("Profiles");
    static const QString active_profile("Active Profile");
    static const QString profile_name_key("Name");
    static const QString profiles_components_array("Components");
    static const QString profiles_component_handle_key("Component Handle");
}

/// @cond INCLUDE_PRIVATE

/**
 * @brief Private class implementing the ComponentManager.
 *
 * A private class implementing the ComponentManager.
 *
 * @author Tobias Hunger <info@basyskom.de>
 */
class ComponentManagerPrivate
{
public:
    /** @brief Constructor */
    ComponentManagerPrivate(ComponentManager * mgr) :
        adaptor(new ComponentManagerAdaptor(mgr))
    {
        Q_ASSERT(adaptor != 0);

        QSettings settings;
        settings.beginGroup(componentmanager_group);

        // search path:
        searchPath = settings.value(search_path_key, QString()).toString().split(";", QString::SkipEmptyParts);
        if (searchPath.isEmpty())
        { searchPath = component_search_path.split(";", QString::SkipEmptyParts); }

        // Read Profiles:
        int num_profiles = settings.beginReadArray(profiles_array);
        for (int i = 0; i < num_profiles; ++i)
        {
            settings.setArrayIndex(i);

            QString profile_name = settings.value(profile_name_key, QString("")).toString();
            if (profile_name.isEmpty())
            {
                qWarning() << "Profile without a name found. Skipping!";
                continue;
            }
            if (profiles.contains(profile_name))
            {
                qWarning() << "Profile of name" << profile_name << "already known. "
                           << "Skipping!";
                continue;
            }
            qDebug() << "Adding profile" << profile_name;

            QList<Decibel::Component> components;
            int num_components = settings.beginReadArray(profiles_components_array);
            for (int j = 0; j < num_components; ++j)
            {
                settings.setArrayIndex(j);

                Decibel::Component component;
                QString key;
                QStringList keys = settings.allKeys();
                foreach (key, keys)
                {
                    QVariant value = settings.value(key);
                    if (protocols_list_key == key)
                    { component.protocol_list = value.toString().split(";", QString::SkipEmptyParts); }
                    else if (types_list_key == key)
                    {
                        QStringList tmp(value.toString().split(";", QString::SkipEmptyParts ));
                        QString current;
                        foreach (current, tmp)
                        {
                            int type = current.toInt();
                            if (type >= QtTapioca::Channel::Text &&
                                type <= QtTapioca::Channel::Stream)
                            { component.type_list << type; }
                        }
                    }
                    else if (target_types_list_key == key)
                    {
                        QStringList tmp(value.toString().split(";", QString::SkipEmptyParts));
                        QString current;
                        foreach (current, tmp)
                        {
                            int type = current.toInt();
                            if (type > QtTapioca::Handle::None &&
                                type <= QtTapioca::Handle::List)
                            { component.target_type_list << type; }
                        }
                    }
                    else if (profiles_component_handle_key == key)
                    {
                        component.component_handle = value.toInt();
                        if (0 == component.component_handle)
                        { continue; }
                    }
                    else
                    {
                        qWarning() << "Unknown key" << key
                                   << "found while reading components of profile"
                                   << profile_name << ".";
                    }
                }
                components.append(component);
            }
            settings.endArray();

            profiles.insert(profile_name, components);
        }
        settings.endArray();

        activeProfile = settings.value(active_profile, QString()).toString();
        if (activeProfile.isEmpty() ||
            !profiles.contains(activeProfile))
        { activeProfile = fallback_profile_name; }
        qDebug() << "Marking profile" << activeProfile << "as active.";

        settings.endGroup();

        // Scan for components:
        scanComponents();
    }

    /** @brief Destructor */
    ~ComponentManagerPrivate()
    {
        QSettings settings;

        settings.beginGroup(componentmanager_group);

        settings.setValue(search_path_key, searchPath.join(";"));

        // Write profiles:
        settings.beginWriteArray(profiles_array, profiles.size());
        int profile_index = 0;
        QString profile_name;
        foreach(profile_name, profiles.keys())
        {
            settings.setArrayIndex(profile_index);
            ++profile_index;

            settings.setValue(profile_name_key, profile_name);

            settings.beginWriteArray(profiles_components_array,
                                     profiles[profile_name].size());
            for(int i = 0; i < profiles[profile_name].size(); ++i)
            {
                settings.setArrayIndex(i);

                // protocols:
                settings.setValue(protocols_list_key,
                                 profiles[profile_name][i].protocol_list.join(QString(';')));

                // types:
                QString tmp;
                int type;
                foreach(type, profiles[profile_name][i].type_list)
                {
                    if (!tmp.isEmpty()) { tmp += ';'; }
                    tmp += QString::number(type);
                }
                settings.setValue(types_list_key, tmp);

                // targets:
                tmp = QString();
                foreach(type, profiles[profile_name][i].target_type_list)
                {
                    if (!tmp.isEmpty()) { tmp += ';'; }
                    tmp += QString::number(type);
                }
                settings.setValue(target_types_list_key, tmp);

                // component handle:
                settings.setValue(profiles_component_handle_key,
                                  profiles[profile_name][i].component_handle);
            }
            settings.endArray();
        }
        settings.endArray();

        settings.setValue(active_profile, activeProfile);

        settings.endGroup();
    }

    /** @brief Remove obsolete components from profiles. */
    void fixProfiles()
    {
        // remove unknown components:
        QString profile_name;
        foreach (profile_name, profiles.keys())
        {
            int i = 0;
            while (i < profiles[profile_name].size())
            {
                uint handle = profiles[profile_name][i].component_handle;
                if (!components.contains(handle))
                {
                    qWarning() << "Removing component handle"
                               << profiles[profile_name][i].component_handle
                               << "from Profile" << profile_name
                               << "at position" << i << ".";
                    profiles[profile_name].removeAt(i);
                }
                else
                { ++i; }
            }
        }

        updateFallbackProfile();
    }

    /** @brief Add new components to the fallback profile. */
    void updateFallbackProfile()
    {
        QList<Decibel::Component> profile = profiles[fallback_profile_name];
        uint component_handle;
        foreach (component_handle, components.keys())
        {
            int i;
            for(i = 0; i < profile.size(); ++i)
            {
                if (profile[i].component_handle == component_handle)
                { break; }
            }

            if (i == profile.size())
            { profile.append(Decibel::Component(component_handle)); }
            else
            { profile[i] = Decibel::Component(component_handle); }
        }
        profiles.insert(fallback_profile_name, profile);
    }

    /** @brief Scan the search path for .component files and read them. */
    void scanComponents()
    {
        QHash<uint, Decibel::ComponentInfo> local_components;

        QString current_dir;
        foreach (current_dir, searchPath)
        {
            qDebug() << "Scanning directory" << current_dir;
            if (current_dir.startsWith('~'))
            { current_dir = qgetenv("HOME") + current_dir.mid(1); }

            QDir dir(current_dir);
            if (!dir.exists())
            {
                qWarning() << "Directory" << current_dir << "does not exist.";
                continue;
            }

            dir.setFilter(QDir::Files);
            dir.setNameFilters(QStringList("*.component"));
            dir.setSorting(QDir::Name);

            QStringList files = dir.entryList();
            QString file;
            foreach (file, files)
            {
                QString fileprofile_name(dir.absoluteFilePath(file));

                // read component info file:
                Decibel::ComponentInfo info;
                QSettings component_settings(fileprofile_name, QSettings::IniFormat);
                component_settings.beginGroup(component_group);
                QString key;
                foreach (key, component_settings.allKeys())
                {
                    QVariant value(component_settings.value(key));

                    if (protocols_list_key == key)
                    { info.possible_protocol_list = value.toString().split(";", QString::SkipEmptyParts); }
                    else if (types_list_key == key)
                    {
                        QStringList tmp(value.toString().split(";", QString::SkipEmptyParts));
                        QString current;
                        foreach (current, tmp)
                        {
                            int type = current.toInt();
                            if (type >= QtTapioca::Channel::Text &&
                                type <= QtTapioca::Channel::Stream)
                            { info.possible_type_list << type; }
                        }
                    }
                    else if (target_types_list_key == key)
                    {
                        QStringList tmp(value.toString().split(";", QString::SkipEmptyParts));
                        QString current;
                        foreach (current, tmp)
                        {
                            int type = current.toInt();
                            if (type > QtTapioca::Handle::None &&
                                type <= QtTapioca::Handle::List)
                            { info.possible_target_type_list << type; }
                        }
                    }
                    else if (service_name_key == key)
                    { info.service_name = value.toString(); }
                    else if (object_path_key == key)
                    { info.object_path.setPath(value.toString()); }
                    else
                    {
                        qWarning() << "Unknown key" << key
                                << "found while reading component from"
                                << fileprofile_name;
                    }
                }
                component_settings.endGroup();

                // Do we have all the data we need?
                if (info.service_name.isEmpty() ||
                    info.object_path.path().isEmpty())
                {
                    qWarning() << fileprofile_name
                               << "is incomplete, skipping.";
                    continue;
                }
                // Is this component info already known?
                bool already_known(false);
                uint tmp_handle;
                foreach (tmp_handle, components.keys())
                {
                    if (components[tmp_handle].sameObject(info))
                    {
                        qWarning() << "        Skipping" << fileprofile_name
                                   << ": I already know about this service.";
                        already_known = true;
                        break;
                    }
                }
                if (already_known) { continue; }

                // store:
                uint handle = hashComponentInfo(qHash(fileprofile_name), info);
                if (local_components.contains(handle))
                {
                    qWarning() << "        Skipping" << fileprofile_name
                               << ": Component handle" << handle
                               << "already in use.";
                    continue;
                }
                else if (0 == handle)
                {
                    qWarning() << "        Skipping" << fileprofile_name
                               << ": Handle is 0.";
                    continue;
                }

                while (local_components.contains(handle) &&
                       !local_components[handle].sameObject(info))
                {
                    qWarning() << "Component handle" << handle
                               << "already in use.";
                    handle = handle + 1;
                }

                qDebug() << "Component:";
                qDebug() << "    protocols   :" << info.possible_protocol_list;
                qDebug() << "    types       :" << info.possible_type_list;
                qDebug() << "    target_types:" << info.possible_target_type_list;
                qDebug() << "    service name:" << info.service_name;
                qDebug() << "    object path :" << info.object_path.path();

                local_components.insert(handle, info);
           }
        }

        // store data:
        components = local_components;

        fixProfiles();
    }

    /**
     * @brief Calculate a handle by hashing the ComponentInfo.
     * @param base Initial value of the hash.
     * @param info The ComponentInfo to hash.
     * @return The hash value.
     */
    uint hashComponentInfo(const uint base,
                           const Decibel::ComponentInfo & info)
    {
        // FIXME: Do I need to use some more evenly distributed algorithm here?
        uint hash = base;
        hash += qHash(info.service_name);
        hash += qHash(info.object_path.path());

        return hash;
    }

    /** @brief The D-Bus adaptor for the ComponetManager. */
    ComponentManagerAdaptor * const adaptor;

    /** @brief A list of components addressable by component handle. */
    QHash<uint, Decibel::ComponentInfo> components;
    /** @brief A list of profiles, addressable by profile name. */
    QHash<QString, QList<Decibel::Component> > profiles;

    /** @brief The active profile. */
    QString activeProfile;

    /** @brief The path used to search for .component files. */
    QStringList searchPath;
};

/// @endcond

// ****************************************************************************

ComponentManager::ComponentManager(QObject * parent) :
    QObject(parent),
    d(new ComponentManagerPrivate(this))
{ Q_ASSERT(d != 0); }

ComponentManager::~ComponentManager()
{ delete d; }

void ComponentManager::setComponentSearchPath(const QStringList & search_path)
{
    d->searchPath = search_path;
    scanComponents();
}

QStringList ComponentManager::componentSearchPath() const
{ return d->searchPath; }

void ComponentManager::scanComponents()
{ d->scanComponents(); }

QList<uint> ComponentManager::listComponents() const
{ return d->components.keys(); }

Decibel::ComponentInfo ComponentManager::component(const uint handle) const
{
    if (d->components.contains(handle))
    { return d->components[handle]; }
    else
    { return Decibel::ComponentInfo(); }
}

void ComponentManager::setProfile(const QString & profile_name,
                                  const QList<Decibel::Component> & components)
{ d->profiles.insert(profile_name, components); }

QStringList ComponentManager::listProfiles() const
{ return d->profiles.keys(); }

QList<Decibel::Component> ComponentManager::profile(const QString & profile_name) const
{
    if (d->profiles.contains(profile_name))
    { return d->profiles[profile_name]; }
    else
    { return QList<Decibel::Component>(); }
}

bool ComponentManager::deleteProfile(const QString & profile_name)
{
    int size = d->profiles.remove(profile_name);
    Q_ASSERT(0 == size || 1 == size);
    return (1 == size);
}

QString ComponentManager::activeProfile() const
{ return d->activeProfile; }

bool ComponentManager::setActiveProfile(const QString & profile_name)
{
    if (d->profiles.contains(profile_name))
    { d->activeProfile = profile_name; }
    return (activeProfile() == profile_name);
}
