/*
 * the Decibel Realtime Communication Framework
 * Copyright (C) 2006 by basyskom GmbH
 *  @author Tobias Hunger <info@basyskom.de>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include "simplisticcontactconnector.h"

#include <QtCore/QSettings>
#include <QtCore/QStringList>
#include <QtCore/QDebug>

namespace
{
    static const QString uri_key("URI");
    static const QString protocol_key("Protocol");

    static const QString filename("~/.decibel_contact_data.ini");
    static const QString array_name("Contacts");
} // namespace

SimplisticContactConnector::SimplisticContactConnector()
{
    QSettings settings(filename, QSettings::IniFormat);
    int size = settings.beginReadArray(array_name);
    for (int i = 0; i < size; ++i) {
        settings.setArrayIndex(i);
        QVariantMap current_contact;

        QStringList keys = settings.childKeys();
        QString current_key;
        foreach(current_key, keys)
        { current_contact[current_key] = settings.value(current_key); }

        Q_ASSERT(current_contact.contains(uri_key));
        Q_ASSERT(current_contact.contains(protocol_key));

        m_contacts.append(current_contact);
    }

    settings.endArray();
}

SimplisticContactConnector::~SimplisticContactConnector()
{
    QSettings settings(filename, QSettings::IniFormat);
    settings.beginWriteArray(array_name);
    int array_index = 0;
    QVariantMap contact;
    foreach (contact, m_contacts)
    {
        settings.setArrayIndex(array_index);
        ++array_index;
        QString key;
        foreach (key, contact.keys())
        { settings.setValue(key, contact[key]); }
    }
    settings.endArray();
}

bool SimplisticContactConnector::gotContact(const uint & contact_id) const
{
    if (m_contacts.size() < int(contact_id)) { return false; }
    if (0 == contact_id) { return false; }
    return true;
}

QVariantMap SimplisticContactConnector::contact(const uint & contact_id) const
{
    if (m_contacts.size() < int(contact_id)) { return QVariantMap(); }
    if (0 == contact_id) { return QVariantMap(); }
    return m_contacts[contact_id - 1];
}

unsigned int SimplisticContactConnector::findURI(const QString & protocol,
                                                 const QString & uri) const
{
    for (int i = 0; i < m_contacts.size(); ++i)
    {
        Q_ASSERT(m_contacts[i].contains(uri_key));
        Q_ASSERT(m_contacts[i].contains(protocol_key));

        if (m_contacts[i][uri_key] == uri &&
            m_contacts[i][protocol_key] == protocol) { return (i + 1); }
    }
    return 0;
}

unsigned int SimplisticContactConnector::addContact(const QVariantMap & new_contact)
{
    m_contacts.append(new_contact);
    return m_contacts.size();
}

void SimplisticContactConnector::setPresence(const uint & contact_id,
                                             const QtTapioca::ContactBase::Presence presence_state,
                                             const QString & message)
{
    if (int(contact_id) > m_contacts.size() ||
        0 == contact_id) { return; }

    qDebug() << m_contacts[contact_id - 1]["Name"] << "has a new presence:"
             << presence_state << message;
}
