/*
 * the Decibel Realtime Communication Framework
 * Copyright (C) 2006 by basyskom GmbH
 *  @author Tobias Hunger <info@basyskom.de>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifndef _DECIBEL_DAEMON_COMPONENTMANAGER_H_
#define _DECIBEL_DAEMON_COMPONENTMANAGER_H_

#include <QtCore/QObject>

#include <Decibel/Types>

class ComponentManagerPrivate;

/**
 * @brief The ComponentManager persistently stores components and their setup.
 *
 * The ComponentManager is used to set up and manage components: Applications
 * started over D-Bus in reaction to events. Componets are the means to
 * interact with the user or desktop dependent functionality that is not
 * available to Decibel (which is desktop independent).
 *
 * Components are defined in .component files. These are stored on in the
 * filesystem. On startup the ComponentManager scans a list of directories
 * for these files (the component search path), whenever the search path
 * is changed or the scanComponents() method is called.
 *
 * Profiles are created from a list of installed components and define
 * which components are started in response to channels being created. A user
 * may register an arbitrary number of profiles and switch between them as
 * necessary.
 *
 * @author Tobias Hunger <info@basyskom.de>
 */
class ComponentManager : public QObject
{
    Q_OBJECT
    Q_DECLARE_PRIVATE(ComponentManager)

public:
    /** @brief Constructor */
    explicit ComponentManager(QObject * parent = 0);
    /** @brief Destructor */
    ~ComponentManager();

public slots:
    /**
     * @brief Set the path that is used to search for .component files.
     * @param search_path A list of directories to search.
     *
     * This method is used to set up a path to search for .compenent files.
     * These files contain ComponentInfo data in ini file format and describe
     * one installed component each.
     */
    void setComponentSearchPath(const QStringList & search_path);
    /**
     * @brief Get the path that is used to search for .component files.
     * @return A list of directories searched.
     */
    QStringList componentSearchPath() const;
    /**
     * @brief Scan the search path for .component files.
     *
     * scanComponents examines the currently set up component search path and
     * examines each .component file in any of these directories. It then
     * proceeds to scan these files and updates the component database with the
     * data found. It then removes unknown entries from the profiles.
     */
    void scanComponents();
    /**
     * @brief List all known component handles.
     * @return A list of known component handles.
     */
    QList<uint> listComponents() const;
    /**
     * @brief Get the component data stored under a handle.
     * @param component_handle The component handle to check.
     * @return The ComponentInfo stored under the given handle. This is empty
     * (service_name.isEmpty() returns bool) if the given handle is undefined.
     */
    Decibel::ComponentInfo component(const uint component_handle) const;

    /**
     * @brief Set up a list of components and store it as a profile under
     * the given name.
     * @param profile_name Name to store the profile under.
     * @param components A list of components.
     */
    void setProfile(const QString & profile_name,
                    const QList<Decibel::Component> & components);
    /**
     * @brief Get the list of defined profile names.
     * @return A list of profile names.
     */
    QStringList listProfiles() const;
    /**
     * @brief Get the list of Components configured under the given profile
     * name.
     * @param profile_name The name of the profile to examine.
     * @return A list of components registered to be used in the given profile.
     * This list is empty if the profile name is unknown.
     */
    QList<Decibel::Component> profile(const QString & profile_name) const;
    /**
     * @brief Delete the profile with the given name.
     * @param profile_name The name of the profile to delete.
     * @return true if the profile was successfully deleted and false otherwise.
     */
    bool deleteProfile(const QString & profile_name);

    /**
     * @brief Get the name of the active profile.
     * @return The name of the active profile.
     *
     * The active profile is the one that is currently used to look up
     * components in whenever a new channel is created. This method returns the
     * name of this profile.
     */
    QString activeProfile() const;
    /**
     * @brief Set the active profile.
     * @param profile_name The name of the profile to activate.
     * @return true if the profile was successfully registered as active and
     * false otherwise
     *
     * Using this method a developer can select the profile to use to
     * lookup components.
     */
    bool setActiveProfile(const QString & profile_name);

private:
    /** @brief Private data */
    ComponentManagerPrivate * const d;
};

#endif
