package com.tildemh.debbuggtk;

import com.tildemh.debbug.*;
import org.gnu.gtk.*;
import org.gnu.gnome.*;
import org.gnu.gtk.event.*;
import org.gnu.glade.*;
import java.util.*;
import java.io.*;
import org.gnu.glib.CustomEvents;

/**
 * Form for downloading listings and reports. This will generally be done in the
 * background, with a small icon in the statusbar. However, if the user wants a
 * progress bar, then a form may be displayed showing this. 
 *
 * <p>This is released under the terms of the GNU Lesser General Public License
 * (LGPL). See the COPYING file for details.
 *
 * @version $Id: Downloader.java,v 1.17 2004/04/21 20:20:05 mh Exp $
 * @author &copy; Mark Howard &lt;mh@debian.org&gt; 2002
 */
public class Downloader implements BugListener, ListingListener{
	

	private static Downloader instance = null;

	public static Downloader getInstance(){
		if (instance == null) instance = new Downloader();
		return instance;
	}
	
	private Label nowUpdating;
	private ProgressBar watchProgress;
	private AppBar appBar;
	private Window app;
	private Image imgBug;
	private Image imgListing;
	private Image imgWatched;
	private Label lblBug;
	private Label lblListing;
	private Label lblWatched;

	private HBox bugBox;
	private HBox listingBox;
	private HBox watchedBox;
	
	private Bug wantedBug = null;
	private Listing wantedListing = null;
	private LinkedList wantedListings = new LinkedList();

	private Thread downloaderThread = null;
	private Runnable downloaderObject = null;
	private LibGlade glade;

	// 
	// For the following:
	//  0 = not done yet
	//  1 = done - success
	//  2 = done - failure
	/** True if finished processing the current bug report */
	private volatile int doneBug = 0;
	/** True if finished processing the current listing */
	private volatile int doneListing = 0;
	/** Number of watched items which have been completed */
	private volatile int completedWatched;
	/** Total number of watched items to process */
	private volatile int totalWatched;
	/** Current watched item being looked at */
	private volatile Object currentWatched;

	/**
	 * Constructs a downloader object
	 */
	private Downloader(){
		DebbugGtk.newWindow();

		WindowIcon.setDefaultIcon(DebbugGtk.ICON);
		
		String filename = DebbugGtk.GLADE_XML_DIR+"downloader.glade";
		try {
			glade = new LibGlade(filename, this, null);
			app = (Window) glade.getWidget("downloader");
			app.addListener( DebbugGtk.lifeCycleListener );
			
			appBar = (AppBar) glade.getWidget("appBar");
			nowUpdating = (Label) glade.getWidget("nowUpdating");
			watchProgress = (ProgressBar) glade.getWidget("watchProgress");
			imgBug = (Image) glade.getWidget("imgBug");
			imgListing = (Image) glade.getWidget("imgListing");
			imgWatched = (Image) glade.getWidget("imgWatched");
			lblBug = (Label) glade.getWidget("lblBug");
			lblListing = (Label) glade.getWidget("lblListing");
			lblWatched = (Label) glade.getWidget("lblWatched");
			bugBox = (HBox) glade.getWidget("bugBox");
			listingBox = (HBox) glade.getWidget("listingBox");
			watchedBox = (HBox) glade.getWidget("watchedBox");
			
		} catch (GladeXMLException e) {
			System.err.println("Error parsing glade XML file." + e);
		} catch (FileNotFoundException e) {
			System.err.println("Glade XML file not found.");
		} catch (IOException e) {
			System.err.println("Error reading glade XML file.");
		}

	}


	/**
	 * The cancel button is hit...
	 */
	public void cancel(){
		interrupt();
	}
	
	// interrupts the update
	private synchronized void interrupt(){
		while ( downloaderThread != null && downloaderThread.isAlive()){
			downloaderThread.interrupt();
			try{
				wait(100);
			}catch(InterruptedException e){
				// do nothing
			}
		}
		app.hide();
	}

	/**
	 * Does the actual updating. This is done in the downloaderThread.
	 */
	private void doUpdate(){
		
		// remove current list/bug from watched items
		for(int i = 0; i < wantedListings.size(); i++){
			Object current = wantedListings.get(i);
			if (current.equals(wantedBug)
					|| current.equals(wantedListing) )
				wantedListings.remove(i);
		}
		totalWatched = wantedListings.size();
		doneBug = 0;
		doneListing = 0;
		completedWatched = 0;
		
		updateDisplay();
		if (Thread.interrupted()) exception("Somebody interrupted me");
		
		/// Download the bug report
System.out.println("----------- BUG -----------------");
		if (wantedBug != null){
		try{
			currentWatched = wantedBug;
			wantedBug.addListener( (BugListener) this );
			wantedBug.update();
		// FIXME: have better responses to these exceptions!
		}catch(java.net.MalformedURLException e){
			// this will not happen
			e.printStackTrace();
			doneBug++;
			throw new RuntimeException( e.toString() );
		}catch(java.io.IOException e){
			e.printStackTrace();
			exception( "IOError while updating bug: "+e.toString());
			doneBug++;
		}catch(com.tildemh.debbug.BugNotFound e){
			exception( "Bug "+wantedBug.toString()+" not found on server" );
			e.printStackTrace();
			doneBug++;
		}
			wantedBug.removeListener( (BugListener) this );
	}
		doneBug++;
		updateDisplay();
		if (Thread.interrupted()) exception("Synchronization interrupted");
		
		/// Download the listing
System.out.println("----------- LISTING -----------------");
		if (wantedListing != null){
			currentWatched = wantedListing;
			wantedListing.addListener( (ListingListener) this );
			wantedListing.update();
			wantedListing.removeListener( (ListingListener) this );
		}
		doneListing++;
		updateDisplay();
		if (Thread.interrupted()) exception("Synchronization interrupted");
		

		/// Download the other watched items
		
		if (Thread.interrupted()) exception("Synchronization interrupted");

System.out.println("----------- WATCHED -----------------");
		while (wantedListings.size() > 0){
			if (Thread.interrupted()) exception("Synchronization interrupted");
			
			ListingStub stub = (ListingStub) wantedListings.removeFirst();
			if (stub.getType().equals( ListingType.BUG ) ){
				Bug b = BTS.getInstance().getBug( new Integer( stub.getName() ) );
				currentWatched = b;
				updateDisplay();
				b.addListener(this);
				try{
					b.update();
				}catch(Exception e){
					// TODO: if server error, don't keep going!
					exception("Error updating bug #"+stub.getName()+" - "+e.toString());
				}
				b.removeListener(this);
			}else{
				// download a listing and all associated bug reports
				Listing l = BTS.getInstance().getListing( stub.getType(), stub.getName() );
				currentWatched = l;
				updateDisplay();
				l.addListener(this);
				if (Thread.interrupted()) exception("Synchronization interrupted");
				l.update();
				l.removeListener(this);
			}
			completedWatched++;
			updateDisplay();
		}
		CustomEvents.addEvent( new Runnable(){
			public void run(){
				app.hide();
			}
		});
	}

	/**
	 * dl thread calls this to update the gui
	 */
	private void updateDisplay(){
		CustomEvents.addEvent( new Runnable(){
			public void run(){
				updateDisplayGUIThread();
			}
		});
	}

	private volatile boolean pulseProgress = false;
	private volatile double progressFraction = 0;
	private volatile String statusMessage = "";
	private void updateProgress(){
		CustomEvents.addEvent( new Runnable(){
			public void run(){
				if (pulseProgress){
					appBar.getProgressBar().pulse();
				}else{
					appBar.setProgressPercentage( progressFraction );
				}
				appBar.clearStack();
				appBar.pushStack( statusMessage );
			}
		});
	}

	private void updateDisplayGUIThread(){
		imgBug.hide();		// temporary until we can change the images
		imgListing.hide();
		imgWatched.hide();
		if (wantedBug == null){
			bugBox.hide();
		}else{
			bugBox.show();
			if (doneBug != 0){
//				imgBug.set();
				lblBug.setMarkup("Current bug: "+wantedBug.toString());
			}else{
				// must be doing it now
//				imgBug.hide();
				lblBug.setMarkup("<span style=\"italic\">Current bug: "+wantedBug.toString()+"</span>");
				nowUpdating.setText( "Current report: "+wantedBug.toString() );
			}
		}
		if (wantedListing == null){
			listingBox.hide();
		}else{
			listingBox.show();
			if (doneListing==1){
//				imgListing.show();
			}else{
//				imgListing.hide();
			}
			if (doneListing!=0 || doneBug==0){
				lblListing.setMarkup("Current listing: "+wantedListing.toString());
			}else{
				// doing listing now
				lblListing.setMarkup("<span style=\"italic\">Current listing: "+wantedListing.toString()+"</span>");
				nowUpdating.setText("Current listing: "+wantedListing.toString());
			}
		}
		if (totalWatched <=0 ){
			watchedBox.hide();	
		}else{
			watchedBox.show();
			if(doneListing!=0 && doneBug!=0){
				lblWatched.setMarkup("<span style=\"italic\">Watched items: </span>");
				if (currentWatched != null)
					nowUpdating.setText( "Watched item #"+(completedWatched+1)+": "+currentWatched.toString() );
				if (completedWatched != 0){
					System.out.println("CompletedWatched != 0");
					watchProgress.setFraction( (double) completedWatched / (double) totalWatched );
				}else{
					System.out.println("CompletedWatched == 0");
					watchProgress.setFraction( 0 );
				}
			}else{
				lblWatched.setMarkup("Watched items: ");
				watchProgress.setFraction( 0 );
			}
		}
		watchProgress.setText( completedWatched + "/" + totalWatched );
		imgWatched.hide();
	}
	
	public synchronized void updateAll( Bug bug, Listing list, WatchedList watchedList ){
		if ( downloaderThread != null && downloaderThread.isAlive()){
			exception("Already updating all items!");
		}
		pulseProgress = false;
		progressFraction = 0;
		statusMessage = "";
		appBar.clearStack();
	 	wantedListings = (LinkedList) watchedList.getItems().clone();
		wantedBug = bug;
		wantedListing = list;
		downloaderObject = new Runnable(){
			public void run(){
				doUpdate();
			}
		};
		downloaderThread = new Thread( downloaderObject );
		downloaderThread.setName("Listing Downloader for synchroniser");
		downloaderThread.setDaemon(true);
		downloaderThread.start();
		updateDisplay();
		app.showAll();
	}

	private void exception( String message ){
		final String msg = message;
		CustomEvents.addEvent( new Runnable(){
			public void run(){
				showError( msg );
			}
		});
		throw new RuntimeException("Error downloading - aborting.");
	}
	public void showError(String txt){
		MessageDialog md = 
			new MessageDialog(app,
						DialogFlags.MODAL.or(DialogFlags.DESTROY_WITH_PARENT), 
						MessageType.ERROR, 
						ButtonsType.CLOSE, 
						txt, true);
		md.run();
		md.destroy();
		app.hide();
	}


////////////////////////////////////////////////////////////////////////////////
	/////// BugListener Methods
	public void bugStatusChanged( Bug bug, Status oldStatus ){ 
		//ignore 
		if (Thread.interrupted()) exception("Synchronization interrupted");
	}
	public void bugSeverityChanged( Bug bug, Severity oldSeverity ){ 
		//ignore 
		if (Thread.interrupted()) exception("Synchronization interrupted");
	}
	public void bugUpdated( Bug bug ){
		//ignore
		if (Thread.interrupted()) exception("Synchronization interrupted");
	}
	public boolean bugException( Bug bug, Exception e ){
		if (Thread.interrupted()) exception("Synchronization interrupted");
		exception("Error loading bug #"+bug.getNumber()+" : "+e.toString());
		e.printStackTrace();
		return true;
	}
	public void retrievingBug( Bug bug ){
		pulseProgress = true;
		//progressFraction = 1.0;
		statusMessage = "Retrieving bug report";
		updateProgress();
	}
	public void bugDownloaded( Bug bug ){
		pulseProgress = false;
		progressFraction = 1.0;
		statusMessage = "Bug downloaded; parsing report";
		updateProgress();
	}
	
////////////////////////////////////////////////////////////////////////////////
	/////// ListingListener Methods
	public void bugCountsChanged( Listing listing ){
		if (Thread.interrupted()) exception("Synchronization interrupted");
		//ignore
	}
	public void listingUpdateStart( Listing listing ){
		if (Thread.interrupted()) exception("Synchronization interrupted");
		statusMessage = "Starting to update listing";
		pulseProgress = false;
		progressFraction = 0.0;
		updateProgress();
	}
	public void listingUpdateDone( Listing listing ){
		if (Thread.interrupted()) exception("Synchronization interrupted");
		statusMessage = "Completed updating listing";
		pulseProgress = false;
		progressFraction = 1.0;
		updateProgress();
	}
	public void downloadingListing( Listing listing ){
		if (Thread.interrupted()) exception("Synchronization interrupted");
		pulseProgress = true;
		progressFraction = 1.0;
		statusMessage = "Downloading listing";
		updateProgress();
	}
	public void interpretingListing( Listing listing ){
		if (Thread.interrupted()) exception("Synchronization interrupted");
		pulseProgress = true;
		progressFraction = 1.0;
		statusMessage = "Interpreting listing";
		updateProgress();
	}
	public void downloadingListingBug( Listing listing, Integer bugNumber, int pending, int total){
		if (Thread.interrupted()) exception("Synchronization interrupted");
		pulseProgress = false;
		progressFraction = (double) pending / (double) total ;
		statusMessage = "Fetching new & modified reports - "+ pending + "/" + total + " - #" + bugNumber.toString();
		updateProgress();
	}
	public void bugRemoved( Listing listing, Integer bugNumber ){
		if (Thread.interrupted()) exception("Synchronization interrupted");
		statusMessage = "Bug #"+bugNumber.toString() +" removed from listing";
		updateProgress();
	}
	public void bugAdded( Listing listing, Bug bug ){
		if (Thread.interrupted()) exception("Synchronization interrupted");
		statusMessage = "New bug in listing - #" +bug.getNumber();
		updateProgress();
	}
	public void bugChanged( Listing listing, Bug bug ){
		if (Thread.interrupted()) exception("Synchronization interrupted");
		statusMessage = "Bug #"+bug.getNumber()+" modified - downloading";
		updateProgress();
	}
	public boolean  listingException( Listing listing, Exception e ){
		if (Thread.interrupted()) exception("Synchronization interrupted");
		statusMessage = "Error: " + e.toString();
		updateProgress();
		exception("Error updating listing "+listing.toString()+" - "+e.toString());
		return false;
	}
	public void loadReportsDone( Listing listing ){ 
		// do nothing
	}



}
