package com.tildemh.debbuggtk;

import com.tildemh.debbug.*;
import org.gnu.gtk.*;
import org.gnu.gtk.event.*;
import org.gnu.gnome.*;
import java.util.LinkedList;
import com.tildemh.debbuggtk.CommentViewer;
import java.util.Date;
import java.text.DateFormat;
import org.gnu.glib.CustomEvents;
import org.gnu.glib.Timer;
import org.gnu.glib.Fireable;

/**
 * Container for displaying bug reports.
 *
 * <p>This is released under the terms of the GNU Lesser General Public License
 * (LGPL). See the COPYING file for details.
 *
 * @version $Id: BugReport.java,v 1.63 2004/01/21 20:03:49 mh Exp $
 * @author &copy; Mark Howard &lt;mh@debian.org&gt; 2002
 */
public class BugReport extends VBox implements BugListener, Fireable{

	private CommentViewer cv;
	private Label pkg;
	private Label numbr;
	private Label title;
	private Label severity;
	private Label age;
	private Label sync;
	private HBox tags;	
	
	private Bug bug = null;
	private BTS bts = BTS.getInstance();;
	
	private App parent;
	private AppBar appBar;

	private Timer timer;

	/**
	 */
	public BugReport(Bug bug, App parent, AppBar appBar){
		super(false, 0);
		this.parent = parent;
		this.appBar = appBar;
		init();
		setBug(bug);
	}
	public BugReport(App parent, AppBar appBar){
		super(false, 0);
		this.parent = parent;
		this.appBar = appBar;
		init();
	}

	private void init(){

		// Line 1
		HBox hb = new HBox(false, 0);
		packStart(hb, false, false, 0);

		numbr = new Label( "" );
		numbr.setSelectable(true);
		numbr.setMinimumSize(70, 0);
		hb.packStart(numbr, false, false, 0);
		
		pkg = new Label( "" );
		hb.packStart(pkg, false, false, 0);
		
		title = new Label( "" );
		title.setSelectable(true);
		hb.packStart(title, false, true, 0);

		// Line 2
		tags = new HBox(false, 0);
		packStart(tags, false, true, 0);


		// Line 3
		HBox l3 = new HBox( false, 0);
		packStart( l3, false, false, 0);

		sync = new Label("");
		l3.packStart(sync, false, true, 0);
		timer = new Timer( 1000 * 60, this );
		timer.start();
		severity = new Label( "No Report Loaded" );
		l3.packStart(severity, true, true, 0);
		age = new Label("");
		l3.packStart(age, false, false, 0);

		cv = new CommentViewer();
		packStart( cv, true, true, 0 );
	}
	
	public void comment(){
		if (bug == null) return;
		BugEdit editor = new BugEdit(bug, cv.getQuote());
		editor.showAll();
	}
	public void control(){
		if (bug == null) return;
		(new ControlWindow( bug )).showAll();
	}

	private Thread downloaderThread = null;
		
	public synchronized void refresh(){
		if (bug == null) return;
		appBar.clearStack();
		appBar.pushStack("Starting to download report");
		appBar.setProgressPercentage(1.0);
		appBar.showAll();
		while (downloaderThread != null &&
				downloaderThread.isAlive()){
			downloaderThread.interrupt();
		}
		downloaderThread = new Thread( new Runnable(){
			public void run(){
				updateBug();
			}
		});
		downloaderThread.start();
	}

	/** This is called in a different thread - act accordingly. **/
	private void updateBug(){
		if (bug == null) return;
		try{
			bug.updateVerbose();
		}catch(Exception e){
			e.printStackTrace();
			parent.error("Error updating bug: "+e );
			// todo
		}
		CustomEvents.addEvent( new Runnable(){
			public void run(){
				setBug();
			}
		});
	}

	public Bug getBug(){
		return bug;
	}
	
	public void setBug(){
		setBug(bug);
	}
	public void setBug(int bug){
		try{
			setBug( BTS.getInstance().getBug( new Integer( bug ) ) );
		}catch( Exception e ){
			e.printStackTrace();
			parent.error("Error updating bug: "+e );
			throw new RuntimeException( "Function not implemented" );
			// todo.
		}
	}
	public void setBug(Bug bug){
		if (this.bug!= null && this.bug != bug ){
			this.bug.removeListener( (BugListener) this );
		}
		this.bug = bug;
		bug.addListener( (BugListener) this );

		if (! bug.getComplete()){
			// bug does not have any data - need to download it
			refresh();
			return;
		}

		appBar.clearStack();
		appBar.pushStack("Setting Report Details");
		appBar.getProgressBar().pulse();
		appBar.showAll();
		if (DebbugGtk.DEBUG) System.out.println("Bug report: setting bug to #"+bug.getNumber());
		cv.setBug(bug);
		numbr.setText( (new Integer( bug.getNumber() )).toString() );
		title.setMarkup( "<span weight=\"bold\">" + bug.getTitle() + "</span>");

		setChecked();

		DateFormat df = DateFormat.getDateInstance( DateFormat.MEDIUM );
		age.setText(  df.format( (new Date(bug.getCreated()))) );
		pkg.setText( bug.getPackageName() + "   " );

		Severity s = bug.getSeverity();
		if(s.equals( Severity.CRITICAL ))
			severity.setMarkup("Severity: <span foreground=\"#ff0000\">CRITICAL</span>");
		else if (s.equals( Severity.SERIOUS ))
			severity.setMarkup("Severity: <span foreground=\"#ff0000\">serious</span>");
		else if (s.equals( Severity.GRAVE ))
			severity.setMarkup("Severity: <span foreground=\"#ff0000\">grave</span>");
		else
			severity.setText( "Severity: " + s.toString() );

		remove(tags);
		tags = new HBox(false, 0);
		packStart(tags, false, true, 0);
		reorderChild(tags, 1);

		LinkedList bugTags = (LinkedList) bug.getTags();
		while (bugTags != null && bugTags.size() > 0){
			String tag = (String) bugTags.removeFirst();
			Label l = new Label("["+tag+"]");
			tags.packStart(l, true, false, 0);
		}
		
		
		if (null != bug.getForwarded() && !bug.getForwarded().equals("")){
			Label f = new Label("[Forwarded: " );
			HBox hb = new HBox(false, 0);
			tags.packStart(hb, true, false, 0);
			hb.packStart(f, false, false, 0);
			String fwd = bug.getForwarded();
			while (fwd.indexOf(">")>=0)
				fwd = fwd.substring(0, fwd.indexOf(">")) + "&gt;" + fwd.substring(fwd.indexOf(">")+1);
			while (fwd.indexOf("<")>=0)
				fwd = fwd.substring(0, fwd.indexOf("<")) + "&lt;" + fwd.substring(fwd.indexOf("<")+1);
			org.gnu.gnome.HRef hr = new org.gnu.gnome.HRef( bug.getForwarded(), fwd );
			Label fEnd = new Label("]");
			hb.packStart(hr, false, false, 0);
			hb.packStart(fEnd, false, false, 0);
		}
		if (null != bug.getDone() && !bug.getDone().equals("")){
			Label f = new Label("foo");
			String str = bug.getDone() ;
			int i = str.indexOf( ">" );
			while (i >= 0 ){
				str = str.substring(0, i) + "&gt;" + str.substring( i + 1 );
				i = str.indexOf( ">" );
			}
			i = str.indexOf( "<" );
			while (i >= 0 ){
				str = str.substring(0, i) + "&lt;" + str.substring( i + 1 );
				i = str.indexOf( "<" );
			}
			f.setMarkup( "<span foreground=\"green\">[Done: "+ str +"]</span>");
			tags.packStart(f);
		}
		int[] mergedBugs = bug.getMerged();
		ButtonBox mbb = null;
		if (mergedBugs.length>0){
			int[] mb = new int[ mergedBugs.length + 1];
			for(int i = 0; i < mergedBugs.length; i++)
				mb[i] = mergedBugs[i];
			mb[mergedBugs.length] = bug.getNumber();
			mergedBugs = mb;
			java.util.Arrays.sort( mergedBugs );
			tags.packStart( new Label("[Merged: "), false, false, 0 );
			mbb = new HButtonBox();
			tags.packStart(mbb, false, false ,0);
			tags.packStart( new Label("]"), false, false, 0 );
		}
		for (int i = 0; i< mergedBugs.length; i++){
			Button merged = new Button(""+mergedBugs[i] );
			final int bugNo = mergedBugs[i];
			merged.addListener(new  ButtonListener(){
				public void buttonEvent(ButtonEvent event){
					if (event.isOfType(ButtonEvent.Type.CLICK)){
						setBug(bugNo);
					}
				}
			});
			if (mergedBugs[i] == bug.getNumber())
				merged.setSensitive(false);
			mbb.packStart( merged );			
		}

		showAll();

		// set status as read 
		// todo: have a timeout (possibly varialbe time) for this
		// todo: have options to set status as unread.
		if (DebbugGtk.DEBUG) System.out.println("About to set bug status to read if not already");
		if ( !bug.getStatus().equals( Status.READ ) ){
			bug.setStatus( Status.READ );
		}
		if (DebbugGtk.DEBUG) System.out.println("Done");
		appBar.clearStack();
		appBar.pushStack("Finished Loading Bug Report");
		appBar.setProgressPercentage(1.0);
		appBar.showAll();
	}

	/**
	 * Sets a label showing how long it is since the report was last checked
	 * with the server.
	 */
	private void setChecked(){
		if (bug == null){
			return;
		}
		long diff = ((new Date()).getTime() - bug.getLastUpdated() ) / 1000; // time difference in seconds
		long days, hours, minutes;
		days = diff / 86400;
		hours = (diff / 3600) % 24;
		minutes = (diff / 60) % 60;
		if (days > 3){
			sync.setMarkup("<span foreground=\"#ff0000\">Checked: "
					+ days + " day"+ (days > 1 ? "s ago</span>" : " ago</span>") );
		}else if (days > 0){
			sync.setMarkup("<span foreground=\"#ff0000\">Checked: "
					+ days + " day"+ (days > 1 ? "s" : "") 
					+ ", " + hours+" hr"+ (hours > 1 ? "s ago</span>" : " ago </span>") );
		}else if (hours > 0){
			sync.setText("Checked: " + hours+" hr"+ (hours > 1 ? "s" : "") 
					+ ", " + minutes +" minute" + (minutes > 1 ? "s ago" : " ago") );
		}else if (minutes <= 0){
			sync.setText("Checked: <1 minute ago");
		}else{
			sync.setText("Checked: " +  minutes +" minute" + (minutes > 1 ? "s ago" : " ago") );
		}
	}
	/**
	 * Handle events from the Timer object. This will be called every 60s from
	 * the main gtk loop.
	 */
	public boolean fire(){
		setChecked();
		return true; // continue calling this function.
	}
	
	public void reload(){
		// todo
	}
	////////////////////////////////////////////////////////////////////////////////
	// BugListener Implementation
	/**
	 * Called whenever the read/unread status of <code>bug</code> changes.
	 */
	public void bugStatusChanged( Bug bug, Status oldStatus ){
		CustomEvents.addEvent( new Runnable(){
			public void run(){
				reload();
			}});
	}

	/**
	 * Called when the severity of <code>bug</code> changes
	 */
	public void bugSeverityChanged( Bug bug, Severity oldSeverity ){
		CustomEvents.addEvent( new Runnable(){
			public void run(){
				reload();
			}});
	}
	/**
	 * Called when bug comments or tags are updated (i.e. everything not covered
	 * by other events)
	 */
	public void bugUpdated( Bug bug ){
		CustomEvents.addEvent( new Runnable(){
			public void run(){
				reload();
			}});
	}
	/**
	 * Called when an exception occurs with a bug (e.g. error contacting server)
	 * @return true if execution should continue
	 */
	public boolean bugException( Bug bug, Exception e ){
		final Exception f = e;
		CustomEvents.addEvent( new Runnable(){
			public void run(){
			f.printStackTrace();
			appBar.clearStack();
			appBar.pushStack("Bug Exception: "+f);
			appBar.setProgressPercentage(0.0);
			appBar.showAll();
			}});
		return false;
	}

	/**
	 * Called when a bug is being retrieved, If verbose progress notifications
	 * have been requested
	 */
	public void retrievingBug( Bug bug ){
		CustomEvents.addEvent( new Runnable(){
			public void run(){
				appBar.clearStack();
				appBar.pushStack("Retrieving Bug Report (currently displayed bug)");
				appBar.getProgressBar().pulse();
				appBar.showAll();
			}});
	}
	/**
	 * Called when downloading of a bug report has completed
	 */
	public void bugDownloaded( Bug bug ){
		CustomEvents.addEvent( new Runnable(){
			public void run(){
				appBar.clearStack();
			appBar.pushStack("Bug Successfully Retrieved - interpreting (currently displayed bug)");
			appBar.setProgressPercentage(1.0);
			appBar.showAll();
			reload();
			}}
		);
	}

}
