package com.tildemh.debbuggtk;

import com.tildemh.debbug.*;
import java.util.*;

/**
 * Abstract class for filtering bug reports. Includes static
 * implementations.
 *
 * <h3>TODO:</h3>
 * <p>I need suggestions on how to implement this better. The current state is a
 * mess. we need to support AND and OR, and set reasonable defaults. For
 * example, it is silly having severities ANDED together.
 * <p>Please help!
 * 
 *
 * <p>This is released under the terms of the GNU Lesser General Public License
 * (LGPL). See the COPYING file for details.
 *
 * @version $Id: BugFilterer.java,v 1.10 2004/01/11 17:11:23 mh Exp $
 * @author &copy; Mark Howard &lt;mh@debian.org&gt; 2002
 */
public abstract class BugFilterer{
	
	/**
	 * Determines whether the given bg is acceptable by the filterer
	 */
	public abstract boolean acceptable( Bug bug );

	/**
	 * Returns a string representing the filter
	 */
	public abstract String toString();

	public static BugFilterer ALL = new BugFilterer(){
		public boolean acceptable(Bug bug){ return true; }
		public String toString(){
			return "";
		}
	};

	public static BugFilterer or(BugFilterer[] filters){
		final BugFilterer[] filt = filters;
		return new BugFilterer(){
			public boolean acceptable(Bug bug){
				for (int i = 0; i < filt.length; i++)
					if (filt[i].acceptable(bug)) return true;
				return false;
			}
			public String toString(){
				StringBuffer Str = new StringBuffer();
				for( int i = 0; i < filt.length - 1; i++){
					Str.append("(");
					Str.append( filt[i].toString() );
					Str.append(") OR (");
				}
				Str.append(filt[filt.length].toString());
				Str.append(")");
				return Str.toString();
			}
		};
	}

	public static BugFilterer and(BugFilterer[] filters){
		final BugFilterer[] filt = filters;
		return new BugFilterer(){
			public boolean acceptable(Bug bug){
				for (int i = 0; i < filt.length; i++)
					if (!filt[i].acceptable(bug)) return false;
				return true;
			}
			public String toString(){
				StringBuffer Str = new StringBuffer();
				for( int i = 0; i < filt.length - 1; i++){
					Str.append("(");
					Str.append( filt[i].toString() );
					Str.append(") AND (");
				}
				Str.append(filt[filt.length].toString());
				Str.append(")");
				return Str.toString();
			}
		};
	}

	public static BugFilterer not(BugFilterer filter){
		System.out.println("not( filterer");
		final BugFilterer filt = filter;
		return new BugFilterer(){
			public boolean acceptable(Bug bug){
				return !filt.acceptable(bug);
			}
			public String toString(){
				return "!"+filt.toString();
			}
		};
	}
	
	public static BugFilterer title(String str){
		final String contains = str;
		return new BugFilterer(){
			public boolean acceptable(Bug bug){
				return (bug.getTitle().indexOf(contains) >= 0);
			}
			public String toString(){
				return "Title:\""+contains+"\"";
			}
		};
	}
	public static BugFilterer severity (Severity severity){
		final Severity sev = severity;
		return new BugFilterer(){
			public boolean acceptable(Bug bug){
				return bug.getSeverity().equals(sev);
			}
			public String toString(){
				return "Severity:"+sev.toString();
			}
		};
	}
	public static BugFilterer tag(String tag) throws BugFilterInvalid{
		if (tag.equals("patch")){
			return new BugFilterer(){ 
				public boolean acceptable(Bug bug){ return bug.getTagPatch(); } 
				public String toString(){ return "Tag:patch"; } };
		}if (tag.equals("help")){
			return new BugFilterer(){ 
				public boolean acceptable(Bug bug){ return bug.getTagHelp(); } 
				public String toString(){ return "Tag:help"; } };
		}if (tag.equals("moreinfo")){
			return new BugFilterer(){ 
				public boolean acceptable(Bug bug){ return bug.getTagMoreinfo(); } 
				public String toString(){ return "Tag:moreinfo"; } };
		}if (tag.equals("unreproducible")){
			return new BugFilterer(){ 
				public boolean acceptable(Bug bug){ return bug.getTagUnreproducible(); } 
				public String toString(){ return "Tag:unreproducible"; } };
		}if (tag.equals("pending")){
			return new BugFilterer(){ 
				public boolean acceptable(Bug bug){ return bug.getTagPending(); } 
				public String toString(){ return "Tag:pending"; } };
		}if (tag.equals("fixed")){
			return new BugFilterer(){ 
				public boolean acceptable(Bug bug){ return bug.getTagFixed(); } 
				public String toString(){ return "Tag:fixed"; } };
		}if (tag.equals("wontfix")){
			return new BugFilterer(){ 
				public boolean acceptable(Bug bug){ return bug.getTagWontfix(); } 
				public String toString(){ return "Tag:wontfix"; } };
		}else if (tag.equals("fwd")){
			return new BugFilterer(){ 
				public boolean acceptable(Bug bug){ return bug.getIsForwarded(); } 
				public String toString(){ return "Tag:fwd"; } };
		}else{
			throw new BugFilterInvalid( "Tag filtering not yet fully implemented");
		}
	}

	public static BugFilterer body(String body){
		final String word = body;
		return new BugFilterer(){
			public boolean acceptable(Bug bug){
				LinkedList c = bug.getComments();
				for(int i = 0; i < c.size(); i++)
					if ( ((BugComment) c.get(i)).getSource().indexOf(word) >= 0)
						return true;
				return false;
			}
			public String toString(){ return "Body:\""+word+"\""; }
		};
				
	}

	public static BugFilterer done = new BugFilterer(){
		public boolean acceptable(Bug bug){
			return bug.getIsDone();
		}
		public String toString(){ return "done"; }
	};
			


	public static BugFilterer ATTN(){
		try{
		return not( 
				or( new BugFilterer[] {
					tag("fwd"), 
					tag("unreproducible"), 
					tag("help"), 
					tag("moreinfo"),
					tag("pending"),
					tag("fixed"),
					tag("wontfix"),
					done
				} ));
		}catch(Exception e){
			throw new RuntimeException(e.toString());
		}
	}
	public static BugFilterer RC(){
		return or( new BugFilterer[] { severity( Severity.CRITICAL ), severity( Severity.GRAVE ), severity( Severity.SERIOUS ) } );
	}

	/** 
	 * Constructs a filter from its string representation
	 */
	public static BugFilterer makeFilterer( String filter ) throws BugFilterInvalid{
		System.out.println("makeFilterer("+filter+")");
		if (filter.equals("")) return ALL;
		if (filter.equals("RC")) return RC();
		if (filter.equals("ATTN")) return ATTN();
		StringTokenizer st = new StringTokenizer(filter);
		LinkedList parts = new LinkedList();
		 while (st.hasMoreTokens()) {
			 parts.addFirst( st.nextToken() );
		 }
		System.out.println(parts.size());
		 if (parts.size() > 1){
			 BugFilterer[] anded = new BugFilterer[parts.size()];
			 for(int i = 0; i < parts.size(); i++){
				 anded[i] = makeFilterer( (String) parts.get(i) );
			 }
			 return and( anded );
		 }
		 // else we just have one thing to deal with
		 if (filter.indexOf("tag:")==0 || filter.indexOf("Tag:")==0 || filter.indexOf("TAG:")==0 ){
			 return tag( filter.substring(4) );
		 }
		 if (filter.indexOf("severity:")==0 || filter.indexOf("Severity:")==0 || filter.indexOf("SEVERITY:")==0 ){
			 Severity s = Severity.parse( filter.substring(9) );
			 if (s.getValue() < 0)
				 throw new BugFilterInvalid( "Unknown severity "+filter.substring(9) );
			 return severity( s );
		 }
		 if (filter.indexOf("body:")==0 || filter.indexOf("Body:")==0 || filter.indexOf("BODY:")==0 ){
			 return body( filter.substring(5) );
		 }

		 System.out.println("filter.substring(0,1)="+filter.substring(0,1)+ "    "+ filter.substring(1));
		 if (filter.substring(0,1).equals("!") ){
			 return not( makeFilterer( filter.substring(1) ) );
		 }
			 
		 return title(filter);	 
			 
	}

}
