/* hey emacs! -*- Mode: C++; c-file-style: "stroustrup"; indent-tabs-mode: nil -*- */
/*
 * $Id: UDP.cc,v 1.8 2001/01/24 04:18:14 rex Exp $
 *
 * Copyright (c) 2000 Remi Lefebvre  <remi@dhis.net>
 *                and Luca Filipozzi <lfilipoz@dhis.net>
 *
 * DDT comes with ABSOLUTELY NO WARRANTY and is licenced under the
 * GNU General Public License (version 2 or later). This license
 * can be retrieved from http://www.gnu.org/copyleft/gnu.html.
 *
 */

#include <stdio.h>
#include <string.h>
#include <netdb.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <unistd.h>

#include "UDP.h"
#include "Exception.h"

// server constructor
UDP::UDP (unsigned int clientport, unsigned int serverport)
{
    isClient_ = false;
    clientport_ = clientport;
    serverport_ = serverport;

    // fill in the server_addr_ structure
    bzero ((void *)&server_addr_, sizeof (server_addr_));
    server_addr_.sin_family = AF_INET;
    server_addr_.sin_port   = htons (serverport);
    bzero ((void *)&client_addr_, sizeof (client_addr_));
    client_addr_.sin_addr.s_addr = INADDR_ANY;

    // don't need to fill in the client_addr_ structure... see UDP::sendTo()

    // get a socket with which to speak to the clients
    if ((send_sd_ = socket(AF_INET, SOCK_DGRAM, 0)) == -1)
    {
        throw DdtException ("socket() failed");
    }

    if ((recv_sd_ = socket(AF_INET, SOCK_DGRAM, 0)) == -1)
    {
        throw DdtException ("socket() failed");
    }

    // and bind the socket to the server_addr_ structure
    if (bind (recv_sd_, (struct sockaddr *)&server_addr_, sizeof (server_addr_)) == -1)
    {
        throw DdtException ("bind() failed");
    }
}

// client constructor
UDP::UDP (char *hostaddr, unsigned int clientport, unsigned int serverport)
{
    isClient_  = true;
    serverport_ = serverport;
    clientport_ = clientport;

    // fill in the server_addr_ structure
    struct hostent *serverHostEntry;
    if ((serverHostEntry = gethostbyname (hostaddr)) == NULL)
    {
        throw DdtException("gethostbyname() failed");
    }

    server_addr_.sin_family      = AF_INET;
    server_addr_.sin_port        = htons (serverport);
    memcpy(&server_addr_.sin_addr, serverHostEntry->h_addr, serverHostEntry->h_length);

    // fill in the client_addr_ structure
    client_addr_.sin_family      = AF_INET;
    client_addr_.sin_port        = htons (clientport);
    client_addr_.sin_addr.s_addr = INADDR_ANY;  // could use next line too
    //memcpy(&client_addr_.sin_addr, serverHostEntry->h_addr, serverHostEntry->h_length);

    // get a socket with which to speak to the server
    if ((send_sd_ = socket (AF_INET, SOCK_DGRAM, 0)) == -1)
    {
        throw DdtException ("socket() failed");
    }

    if ((recv_sd_ = socket (AF_INET, SOCK_DGRAM, 0)) == -1)
    {
        throw DdtException ("socket() failed");
    }

    // and bind the socket to the client_addr_ structure
    if (bind (recv_sd_, (struct sockaddr *)&client_addr_, sizeof (client_addr_)) == -1)
    {
        throw DdtException ("bind() failed");
    }
}

UDP::~UDP ()
{
    close (send_sd_);
    close (recv_sd_);
}

// recv() .. both for client and server.
int UDP::recv (void *buf, int bufSize, unsigned long *from)
{
    struct sockaddr_in addr;
    socklen_t addrSize = sizeof(struct sockaddr_in);

    if (recvfrom (recv_sd_,                // the socket descriptor
                  buf,                     // the buffer received
                  bufSize,                 // the buffer's size
                  0,                       // flags
                  (struct sockaddr*)&addr, // who we received from
                  &addrSize) == -1)        // size of the receive struct
    {
        return -1;
        
    }

    if (isClient_ && addr.sin_addr.s_addr != server_addr_.sin_addr.s_addr)
    {
        // we are a "client" but we received a packet from an
        // ip address different than that of the "server"
        return -2;
    }

    if (from != NULL)
    {
        *from = addr.sin_addr.s_addr;
    }
    return 0;
}

// for server, send a packet to a specified address
int UDP::sendTo (unsigned long ipAddress, void *buf, int bufSize)
{
    if (isClient_)
    {
        return -1;
    }

    struct sockaddr_in addr;
    addr.sin_family      = AF_INET;
    addr.sin_port        = htons (clientport_);
    addr.sin_addr.s_addr = ipAddress;

    socklen_t addrSize = sizeof (struct sockaddr_in);
    return sendto (send_sd_,                  // the socket descriptor
                   buf,                       // the buffer to send
                   bufSize,                   // the buffer's size
                   0,                         // flags
                   (struct sockaddr *)&addr,  // who we are sending to
                   addrSize);                 // size of the receive struct
}

// for client. send a packet to server.
int UDP::send (void *buf, int bufSize)
{
    if (!isClient_)
    {
        return -1;
    }

    socklen_t addrSize = sizeof (struct sockaddr_in);
    return sendto (send_sd_,                  // the socket descriptor
                   buf,                       // the buffer to send
                   bufSize,                   // the buffer's size
                   0,                         // flags
                   (struct sockaddr *)&server_addr_, // who we are sending to
                   addrSize);                 // size of the receive struct
}
