/* DCTC - a Direct Connect text clone for Linux
 * Copyright (C) 2001 Eric Prevoteau
 *
 * dc_xfer_common.h: Copyright (C) Eric Prevoteau <www@a2pb.gotdns.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
/*
$Id: dc_xfer_common.h,v 1.3 2004/01/09 18:16:01 ericprev Exp $
*/

#ifndef __DC_XFER_COMMON_H__
#define __DC_XFER_COMMON_H__

#include "action.h"

typedef enum
{
	MY_DIR_ERROR=-1,				/* invalid direction */
	MY_DIR_DOWNLOAD=0,			/* what we should do on the connection: Download */
	MY_DIR_UPLOAD=1,				/* what we should do on the connection: Upload */
} MY_DIR;

typedef struct
{
	const char *cmd;  
	size_t cmd_len;
	int (*fnc)(const char *cmd,int sck,GString *input, char *xtra_param);
	char *xtra_param;
} CMD_REPLY;

typedef struct
{
	const char *cmd;
	size_t cmd_len;
	int (*fnc)(const char *cmd,WAIT_ACT *act,int sck,GString *input, char *xtra_param);
	char *xtra_param;
} CMD_REPLY_ACT;

extern int hub_logged;     /* set to 1 when dctc is logged on the hub */

/********************************************************/
/* when someone connects on com port, send our nickname */
/********************************************************/
/* output: 0=ok, !=0=fail */
/**************************/
int send_nick(int sck);

/******************************************************/
/* take the given $Direction string and decode fields */
/**********************************************************/
/* output: MY_DIR_ERROR, MY_DIR_DOWNLOAD or MY_DIR_UPLOAD */
/*         if no error occurs, *level contains the level  */
/**********************************************************/
MY_DIR decode_direction(GString *input, unsigned int *level);

/***************************************************************************/
/* create a lock inside the 'created_lock' array and send it on the socket */
/***************************************************************************/
int send_a_lock(int sck,char *created_lock);

/*************************************/
/* send the length of the given file */
/*************************************************/
/* output: 0=ok, 1=error                         */
/*         if ok, *file_len is the reported size */
/*************************************************/
int send_file_length(int sck,char *filename, unsigned long long *file_len);

/***********************************/
/* send the data of the given file */
/***********************************/
/* output: 0=ok, 1=error */
/*************************/
int send_file_data(int sck,char *filename, unsigned long long start_pos, unsigned long long file_len,WAIT_ACT *act);

/************************************/
/* send the data of the given array */
/************************************/
/* output: 0=ok, 1=error */
/*************************/
int send_array_data(int sck,GByteArray *ba,WAIT_ACT *act);

/* convert path from dos to unix format */
void unconvert_path(char *str);

/*******************************************************/
/* process the "$GetListLen " command                  */
/*******************************************************/
/* $GetListLen is always followed by               */
/* a $Send| which is processed by this function    */
/**************************************************************************/
/* this function should return the list of all files shared by the client */
/**************************************************************************/
int com_up_get_list_len_process(const char *cmd,WAIT_ACT *act,int sck,GString *input, char *xtra_param);

/*******************************************************/
/* process the "$Get " command                         */
/*******************************************************/
/* $Get file full path$val| is always followed by */
/* a $Send| which is processed by this function    */
/***************************************************/
int com_up_get_process(const char *cmd,WAIT_ACT *act,int sck,GString *input, char *xtra_param);

/*********************************************************************/
/* process the "$GetFileLength " command (it is an extended command) */
/*********************************************************************/
/* $GetFileLength file full path| */
/**********************************/
int com_up_getfilelength_process(const char *cmd,WAIT_ACT *act,int sck,GString *input, char *xtra_param);

/**************************************************/
/* process the $Capabilities from the xfer socket */
/**************************************************/
int xfer_capabilities_process(const char *cmd,WAIT_ACT *act,int sck,GString *input, char *xtra_param);

/**************************************************************************/
/* when we enter in "Upload" direction, this function process all uploads */
/**************************************************************************/
void manage_com_upload(WAIT_ACT *act);

/***************************************************************/
/* take a "$FileLength xxx" string and return the value of xxx */
/***************************************************************/
/* output: 0=ok (*file_size=xxx), 1=error */
/******************************************/
int extract_filelength(char *str, unsigned long *file_size);

/************************************/
/* process the given WAITING_REVCON */
/*************************************************************/
/* 0=operation done, 1=operation killed or error encountered */
/*************************************************************/
int start_a_xdownload(WAIT_ACT *act, WAIT_REVCON *act_to_do);

/**************************************************************************************/
/* read amount bytes from remote file descriptor and write them into given byte array */
/**************************************************************************************/
/* output: 2=network error, 0=ok                                    */
/*         in all cases, (*ba) will be a newly allocated GByteArray */
/********************************************************************/
/* warning: don't use read to get data from network, prefer recv */
/*****************************************************************/
int copie_fd_to_bytearray(int remote, GByteArray **ba, unsigned long amount,WAIT_ACT *act);

/***********************************************/
/* convert \r\n into \n in the given string    */
/* In fact, we just remove all \r we encounter */
/***********************************************/
GString *my_g_string_dos2unix(GString *str);

/****************************************************************************************************/
/* take the file list (in content), save it in the cache and send the "cache" message to the client */
/****************************************************************************************************/
void generate_shared_file_list(WAIT_REVCON *act_to_do, GString *content);

/************************************/
/* process the given WAITING_REVCON */
/*************************************************************/
/* 0=operation done, 1=operation killed or error encountered */
/*************************************************************/
int start_a_ls(WAIT_ACT *act, WAIT_REVCON *act_to_do);

/***********************/
/* start a queued task */
/******************************/
/* output: 0=task done        */
/*       !=0=unable to comply */
/******************************/
int run_action_with_remote_side(WAIT_ACT *act,WAIT_REVCON *act_to_do);

/***************************************************/
/* send this client capabilities to the remote one */
/***************************************************/
void send_client_capabilities(WAIT_ACT *act);

/**************************************************************/
/* copy the given wait_recon into GTS (or sim_input on error) */
/**************************************************************/
void requeue_act_to_do(WAIT_REVCON *act_to_do);

/******************************/
/* thread managing connect_me */
/******************************/
void *connect_me_thread(WAIT_ACT *act);

/*************************************************************************************/
/* when a remote host ask this client to open a connection to himself, this function */
/* opens the connection and create a thread to manage it.                            */
/*************************************************************************************/
void *do_connect_to_me(GString *host_port);

/*******************************************************/
/* process the "$HubName " command                     */
/* this function is called when an user enters the hub */
/*******************************************************/
int hubname_process(const char *cmd,int sck,GString *input, char *xtra_param);

/*******************************************/
/* send the client capabilities to the hub */
/*******************************************/
void send_client2hub_capabilities(void);

/***********************************************************************/
/* in the string 'msg', replace all 'in' characters by 'out' character */
/***********************************************************************/
void subst_char(char *msg,char in, char out);

/*********************************************************/
/* process the "$NickList xxx$$[yyy$$zzz$$...]|" command */
/*********************************************************/
int nicklist_process(const char *cmd,int sck,GString *input, char *xtra_param);

/*****************************************************************/
/* process the "$MyINFO $ALL xxx yyyy$ $zzzf$aaa$bbbb$" command  */
/* xxx= nickname                                                 */
/* yyy= file description (can be empty)                          */
/* zzz= connection type (Eg. DSL,Cable)                          */
/* f is a 1 byte flag. default value: 0x01                       */
/*         if the user is here, bit 1=1, else bit1=0 */
/* aaa=e-mail (can by empty)                                     */
/* bbb=size of shared data (in bytes)                            */
/*****************************************************************/
int myinfo_process(const char *cmd,int sck,GString *input, char *xtra_param);

/*****************************************/
/* convert the given letter into boolean */
/*****************************************/
/* v='T' -> *result=1, return 0 */
/* v='F' -> *result=0, return 0 */
/* else return 1                */
/******************************************************************/
/* WARNING: don't inline this function. With gcc2.96, if inlined, */
/* search_process no more works                                   */
/******************************************************************/
int bool_letter(char v,int *result);

/*******************************************************/
/* process the "$GetPass" command                      */
/* this function is called when an user enters the hub */
/*******************************************************/
int getpass_process(const char *cmd,int sck,GString *input, char *xtra_param);

/*******************************************************/
/* process the "$BadPass" command                      */
/* this function is called when an user enters the hub */
/*******************************************************/
int badpass_process(const char *cmd,int sck,GString *input, char *xtra_param);

/***************************************************************/
/* process the "$LogedIn " command                             */
/* this function is called when the user is logged as an admin */
/***************************************************************/
int logedin_process(const char *cmd,int sck,GString *input, char *xtra_param);

/*******************************************************/
/* process the "$ValidateDenied" command               */
/* this function is called when the nick is already    */
/* use on the hub                                      */
/*******************************************************/
int validate_denied_process(const char *cmd,int sck,GString *input, char *xtra_param);

extern GPtrArray *hub_capabilities;

/***************************************************************/
/* process the "$Capabilities" command                         */
/* this function is called when the hub sends its capabilities */
/***************************************************************/
int capabilities_process(const char *cmd,int sck,GString *input, char *xtra_param);

/********************************************************************************/
/* process the "$ForceMove " command                                            */
/* this function is called when a hub tries to reroute the client to another op */
/********************************************************************************/
int force_process(const char *cmd,int sck,GString *input, char *xtra_param);

/***************************************************************************************/
/* check if the given message is a kick/ban message and add the IP into the UADDR list */
/***************************************************************************************/
void grab_ban_ip_fnc(GString *input);

/*******************************************************/
/* check if the given string has the following format: */
/* <Hub-Security> The user xxx was kicked by yyy       */
/*******************************************************/
/* output: 1=yes, 0=no */
/***********************/
int is_a_kick_message(char *str);

/***********************************************************************************************************/
/* check if the given public chat message is not emitted by a nickname having the "IGNORE_PUBMSG" flag set */
/***********************************************************************************************************/
/* output: 1= message is ok          */
/*         0= message is not allowed */
/*************************************/
int nick_does_not_have_ignore_pubmsg_flag(const GString *input);

/*************************************************************************************/
/* this function is called each time the client receives a full line from the server */
/*****************************************************************************************/
/* sck is provided because the function to process may requires and access to the socket */
/*****************************************************************************************/
/* output: 0: continue */
/*         1: end      */
/***********************/
int process_incoming_dc_data(int sck,GString *input);

/*************************************************/
/* get dc lines from network until nothing comes */
/**************************************************************************************/
/* all incoming lines are processed either to send new commands or to display results */
/**************************************************************************************/
void get_dc_line_and_process(int sck);

/********************************************/
/* someone tries to connect to the com port */
/********************************************/
void manage_com_port(int in_sck);

#endif
