/* DChub - a Direct Connect Hub for Linux
 * Copyright (C) 2001 Eric Prevoteau
 *
 * map_tools.c: Copyright (C) Eric Prevoteau <www@ac2i.tzo.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
/*
$Id: mmap_tools.c,v 2.1 2003/04/26 10:09:33 ericprev Exp $
*/

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <errno.h>
#include <string.h>

#ifdef HAVE_MMAP

#include <sys/mman.h>

/*************************/
/* map a file in memory. */
/********************************************************/
/* output: ==NULL -> error                              */
/*         !=NULL -> address of the mapped file         */
/*                   and *file_length contains its size */
/********************************************************/
void *map_file(const char *file_name, size_t *file_length)
{
	void *adr;
	int fd;
	struct stat st;

	fd=open(file_name,O_RDONLY);
	if(fd==-1)
	{
		fprintf(stderr,"map_file: open %s fails (%s)\n",file_name,strerror(errno));
		return NULL;
	}
	
	if(fstat(fd,&st)==-1)
	{
		fprintf(stderr,"map_file: stat %s fails (%s)\n",file_name,strerror(errno));
		close(fd);
		return NULL;
	}

	if(!S_ISREG(st.st_mode))
	{
		fprintf(stderr,"map_file: %s is not a regular file.\n",file_name);
		close(fd);
		return NULL;
	}

	if(st.st_size==0)
	{
		fprintf(stderr,"map_file: %s is an empty file.\n",file_name);
		close(fd);
		return NULL;
	}

	*file_length=st.st_size;
	adr=mmap(NULL,*file_length,PROT_READ,MAP_SHARED,fd,0);
	if(adr==MAP_FAILED)
	{
		fprintf(stderr,"map_file: fail to map %s (%s).\n",file_name,strerror(errno));
	}

	close(fd);
	return adr;
}

/**********************************/
/* unmap a previously mapped file */
/**********************************/
void unmap_file(void *mapped_addr, const size_t mapped_length)
{
	if(munmap(mapped_addr,mapped_length)==-1)
	{
		perror("unmap_file -> munmap failed");
	}
}

#else
#warning MMAP not available, this code does not simulate it, it just does a malloc instead and loads the file
/*************************/
/* map a file in memory. */
/********************************************************/
/* output: ==NULL -> error                              */
/*         !=NULL -> address of the mapped file         */
/*                   and *file_length contains its size */
/********************************************************/
void *map_file(const char *file_name, size_t *file_length)
{
	void *adr;
	int fd;
	struct stat st;

	fd=open(file_name,O_RDONLY);
	if(fd==-1)
	{
		fprintf(stderr,"map_file: open %s fails (%s)\n",file_name,strerror(errno));
		return NULL;
	}
	
	if(fstat(fd,&st)==-1)
	{
		fprintf(stderr,"map_file: stat %s fails (%s)\n",file_name,strerror(errno));
		close(fd);
		return NULL;
	}

	if(!S_ISREG(st.st_mode))
	{
		fprintf(stderr,"map_file: %s is not a regular file.\n",file_name);
		close(fd);
		return NULL;
	}

	if(st.st_size==0)
	{
		fprintf(stderr,"map_file: %s is an empty file.\n",file_name);
		close(fd);
		return NULL;
	}

	*file_length=st.st_size;

	adr=malloc(*file_length);
	if(adr==NULL)
	{
		fprintf(stderr,"map_file: fail to map %s (%s).\n",file_name,strerror(errno));
	}
	else
	{
		if(read(fd,adr,*file_length)!=(*file_length))
		{
			free(adr);
			adr=NULL;
		}
	}
	close(fd);
	return adr;
}

/**********************************/
/* unmap a previously mapped file */
/**********************************/
void unmap_file(void *mapped_addr, const size_t mapped_length)
{
	free(mapped_addr);
}

#endif
