/* DChub - a Direct Connect Hub for Linux
 * Copyright (C) 2001 Eric Prevoteau
 *
 * main_cnx_handshake.c: Copyright (C) Eric Prevoteau <www@ac2i.tzo.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
/*
$Id: main_cnx_handshake.c,v 2.30 2003/06/30 17:20:06 ericprev Exp $
*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/socket.h>
#include <stdarg.h>
#include <glib.h>

#include "main_cnx_handshake.h"
#include "ged.h"
#include "xf_io.h"
#include "gvar.h"
#include "main.h"
#include "key.h"
#include "timed_out_string.h"
#include "tos_key.h"
#include "multi_public_chat.h"
#include "user_cnx_lst.h"
#include "seen_list.h"
#include "global_user_if.h"
#include "mmap_tools.h"
#include "passwd.h"

static GArray *cnx_in_handshake=NULL;

static char *create_a_lock(void);
/* ************************************************************************** */
/* ************************************************************************** */
/* ************************************************************************** */
/* ********************* connection handshake handler *********************** */
/* ************************************************************************** */
/* ************************************************************************** */
/* ************************************************************************** */

/****************************************************/
/* register a new socket into the handshake handler */
/****************************************************/
/* if remote_addr ==NULL, the function computes it */
/***************************************************/
void cnx_handshake_register_new_fd(int sock_fd, struct in_addr *remote_addr)
{
	CNX_HSHAKE nw;
	char dummy_pk[17];
	int i;
	GString *tmp_buf;

	nw.xfio=create_xfio(sock_fd,remote_addr);
	nw.created_lock=create_a_lock();
	nw.log_step=STEP_LOCK_SENT;

	nw.temp_nick=NULL;
	nw.temp_passwd=NULL;
	nw.temp_privilege=0;
	nw.temp_ext_flag=0;

	g_array_append_val(cnx_in_handshake,nw);

	/* generate and send the '$Lock'. Doing this, we start the login handshake */
	for(i=0;i<16;i++)
	{
  		static const char * const valid_char="0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz?@*+-.,>=&';_]";
  		dummy_pk[i]=valid_char[rand()%strlen(valid_char)];
  	}
  	dummy_pk[16]='\0';

	tmp_buf=g_string_new("");
	g_string_sprintf(tmp_buf,"$Lock %s Pk=%s|",nw.created_lock,dummy_pk);
	send_gstr_to_xfio(nw.xfio,tmp_buf);
	g_string_free(tmp_buf,TRUE);

	send_gstr_to_xfio(nw.xfio,hubname);
	send_const_str_to_xfio(nw.xfio,"<Hub-Security> This hub is running version " VERSION " of the Linux Direct Connect Hub Software.|");
}

/*************************************/
/* destroy an entry under monitoring */
/**********************************************/
/* if keep_xfio == FALSE, the xfio is deleted */
/* else, the xfio is left untouched           */
/**********************************************/
static void cnx_handshake_destroy_cnx_hshake(int entry_idx, gboolean keep_xfio)
{
	CNX_HSHAKE *ch;

	ch=&(g_array_index(cnx_in_handshake,CNX_HSHAKE,entry_idx));

	if(ch->temp_nick)
		free(ch->temp_nick);
	if(ch->temp_passwd)
		free(ch->temp_passwd);
	if(keep_xfio==FALSE)
		delete_xfio(ch->xfio);
	free(ch->created_lock);
	g_array_remove_index_fast(cnx_in_handshake,entry_idx);
}

/**************************************************************************/
/* convert a registered CNX_HSHAKE into a registered LOCAL_USER_CNX_ENTRY */
/**************************************************************************/
static void cnx_handshake_convert_cnx_hshake_to_local_user_cnx_entry(CNX_HSHAKE *ch,int entry_idx)
{
	LOCAL_USER_CNX_ENTRY *luce;
	int flag;
	va_list ap;			/* technically, we don't care about the content of this variable but on Alpha, we cannot use NULL as a va_list */

	/* as soon the user is accepted add the nick to the seen list */
	add_seen_entry("hello",ch->temp_nick,0,ap,NULL);
	/* send_const_str_to(ptr,"<welcome> This hub is running " PACKAGE " " VERSION". build on "__DATE__" at "__TIME__".|"); */

	/* set the keep-alive flag to detect broken connection */
	flag=1;
   setsockopt(ch->xfio->sock_fd, SOL_SOCKET, SO_KEEPALIVE, &flag, sizeof(flag));

	luce=uc_create_entry(ch->xfio, ch->temp_nick,0,NULL,NULL,NULL,1,ch->temp_ext_flag,ch->temp_privilege);

	/* say hello to everyone */
	glus_do_hello(luce->user_nick->str);

	/* we still can use ch constant because it is shared with the user_cnx_lst_entry */
	if(luce->privilege & OP_PRIV)
	{
		/* if the new user is op, rebuild the oplist and resend it */
		GString *lists[2];
		glus_get_nick_lists(&lists[0],&lists[1]);

		g_string_prepend(lists[1],"$OpList ");
		g_string_append(lists[1],"||");
		GLUS_SEND_TO_EVERYONE(lists[1]);		/* don't free lists[1] */

		g_string_sprintf(lists[0],			/* let's reuse this unused variable */
				 "$Hello Hub-Security|$To: %s From: Hub-Security $Welcome %s\r\n"
				 "\r\n"
				 "Using this private chat, you can enter special commands:\r\n"
				 "Use -help for the command list.\r\n|",
				 luce->user_nick->str,
				 (luce->privilege & MASTER_PRIV)?"Master of the hub":"operator");

		
		send_stolen_gstr_to_luce(luce,lists[0]);	/* don't free lists[0] */
	}

	/* Send welcome message, reads from database */
	{
		char *rd;
		GString *msg;

		/* first, check MOTD key */
		G_LOCK(conf_file);
		rd=e_db_str_get(conf_file,"MOTD");
		G_UNLOCK(conf_file);

		if(rd!=NULL)
		{
			size_t flen;
			unsigned char *fadr;

			fadr=map_file(rd,&flen);
			free(rd);

			if(fadr==NULL)
				goto no_valid_motd;

			msg=g_string_new("");
			g_string_sprintf(msg,"$To: %s From: Hub $",luce->user_nick->str);
#ifdef WITH_GLIB2
			g_string_append_len(msg,fadr,flen);
#else
			{
				int i;

				for(i=0;i<flen;i++)
				{
					g_string_append_c(msg,fadr[i]);
				}
			}
#endif
			g_string_append_c(msg,'|');
			send_stolen_gstr_to_luce(luce,msg);		/* don't free msg */

			unmap_file(fadr,flen);
		}
		else
		{
			/* if MOTD does not exist or is invalid, check WELCOME key */
			no_valid_motd:
			G_LOCK(conf_file);
			rd=e_db_str_get(conf_file,"WELCOME");
			G_UNLOCK(conf_file);
			if(rd!=NULL)
			{
				if(strlen(rd)!=0)
				{
					msg=g_string_new("");
					g_string_sprintf(msg,"$To: %s From: Hub $%s|",ch->temp_nick,rd);
					send_stolen_gstr_to_luce(luce,msg);		/* don't free msg */
				}
				free(rd);
			}
		}
	}

	{
		unsigned int hip;
		GString *ip_number;

		ip_number = g_string_new("");
		hip=ntohl(luce->user_xfio->user_ip.s_addr);
							  
		g_string_sprintf(ip_number,"%hhu.%hhu.%hhu.%hhu",
								(unsigned char)(hip>>24)&0xff,
								(unsigned char)(hip >>16)&0xff,
								(unsigned char)(hip>>8)&0xff,
								(unsigned char)hip&0xff);
		SEND_EVT_TO("login",luce->user_nick->str,1,ip_number->str);
		g_string_free(ip_number,TRUE);
	}

	cnx_handshake_destroy_cnx_hshake(entry_idx,TRUE);	/* don't delete XFIO */
}

/*********************************************/
/* find a CNX_HSHAKE with the given nickname */
/************************************************************/
/* if avoid_this_ch is not NULL, this CNX_HSHAKE is ignored */
/**************************************************************/
/* only CNX_HSHAKE not having a CNX_CLOSED status is returned */
/**************************************************************/
static CNX_HSHAKE *get_cnx_hshake_by_nickname(const char *nickname, CNX_HSHAKE *avoid_this_ch)
{
	int i;
	CNX_HSHAKE *ch;

	for(i=0;i<cnx_in_handshake->len;i++)
	{
		ch=&(g_array_index(cnx_in_handshake,CNX_HSHAKE,i));
		if(ch==avoid_this_ch)
			continue;
		
		if((ch->temp_nick!=NULL)&&
			(!XFIO_IS_CLOSED(ch->xfio))&&
		   (!strcmp(nickname,ch->temp_nick)))
		{
			return ch;
		}
	}
	return NULL;
}

/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */
/* ------------------------------ misc functions ---------------------------- */
/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */
static void disp_cnx_hshake_info(CNX_HSHAKE *ch)
{
	unsigned int hip;

	hip=ntohl(ch->xfio->user_ip.s_addr);
	printf("loginfailure <%s> %hhu.%hhu.%hhu.%hhu\n",
			 (ch->temp_nick == NULL)?"":ch->temp_nick,
			 (unsigned char)(hip>>24)&0xff,
			 (unsigned char)(hip>>16)&0xff,
			 (unsigned char)(hip>>8) &0xff,
			 (unsigned char) hip	  &0xff);
	return;
}

/**************************************/
/* check if the given string is an IP */
/**************************************/
/* output: TRUE=yes, FALSE=no */
/******************************/
gboolean is_ip(char *str)
{
	unsigned int ip[4];
	if (sscanf(str, "%u.%u.%u.%u", ip + 0, ip + 1, ip + 2, ip + 3) == 4)
		return TRUE;
	return FALSE;
}  

/******************************/
/* create a valid $Lock value */
/******************************/
static char *create_a_lock(void)
{
	int i;
	int ln;
	char *lck;

	lck = malloc(100);
	ln=50+rand()%50;

	for(i=0;i<ln;i++)
	{
		lck[i]='%'+rand()%('z'-'%');	/* create a value between '%' and 'z' */
	}
	lck[ln]='\0';
	return (lck);
}

/******************************************************/
/* try to redirect a user to the default redirect hub */
/* return 0 if ok != else                             */
/******************************************************/
static int try_auto_redir(XF_IO *xfio)
{
	char *rd;
	GString *rd_msg;

	G_LOCK(conf_file);
	rd=e_db_str_get(conf_file,"REDIR_ADDR");
	G_UNLOCK(conf_file);
	if(rd==NULL)
		return (1);
	if(strlen(rd)==0)
	{
		free(rd);
		return (2);
	}

	/* send a forcemove with the destination address */
	rd_msg=g_string_new("");

	/* first the message to display */
	g_string_sprintf(rd_msg,"<Hub-Security> This hub is currently full! You are being redirected to %s.|",rd);
	send_gstr_to_xfio(xfio,rd_msg);

	/* and then the redirection command */
	g_string_sprintf(rd_msg,"$ForceMove %s|",rd);
	send_gstr_to_xfio(xfio,rd_msg);

	g_string_free(rd_msg,TRUE);

	free(rd);
	return(0);
}

/*****************************************/
/* check if too many users are connected */
/*************************************************/
/* output: 0=no, 1=yes                           */
/* if ==1, this function has handled redirection */
/*************************************************/
static int too_many_users(XF_IO *xfio)
{
	int cur_nb;
	int max_nb;

	cur_nb=user_cnx_entry_get_nb_users();
	
	G_LOCK(conf_file);
	if(!e_db_int_get(conf_file,"MAXUSER",&max_nb))
		max_nb=100;
	G_UNLOCK(conf_file);

	if(cur_nb>=max_nb)
	{
		int v;
		/* too many user connected */
		G_LOCK(conf_file);
		if(!e_db_int_get(conf_file,"REDIR_FULL",&v))
			v=0;
		G_UNLOCK(conf_file);

		if(v==0)
		{
			/* display an error message */
			disp_msg:
			send_const_str_to_xfio(xfio,"<Hub-Security> This hub is currently full!|");
			send_const_str_to_xfio(xfio,"$HubIsFull|");
		}
		else
		{
			if (try_auto_redir(xfio) != 0)
				goto disp_msg;
			/* can not redirect but the user must not log in */
		}
		XFIO_SET_CLOSE_STATUS(xfio,CNX_WILL_DIE); /* just send stored data */
		return 1;
	}
	else
	{
		int v;

		G_LOCK(conf_file);
		if(!e_db_int_get(conf_file,"ALL_REDIR",&v))
			v=0;
		G_UNLOCK(conf_file);
		if (v)
		{
			if (try_auto_redir(xfio) != 0)
				goto disp_msg;
			/* can not redirect but the hub is not full, so let him log in */
			return 0;
		}
	}
	return 0;
}

/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */
/* -------------------------------- step handler ---------------------------- */
/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */
/**************************************/
/* $Lock was sent, we wait for a $Key */
/******************************************/
/* next steps: on_success, STEP_KEY_VALID */
/*             else CNX_CLOSED is set     */
/******************************************/
static void sf_step_lock_sent(CNX_HSHAKE *ch)
{
	int lret;
	GString *inp;
	
	inp=xfio_take_first_string_of_glist_incoming(ch->xfio);
	if(inp==NULL)
	{
		XFIO_SET_CLOSE_STATUS(ch->xfio,CNX_CLOSED);	/* immediat termination */
		return;
	}

	lret=verify_key(ch->created_lock,inp);
	g_string_free(inp,TRUE);
	if (lret!=0)
	{
		disp_cnx_hshake_info(ch);
		XFIO_SET_CLOSE_STATUS(ch->xfio,CNX_CLOSED);	/* immediat termination */
		return;
	}

	/* CAPA sending */
	//send_const_str_to_xfio(ch->xfio,"$Capabilities UTF8$TAG|");
	send_const_str_to_xfio(ch->xfio,"$Capabilities UniSearch$XSearch$MD4x|");

	ch->log_step=STEP_KEY_VALID;
}

/***************************************************/
/* the nick is valid but it may require a password */
/***************************************************/
/* if a ghost exists, its luce is in ghost  */
/* else, it is NULL                         */
/* next steps:                              */
/* on passwd_protect_nick, STEP_WANT_PASSWD */
/* on unprotected_nick, STEP_HANDSHAKE_DONE */
/*   (or if ghost exist, stay on this step) */
/*             else CNX_CLOSED or           */
/*                  CNX_WILL_DIE is set     */
/********************************************/
static void perform_password_checking(CNX_HSHAKE *ch, LOCAL_USER_CNX_ENTRY *ghost, int only_reg)
{
	/* check if this user is a registered user (password protected) */
	ch->temp_passwd=get_user_password(ch->temp_nick,&(ch->temp_privilege),&(ch->temp_ext_flag));

	/* check if too many users on the hub */
	if(only_reg==0)
	{
		if((ch->temp_passwd==NULL)||(ch->temp_privilege==0)) /* only normal users (registered or not) are limited in number */
		{
#if 0
			if(ghost)
			{
				gchar *validedenide;
				validedenide=g_strconcat("$ValidateDenide ",ch->temp_nick,"|",NULL);
				send_const_str_to_xfio(ch->xfio,validedenide);
				g_free(validedenide);
				return;	  /* stay on this step */
			}
#endif
	
			if (too_many_users(ch->xfio))
			{
				disp_cnx_hshake_info(ch);
				XFIO_SET_CLOSE_STATUS(ch->xfio,CNX_WILL_DIE);	/* send the reply before leaving */
				return;
			}
		}
	}

	if((only_reg != 0) && (ch->temp_passwd==NULL))
	{
		/* only reg users can connect */
		printf("Unregistered user: %s (ONLY_REG HUB)\n",ch->temp_nick);
		send_const_str_to_xfio(ch->xfio,"<Hub-Security> This Hub is for registered users only.|");
		if (only_reg == 2)
			try_auto_redir(ch->xfio);
		disp_cnx_hshake_info(ch);
		XFIO_SET_CLOSE_STATUS(ch->xfio,CNX_WILL_DIE);	/* send the reply before leaving */
		return;
	}

	if (ch->temp_passwd != NULL)
	{	/* this is a password protected user */
		if( (strlen(ch->temp_passwd) > 0) || (ch->temp_privilege> 0) )
		{ /* OP and MASTER must enter password */
			/* send password query */
			send_const_str_to_xfio(ch->xfio,"<Hub-Security> Your name is registered, please supply the password.|");
			send_const_str_to_xfio(ch->xfio,"$GetPass|");
			ch->log_step=STEP_WANT_PASSWD;
			return;
		}
	}

	/* for all the others, the handshake is over */
	ch->log_step=STEP_HANDSHAKE_DONE;
}

/**********************************************/
/* $Key is valid, we wait for a $ValidateNick */
/**********************************************/
/* next steps: on busy_nick, stay here      */
/* on passwd_protect_nick, STEP_WANT_PASSWD */
/* on unprotected_nick, STEP_HANDSHAKE_DONE */
/*             else CNX_CLOSED or           */
/*                  CNX_WILL_DIE is set     */
/********************************************/
static void sf_step_key_valid(CNX_HSHAKE *ch)
{
	char *etn;
	GString *inp;
	LOCAL_USER_CNX_ENTRY *ghost_luce;
	int only_reg;

	inp=xfio_take_first_string_of_glist_incoming(ch->xfio);
	if(inp==NULL)
	{
		XFIO_SET_CLOSE_STATUS(ch->xfio,CNX_CLOSED);	/* immediat termination */
		return;
	}

	if (strncmp(inp->str,"$ValidateNick ",sizeof("$ValidateNick ")-1))
	{
		printf("$ValidateNick expected but (%s) found\n", inp->str);
		g_string_free(inp,TRUE);
		disp_cnx_hshake_info(ch);
		XFIO_SET_CLOSE_STATUS(ch->xfio,CNX_CLOSED);	/* immediat termination */
		return;
	}

	ch->temp_nick=strdup(inp->str+strlen("$ValidateNick "));
	g_string_free(inp,TRUE);	/* inp is no more useful */

	/* theoretically, ' ' and '$' are not allowed in string */
	/* the  character is needed by the regexp lock */
	/* username with " can not be kick with the Hub-security*/
	/* dc protocol do not allow username longer than 47 char */
	if( (strpbrk(ch->temp_nick," $\"")!=NULL) ||
	    (strlen(ch->temp_nick) > 47) )
	{
		printf("' ' or '$' or '' in nick.\n");
		disp_cnx_hshake_info(ch);
		XFIO_SET_CLOSE_STATUS(ch->xfio,CNX_CLOSED);	/* immediat termination */
		return;
	}

	etn=strchr(ch->temp_nick,'|');
	if (etn != NULL)
		*etn='\0';

	/* check if the nick is not banned */
	{
		char	*dummy1;
		int	dummy_len;

		if( (strstr(ch->temp_nick,"www.glosearch.tk")) ||
		    (get_tos_entry(TBAN_TOSKEY,ch->temp_nick,strlen(ch->temp_nick),1,&dummy1,&dummy_len)==1) ||
		    (get_tos_entry(PBAN_TOSKEY,ch->temp_nick,strlen(ch->temp_nick),1,&dummy1,&dummy_len)==1) )
		{
			printf("Connection attempt from a banned nick: %s\n",ch->temp_nick);
			send_const_str_to_xfio(ch->xfio,"<Hub-Security> Your Nick is banned!!!|");
			disp_cnx_hshake_info(ch);
			XFIO_SET_CLOSE_STATUS(ch->xfio,CNX_WILL_DIE);	/* send the reply before leaving */
			return;
		}
	}

	/* check if the nick is not already used on the hub */
	/* first, check with some reserved or well known fake user */
	if ((!strcasecmp("Hub-Security",ch->temp_nick)) || 
	    (!strcasecmp("Hub-Mass-Message",ch->temp_nick)) ||
	    is_ip(ch->temp_nick))
	{
		gchar *validedenide;

		validedenide=g_strconcat("$ValidateDenide ",ch->temp_nick,"|",NULL);
		send_const_str_to_xfio(ch->xfio,validedenide);

		g_free(validedenide);
		return;		/* stay on this step */
	}

	/* check if hub is public or not */
	G_LOCK(conf_file);
	if(!e_db_int_get(conf_file,"ONLY_REG",&only_reg))
		only_reg=0;
	G_UNLOCK(conf_file);

	/* check against a ghost user (or another user/chat with the same name) */
	ghost_luce=user_cnx_entry_get_by_nickname(ch->temp_nick);
	if(ghost_luce!=NULL)
	{	
		/* 2 users with the same nickname ? */
		if((ghost_luce->user_xfio->user_ip.s_addr==ch->xfio->user_ip.s_addr))
		{
			/* and on the same IP, close the other one, probably a ghost */
			XFIO_SET_CLOSE_STATUS(ghost_luce->user_xfio,CNX_CLOSED);
		}
		else
		{
			if((ghost_luce->privilege==0)&&(only_reg==0))
			{
				/* not register only and not privilege user but perhaps it is a password protected normal account */
				char *ghost_luce_passwd;
				int dummy;

				ghost_luce_passwd=get_user_password(ch->temp_nick,&dummy,&dummy);
			
				if(ghost_luce_passwd==NULL)
				{
					/* not luck for you cowboy, it is not a password protected normal account */
					gchar *validedenide;
					validedenide=g_strconcat("$ValidateDenide ",ch->temp_nick,"|",NULL);
					send_const_str_to_xfio(ch->xfio,validedenide);
					g_free(validedenide);
					return;		/* stay on this step */
				}

				free(ghost_luce_passwd);
			}
		}
	}

	/* close anyother connection attempt with the same nickname */
	{
		CNX_HSHAKE *ghost_ch;

		ghost_ch=get_cnx_hshake_by_nickname(ch->temp_nick,ch);
		while(ghost_ch!=NULL)
		{
			shutdown(ghost_ch->xfio->sock_fd,SHUT_RD);
			XFIO_SET_CLOSE_STATUS(ghost_ch->xfio,CNX_CLOSED);

			/* because only entry without CNX_CLOSED status is returned, we can retry without hanging */
			ghost_ch=get_cnx_hshake_by_nickname(ch->temp_nick,ch);
		}
	}

	perform_password_checking(ch, ghost_luce, only_reg);
	return;
}
	
/****************************************/
/* Nick is valid, we wait for a $MyPass */
/********************************************/
/* next steps:                              */
/* on valid password, STEP_HANDSHAKE_DONE   */
/*             else CNX_CLOSED or           */
/*                  CNX_WILL_DIE is set     */
/********************************************/
static void sf_step_want_passwd(CNX_HSHAKE *ch)
{
	GString *inp;
	GString *okpass;
	LOCAL_USER_CNX_ENTRY *ghost_luce;

	inp=xfio_take_first_string_of_glist_incoming(ch->xfio);
	if(inp==NULL)
	{
		XFIO_SET_CLOSE_STATUS(ch->xfio,CNX_CLOSED);	/* immediat termination */
		return;
	}

	if (strncmp(inp->str,"$MyPass ",sizeof("$MyPass ")-1))
	{
		printf("no $MyPass returned on $GetPass (%s obtained)\n",inp->str);
		g_string_free(inp,TRUE);
		disp_cnx_hshake_info(ch);
		XFIO_SET_CLOSE_STATUS(ch->xfio,CNX_CLOSED);	/* immediat termination */
		return;
	}

	/* remove trash before and after the password */
	inp=g_string_erase(inp,0,sizeof("$MyPass ")-1);
	inp = g_string_truncate(inp, ((inp->len -1) > 0) ? (inp->len -1): 0);

	if(strcmp(inp->str, ch->temp_passwd))
	{
		printf("Invalid password (%s) for %s. Wanted: %s\n",
	 				inp->str,ch->temp_nick,ch->temp_passwd);
		send_const_str_to_xfio(ch->xfio,"<Hub-Security> Incorrect password.|$BadPass|");
		g_string_free(inp,TRUE);
		disp_cnx_hshake_info(ch);
		XFIO_SET_CLOSE_STATUS(ch->xfio,CNX_WILL_DIE); /* send the reply before leaving */
		return;
	}

#if 0
	if (gl_gui_mode == 0)
	{
		unsigned int hip;

		hip=ntohl(ptr->ip.s_addr);
		printf("valid_pass %s for %s %hhu.%hhu.%hhu.%hhu\n", 
			 		ptr->login_data->user_pass,
			 		ptr->login_data->temp_nick,
			 		(unsigned char)(hip>>24) & 0xff,
			 		(unsigned char)(hip>>16) & 0xff,
			 		(unsigned char)(hip>>8) & 0xff,
			 		(unsigned char) hip & 0xff);
	}
#endif
	/* close the connection of the user using the same account if exists */
	ghost_luce=user_cnx_entry_get_by_nickname(ch->temp_nick);
	if(ghost_luce!=NULL)
	{
		printf("ghost found. Shutting down its socket.\n");
		XFIO_SET_CLOSE_STATUS(ghost_luce->user_xfio,CNX_CLOSED); /* immediate termination */
		rename_luce_for_deletion(ghost_luce);
	}

	/* freeing unuseful string */
	g_string_free(inp,TRUE);
	free(ch->temp_passwd);
	ch->temp_passwd=NULL;

	/* ok, the password is valid */
	okpass=g_string_new("<Hub-Security> Logged in.|$LogedIn ");
	okpass=g_string_append(okpass,ch->temp_nick);
	okpass=g_string_append_c(okpass,'|');

	send_stolen_gstr_to_xfio(ch->xfio,okpass);		/* don't free okpass */

	ch->log_step=STEP_HANDSHAKE_DONE;
}


/* -------------------------------------------------------------------------- */
/*************************************/
/* all functions of the step handler */
/************************************************************************************/
/* on problem, the xfio status must be changed to CNX_CLOSED (immediat termination) */
/* or to CNX_WILL_DIE (delay termination until output buffer is flushed or timeout) */
/************************************************************************************/

#define STEP_FLAGS_NONE 0
#define STEP_FLAGS_LINK 1
typedef struct 
{
	void (*fnc)(CNX_HSHAKE *);
	int step_flags;
} STEP_FUNCTIONS;

static STEP_FUNCTIONS step_function[]=
				{
					{sf_step_lock_sent, STEP_FLAGS_NONE},		/* STEP_LOCK_SENT function */
					{sf_step_key_valid, STEP_FLAGS_NONE},		/* STEP_KEY_VALID function */
					{sf_step_want_passwd, STEP_FLAGS_NONE},	/* STEP_WANT_PASSWD function */
					{NULL, STEP_FLAGS_NONE},						/* final step */
				};

/*******************************/
/* perform the following steps */
/*******************************/
static void do_steps(CNX_HSHAKE *ch)
{
	int flags;
	int cur_step;

	do
	{
		cur_step=ch->log_step;

		flags=step_function[cur_step].step_flags;
		(step_function[ch->log_step].fnc)(ch);
	}while((ch->xfio->closed==CNX_OPENED)&&
	       (cur_step!=ch->log_step)&&			/* prevent eternal loop on a stage without I/O */
	       (flags==STEP_FLAGS_LINK));
}
	
/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */
/* ----------------------------- GED handlers ------------------------------- */
/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */

/**************************/
/* function called at end */
/**************************/
static int ged_cnx_hshake_exit(const GED_CALLBACKS *ged)
{
	return 0;
}

/********************************/
/* function called periodically */
/********************************/
static void ged_cnx_hshake_periodic(const struct ged_callbacks *ged)
{
	int i;
	CNX_HSHAKE *ch;

	/* expire connection opened for a too long time */
	/* reverse loop to handle entry destruction without tips */
	for(i=cnx_in_handshake->len-1;i>=0;i--)
	{
		ch=&(g_array_index(cnx_in_handshake,CNX_HSHAKE,i));

		if((gl_cur_time - ch->xfio->start_time)> MAX_CNX_HANDSHAKE_TIME)
		{	/* no need to check for closed connection, they are managed in the always1 function */
			cnx_handshake_destroy_cnx_hshake(i,FALSE);
		}
	}
}

/********************************/
/* function called at each loop */
/********************************/
static void ged_cnx_hshake_always1(const struct ged_callbacks *ged)
{
	int i;
	CNX_HSHAKE *ch;

	/* expire connection opened for a too long time */
	/* reverse loop to handle entry destruction without tips */
	for(i=cnx_in_handshake->len-1;i>=0;i--)
	{
		ch=&(g_array_index(cnx_in_handshake,CNX_HSHAKE,i));
#ifdef DEBUG
		printf("ch: %p  step: %d\n",ch, (int)(ch->log_step));
#endif
		if(XFIO_IS_CLOSED(ch->xfio))
		{
			cnx_handshake_destroy_cnx_hshake(i,FALSE);
			continue;
		}

		if(XFIO_INPUT_AVAILABLE(ch->xfio))
		{
			do_steps(ch);

			if(XFIO_IS_CLOSED(ch->xfio))
			{	/* maybe the connection is dead */
				cnx_handshake_destroy_cnx_hshake(i,FALSE);
				continue;
			}
		}
	}

	/* convert successful handshake into full local user entry */
	for(i=cnx_in_handshake->len-1;i>=0;i--)
	{
		ch=&(g_array_index(cnx_in_handshake,CNX_HSHAKE,i));
		if(ch->log_step==STEP_HANDSHAKE_DONE)
		{
			cnx_handshake_convert_cnx_hshake_to_local_user_cnx_entry(ch,i);
		}
	}
}

/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */
/* ---------------------- initialization of the handler --------------------- */
/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */
static GED_CALLBACKS cnx_hshake_ged=
				{
					"connection handshake GED", /* ged_name For debug */
					ged_cnx_hshake_exit, /* destructor */
               NULL,
               NULL,
               ged_cnx_hshake_periodic,
               ged_cnx_hshake_always1,
               NULL,
					NULL
				};

/***********************************************/
/* function initializing the handshake handler */
/***********************************************/
GED_CALLBACKS *cnx_handshake_init(void)
{
	cnx_in_handshake=g_array_new(FALSE,FALSE,sizeof(CNX_HSHAKE));
	return &cnx_hshake_ged;
}

