/* DChub - a Direct Connect hub clone for Linux
 * Copyright (C) 2001 Eric Prevoteau
 *
 * extprog_toolkit.c: Copyright (C) Eric Prevoteau <www@ac2i.tzo.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
/*
$Id: extprog_toolkit.c,v 2.2 2003/05/05 16:06:58 ericprev Exp $
*/

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <glib.h>

#include "extprog_toolkit.h"
#include "dc_com.h"
#include "macro.h"

void  str_to_llunsigned(char  *str, long long unsigned *lu)
{
   *lu = 0;
   conv_loop:
   *lu += *str - '0';
   ++str;
   if (*str)
   {
      *lu*=10;
      goto conv_loop;
   }
}

char  *llunsigned_to_str(long long unsigned a)
{
   static char out[32];
   int      i=30;
   out[31] = '\0';

   while (a > 0)
   {
      out[i--] = a%10 + '0';
      a /= 10;
   }
   out[i] = a%10 + '0';
   return (out+i+1);
}

/*********************************************************************/
/* setup our info on hub. the function builds a string having the    */
/* following format and sends it on the given socket.                */
/* string format:  $MyINFO $ALL nickname aaa$ $xxxf$bbb$yyy$         */
/*       aaa is user description                                     */
/*			xxx is connection type (ex: Cable)                          */
/*       f is a 1 byte flag. Default value: \x01                     */
/*             if not behind firewall, bit 1 must be set, else clear */
/*			bbb is e-mail (empty string)                                */
/*			yyy is size of shared data in bytes                         */
/****************************************************************************/
/* input: sck: socket to send the DC string                                 */
/*        nick: your nickname (you cannot change this after the connection, */
/*              it is a protocol limitation                                 */
/*        description: your description on the hub (can be NULL)            */
/*        cnx: your connection type (should be in the following list:       */
/*             "56Kbps", "33.6Kbps", "28.8Kbps", "Satellite", "ISDN", "DSL",*/ 
/*             "Cable", "LAN(T1)", "LAN(T3)")                               */
/*        cnx_opt: this value is a bit field                                */
/*                 bit 0 always set.                                        */
/*                 bit 1   ==0, user here, ==1, user away                   */
/*                 bit 2   ==0, normal, ==1, is server                      */
/*                         according to DC, a server is a client online     */
/*                         for more than 2 hours.                           */
/*                 bit 3   ==0, normal, ==1, fast upload                    */
/*                         according to DC, a fast upload is a client       */
/*                         having an upload speed greater than 100KB/s.     */
/*        mail: your e-mail on the hub (can be NULL)                        */
/*        amount: size of the share size in bytes.                          */
/****************************************************************************/
void set_user_info(int sck, const char *nick, char *description, char *cnx,char cnx_opt,char *mail, long long unsigned amount)
{
	GString *user_info;

	user_info=g_string_new("$MyINFO $ALL ");
	g_string_sprintfa(user_info,"%s %s$ $%s%c$%s$%s$",
										nick,
										(description!=NULL)?description:"",
										cnx,
										cnx_opt|1,					/* to be sure the bit 0 is always set */
										(mail!=NULL)?mail:"",
										llunsigned_to_str(amount));
	send_dc_line(sck,user_info->str,NULL);
	g_string_free(user_info,TRUE);
}

void send_commandfile(int sck, const char *filename, const char *source_nick, const char *dest_nick)
{
	FILE *testfile;
	long lSize=0;
	char *testtxt;
	char filepath[80];
	strcpy(filepath,"/usr/local/dchub/etc/textfiles/");

	printf("filepath1: %s\n",filepath);
	strcat(filepath,filename);
	printf("filepath2: %s\n",filepath);
	testfile=fopen(filepath,"rt");
	if (testfile==NULL)
		perror ("Error opening file\n");
	else
   {
		fseek(testfile,0,SEEK_END);
		lSize=ftell(testfile);
		/* while (!feof(testfile))
		{
			fgetc(testfile);
			lSize++;
		} */
		rewind (testfile);

		/* lSize--; */

		testtxt=(char*) malloc (lSize);
		if (testtxt==NULL)
			perror ("not enought memory\r\n");
		else
		{
			fread (testtxt,1,lSize,testfile);
			testtxt[lSize]='\0';
			send_private_chat_message(sck,(char *)source_nick,(char *)dest_nick,testtxt,1);
		}
		fclose(testfile);
		free(testtxt);
	}
}

/*****************************************/
/* send a message on the hub public chat */
/*******************************************************************************************/
/* input: sck: socket to send the DC string                                                */
/*        nick: your nickname                                                              */
/*        message: the message to send. Remember you cannot use the characters '$' and '|' */
/*                 Also remember that newline is in DOS format ("\r\n") and not Unix ("\n")*/
/*******************************************************************************************/
void send_public_chat_message(int sck, char *nick, char *message)
{
	GString *output;

	output=g_string_sized_new(512);
	g_string_sprintf(output,"<%s> %s",nick,message);

	send_dc_line(sck,output->str,NULL);
	g_string_free(output,TRUE);
}

/*****************************************/
/* send a message on the hub public chat */
/*******************************************************************************************/
/* input: sck: socket to send the DC string                                                */
/*        nick: your nickname                                                              */
/*        message: the message to send. Remember you cannot use the characters '$' and '|' */
/*                 Also remember that newline is in DOS format ("\r\n") and not Unix ("\n")*/
/*******************************************************************************************/
void send_private_gchat_message(int sck, char *dest_nick, char *source_nick, char *message)
{
   GString *output;

   output=g_string_sized_new(512);
   g_string_sprintf(output,"$ForceSendTo %s <%s> %s",dest_nick,source_nick,message);

   send_dc_line(sck,output->str,NULL);
   g_string_free(output,TRUE);
}

/************************************/
/* send a message on a private chat */
/*******************************************************************************************/
/* input: sck: socket to send the DC string                                                */
/*        nick: your nickname                                                              */
/*        destnick: destination nickname                                                   */
/*        message: the message to send. Remember you cannot use the characters '$' and '|' */
/*                 Also remember that newline is in DOS format ("\r\n") and not Unix ("\n")*/
/*                 for a normal private chat, the message should start with "<yournick> "  */
/*        auto_prefix: if set, the function will automatically prefix the "<yournick> "    */
/*                     to your message. If not set, your message is used as is.            */
/*******************************************************************************************/
void send_private_chat_message(int sck, char *nick, char *destnick,  char *message, int autoprefix)
{
	GString *output;

	output=g_string_sized_new(512);
	g_string_sprintf(output,"$To: %s From: %s $",destnick,nick);
	if(autoprefix)
	{
		g_string_sprintfa(output,"<%s> ",nick);
	}
	output=g_string_append(output,message);

	send_dc_line(sck,output->str,NULL);
	g_string_free(output,TRUE);
}

/**************************************/
/* kick the given user out of the hub */
/****************************************************/
/* input: sck: socket to send the DC string         */
/*        nick: your nickname                       */
/*        kicknick: nickname of the user to kick    */
/*        reason: a short message giving the reason */
/*                NOTE: the message cannot contains */
/*                " characters                      */
/****************************************************/
void kick_user(int sck, const char *nick, const char *kicknick, const char *reason)
{
	GString *output;
	output=g_string_sized_new(512);
	g_string_sprintf(output,"$To: Hub-Security From: %s $<%s> -kick %s \"%s\"",nick,nick,kicknick,reason);
	
	send_dc_line(sck,output->str,NULL);
	g_string_free(output,TRUE);
}

/***********************************/
/* Ban the given user from the hub */
/****************************************************/
/* input: sck: socket to send the DC string         */
/*        nick: your nickname                       */
/*        kicknick: nickname of the user to ban     */
/*        reason: a short message giving the reason */
/*                NOTE: the message cannot contains */
/*                " characters                      */
/****************************************************/
void ban_user(int sck, const char *nick, const char *kicknick, const char *reason)
{
	GString *output;
	output=g_string_sized_new(512);
	g_string_sprintf(output,"$To: Hub-Security From: %s $<%s> -pbankick %s \"%s\"",nick,nick,kicknick,reason);

	send_dc_line(sck,output->str,NULL);
	g_string_free(output,TRUE);
}

/******************************************************************************/
/* perform a search on the hub. The function builds and sends a string having */
/* the following format:                                                      */
/* "$Search Hub:nick a?b?c?d?eeeee| where                                     */
/* a is F if size doesn't matter, else T                                      */
/* b is F if size is "at least", else T (at most)                             */
/* c is the size in byte                                                      */
/* d is data type: 1=any,2=audio,3=compressed,4=document,5=exe,6=picture,     */
/*                 7=videos,8=folder                                          */
/* and eeee is the pattern to find                                            */
/* Hub:nick is replaced by hostip:hostport if the search is performed in      */
/* active mode (this requires the client to create a socket to receive result)*/
/*******************************************************************************************/
/* input: sck: socket to send the DC string                                                */
/*        mode: 0=ACTIVE, !=0=PASSIVE                                                      */
/*        dest: if mode is active, it is a string "hostip:hostport" (==address to reply)   */
/*              if mode is passive, it is your nickname                                    */
/*        size_matter: 0= size has no meaning. !=0, size must be taken into account        */
/*        size_at_most: (only if size_matter is set). 0= size is at least the given one    */
/*                      !=0, size is at most the given one                                 */
/*        wanted_size: (only if size_matter is set). size (in byte) used with size_at_most */
/*        wanted_type: see 'd' value in the beginning of the description.                  */
/*        pattern: pattern to search.                                                      */
/*******************************************************************************************/
void do_search(int sck, int mode, char *dest, int size_matter, int size_at_most, unsigned long long wanted_size, unsigned int wanted_type, char *pattern)
{
	GString *output;

	output=g_string_new("$Search ");
	if(mode==0)		
	{	/* active mode */
		output=g_string_append(output,dest);
	}
	else
	{	/* passive mode */
		g_string_sprintfa(output,"Hub:%s",dest);
	}

	if(size_matter==0)
	{	/* size must be ignored */
		output=g_string_append(output," F?F?0");
	}
	else
	{
		g_string_sprintfa(output," T?%c?%s",
											(size_at_most==0)?'F':'T',
											llunsigned_to_str(wanted_size));
	}

	g_string_sprintfa(output,"?%u?%s",wanted_type,pattern);

	send_dc_line(sck,output->str,NULL);
	g_string_free(output,TRUE);
}

void do_uni_search(int sck, int mode, const char *dest, int size_matter, int size_at_most, unsigned long long wanted_size, unsigned int wanted_type, const char *pattern, const char* dest_sr)
{
	GString *output;

	output=g_string_new("$UniSearch ");
	output=g_string_append(output, dest_sr);
	output=g_string_append(output, " ");
	if(mode==0)		
	{	/* active mode */
		output=g_string_append(output,dest);
	}
	else
	{	/* passive mode */
		g_string_sprintfa(output,"Hub:%s",dest);
	}

	if(size_matter==0)
	{	/* size must be ignored */
		output=g_string_append(output," F?F?0");
	}
	else
	{
		g_string_sprintfa(output," T?%c?%s", (size_at_most==0)?'F':'T',	llunsigned_to_str(wanted_size));
	}
	g_string_sprintfa(output,"?%u?%s",wanted_type,pattern);
 fprintf(stderr, "[Search]-[ %s ]\n", output->str);
	send_dc_line(sck, output->str, NULL);
	g_string_free(output,TRUE);
}
/******************************************/
/* redirect the given user to another hub */
/******************************************************/
/* input: sck: socket to send the DC string           */
/*        nick: your nickname                         */
/*        redirnick: nickname of the user to redirect */
/*        address: hub address where the user must be */
/*                 redirected.                        */
/******************************************************/
void redir_user(int sck, char *nick, char *redirnick, char *address)
{
	GString *output;
	output=g_string_sized_new(512);
	g_string_sprintf(output,"$To: Hub-Security From: %s $<%s> -redir %s %s",nick,nick,redirnick,address);
	
	send_dc_line(sck,output->str,NULL);
	g_string_free(output,TRUE);
}


/*********************************************************************************/
/* the following function provides a generic message decoder. Known messages are */
/* predecoded into an easiest format to use.                                     */
/*********************************************************************************/
int generic_decoder(GString *input, TBLDECODE *fncs, void *xtra_param)
{
	int ret;
	GString *str;
	char *t;

	if((input==NULL)||(fncs==NULL))
		return -1;

	if((input->len<1)||(input->str[input->len-1]!='|'))
		return -2;

#define STRNCMP(x,y)               (strncmp((x),(y),(strlen(y))))

	/* the following pieces of code more or less come from dchub or DCTC */

	if(!STRNCMP(input->str,"$MyINFO $ALL "))
	{	/* MyINFO handler */
		if(fncs->myinfo_fnc==NULL)
			return 0;		/* no function to call, why wasting time to decode */

		str=g_string_new(input->str+strlen("$MyINFO $ALL "));
		ret=-3;
		/*********************************************************************/
		/* string format:  $MyINFO $ALL nickname aaa$ $xxxf$bbb$yyy$         */
		/*       aaa is user description                                     */
		/*       xxx is connection type (ex: Cable)                          */
		/*       f is a 1 byte flag. Default value: \x01                     */
		/*             if not behind firewall, bit 1 must be set, else clear */
		/*       bbb is e-mail (empty string)                                */
		/*       yyy is size of shared data in bytes                         */
		/*********************************************************************/
		{
			char *fnd_nick;
			char *fnd_desc;
			char *fnd_cnx_type;
			unsigned char fnd_flag;
			char *fnd_email;
			char *fnd_size;
			unsigned long long shared_size;

			t=str->str;
			SKIP_SPACE(t);

			/* extract the nickname */
			fnd_nick=t;
			t=strchr(fnd_nick,' ');
			if(t==NULL)
				goto have_err_my_info;
			*t++='\0';
			SKIP_SPACE(t);

			/* extract the description */
			fnd_desc=t;
			t=strchr(fnd_desc,'$');
			if(t==NULL)
				goto have_err_my_info;
			*t++='\0';

			if(strncmp(t," $",2))
				goto have_err_my_info;
			t+=2;

			/* extract the connection type */
			fnd_cnx_type=t;
			t=strchr(fnd_cnx_type,'$');
			if(t==NULL)
				goto have_err_my_info;
			*t++='\0';

			/* extract the email */
			fnd_email=t;
			t=strchr(fnd_email,'$');
			if(t==NULL)
				goto have_err_my_info;
			*t++='\0';

			/* extract the shared size */
			fnd_size=t;
			t=strchr(fnd_size,'$');
			if(t==NULL)
				goto have_err_my_info;
			*t++='\0';
		
			/* well, nearly all fields have been extracted */
			/* only fnd_flag still is merged with fnd_cnx_type */
			if(strlen(fnd_cnx_type)>1)
			{
				fnd_flag=fnd_cnx_type[strlen(fnd_cnx_type)-1];
				fnd_cnx_type[strlen(fnd_cnx_type)-1]='\0';
			}
			else
				goto have_err_my_info;
			str_to_llunsigned(fnd_size, &shared_size);
			ret=(fncs->myinfo_fnc)(xtra_param,fnd_nick,fnd_desc,fnd_cnx_type,fnd_flag,fnd_email,shared_size);
		}

		have_err_my_info:
		g_string_free(str,TRUE);
		return ret;
	}
	else if(!STRNCMP(input->str,"<"))
	{	/* MyINFO handler */
		if(fncs->publicmsg_fnc==NULL)
			return 0;		/* no function to call, why wasting time to decode */

		/* input format: "<nickname> message|" */
		str=g_string_new(input->str+strlen("<"));
		str=g_string_truncate(str,str->len-1);			/* remove trailing | */
		ret=-3;

		t=strstr(str->str,"> ");
		if(t==NULL)
			goto have_err_gchat;

		*t='\0';

		ret=(fncs->publicmsg_fnc)(xtra_param,str->str,t+2   /* skip "> " */ );
		
		have_err_gchat:
		g_string_free(str,TRUE);
		return ret;
	}
	else if(!STRNCMP(input->str,"$To: "))
	{	/* To handler */
		if(fncs->privmsg_fnc==NULL)
			return 0;		/* no function to call, why wasting time to decode */

		/* input format: "$To: destnick From: sourcenick $message|" */
		str=g_string_new(input->str+strlen("$To: "));
		str=g_string_truncate(str,str->len-1);			/* remove trailing | */
		ret=-3;
		{
			char *dest_nick;
			char *source_nick;
			char *t;
	
			/* extract the dest nickname and the source nickname */
			t=str->str;
			SKIP_SPACE(t);
			dest_nick=t;
			t=strchr(dest_nick,' ');
			if(t==NULL)
				goto have_err_to;
			*t++='\0';
			if(strncmp(t,"From: ",strlen("From: ")))
				goto have_err_to;
			t+=strlen("From: ");
			SKIP_SPACE(t);
			source_nick=t;
			t=strchr(source_nick,' ');
			if(t==NULL)
				goto have_err_to;
			*t++='\0';
			if(*t!='$')
				goto have_err_to;
			t++;

			ret=(fncs->privmsg_fnc)(xtra_param,source_nick,dest_nick,t);
		}
		have_err_to:
		g_string_free(str,TRUE);
		return ret;
	}
	else if(!STRNCMP(input->str,"$Search "))
	{	/* Search handler */
		if(fncs->search_fnc==NULL)
			return 0;		/* no function to call, why wasting time to decode */

		/* input format: "$Search Hub:nick a?b?c?d?eeeee|" (passive search) */
		/*    or format: "$Search ip:port a?b?c?d?eeeee|" (active search) */
		str=g_string_new(input->str+strlen("$Search: "));
		str=g_string_truncate(str,str->len-1);			/* remove trailing | */
		ret=-3;

		{
			char *nick;
			int mode,size_matter,size_at_most;
			unsigned long long wanted_size;
			unsigned int wanted_type;

			t=str->str;
			if(!strncmp(t,"Hub:",4))
			{
				mode=1;		/* passive search */
				nick=t+strlen("Hub:");
			}
			else
			{
				mode=0;		/* active size */
				nick=t;
			}

			t=strchr(t,' ');
			if(t==NULL)
				goto have_err_search;
			t++;
			SKIP_SPACE(t);

			/* size matter flag */
			if(*t=='F') size_matter=FALSE;
			else if(*t=='T') size_matter=TRUE;
			else goto have_err_search;
			t=strchr(t,'?');
			if(t==NULL) goto have_err_search;
			t++;
			
			/* size at most flag */
			if(*t=='F') size_at_most=FALSE;
			else if(*t=='T') size_at_most=TRUE;
			else goto have_err_search;
			t=strchr(t,'?');
			if(t==NULL) goto have_err_search;
			t++;

			/* wanted size */
			str_to_llunsigned(t, &wanted_size);
			
			t=strchr(t,'?');
			if(t==NULL) goto have_err_search;
			t++;

			/* size matter flag */
			sscanf(t,"%u",&wanted_type);
			t=strchr(t,'?');
			if(t==NULL) goto have_err_search;
			t++;

			ret=(fncs->search_fnc)(xtra_param,nick,mode,size_matter,size_at_most,wanted_size,wanted_type,t);
		}
		have_err_search:
		g_string_free(str,TRUE);
		return ret;
	}
	else 	if(!STRNCMP(input->str,"$SR "))
	{	/* SR handler */
		if(fncs->sr_fnc==NULL)
			return 0;		/* no function to call, why wasting time to decode */

		/* string format:  $SR nickname filename\005filesize slot/ratio\005hubname (hub addr)| */
		/* or if folder:	 $SR nickname filename slot/ratio\005hubname (hub addr)| */
		str=g_string_new(input->str+strlen("$SR "));
		str=g_string_truncate(str,str->len-1);			/* remove trailing | */
		ret=-3;

		{
			char *nick, *fname, *fsize, *rest;
			unsigned long long filesize=0;
			unsigned int free_slot, ttl_slot;
			t=str->str;
			SKIP_SPACE(t);

			/* the nick should not contain any spaces */
			/* so devide at the first space */

			nick=t;
			t=strchr(t,' ');
			if(t==NULL)
				goto have_err_sr;
			*t++='\0';

			/* we can't decode the filename without knowing if we've got just a folder */
			/* so now lets check if we've got a folder first */
			/* if it's a file we've got the '\005' twice */

			fname=t;
			t=strchr(t,5);
			if(t==NULL)
				goto have_err_sr;
			*t++='\0';

			rest=t;
			t=strchr(t,5);
			if(t!=NULL)
			{
				/* gotcha - found the second key - this should be a filename */
				/* 'rest' should now include filesize and slot-ratio */
				*t++='\0';
				fsize=rest;
				rest=strchr(rest,' ');
				if(rest==NULL)
					goto have_err_sr;
				*rest++='\0';
				str_to_llunsigned(fsize, &filesize);
				if(sscanf(rest,"%u/%u",&free_slot,&ttl_slot)!=2)
					goto have_err_sr;
			}
			else
			{
				/* '/005' just found once - this seems to be a folder */
				/* 'fname' includes the slot-ratio :(  and 'rest' the hubname*/
				/* we don't now if the filename includes any spaces - so split at the last space */
				t=fname;
				fname=t;
				t=strrchr(t,' ');
				if(t==NULL)
					goto have_err_sr;
				*t++='\0';
				if(sscanf(t,"%u/%u",&free_slot,&ttl_slot)!=2)
					goto have_err_sr;
			}
			ret=(fncs->sr_fnc)(xtra_param,nick,fname,filesize,free_slot,ttl_slot);
		}
		have_err_sr:
		g_string_free(str,TRUE);
		return ret;
	}
	else if(!STRNCMP(input->str,"$Version "))
	{	/* Version handler */
		if(fncs->version_fnc==NULL)
			return 0;		/* no function to call, why wasting time to decode */

		str=g_string_new(input->str+strlen("$Version "));
		str=g_string_truncate(str,str->len-1);
		ret=(fncs->version_fnc)(xtra_param,str->str);
		g_string_free(str,TRUE);
		return ret;
	}
	else	if(!STRNCMP(input->str,"$Hello "))
	{	/* Version handler */
		if(fncs->hello_fnc==NULL)
			return 0;		/* no function to call, why wasting time to decode */

		str=g_string_new(input->str+strlen("$Hello "));
		str=g_string_truncate(str,str->len-1);			/* remote the trailing | */
		ret=(fncs->hello_fnc)(xtra_param,str->str);
		g_string_free(str,TRUE);
		return ret;
	}
	else	if(!STRNCMP(input->str,"$Quit "))
	{	/* Version handler */
		if(fncs->quit_fnc==NULL)
			return 0;		/* no function to call, why wasting time to decode */

		str=g_string_new(input->str+strlen("$Quit "));
		str=g_string_truncate(str,str->len-1);			/* remote the trailing | */
		ret=(fncs->quit_fnc)(xtra_param,str->str);
		g_string_free(str,TRUE);
		return ret;
	}
	else    if(!STRNCMP(input->str,"$NickList "))
	{       /* NickList handler */
		if(fncs->nicklist_fnc==NULL)
			return 0;	/* no function to call, why wasting time to decode */

		str=g_string_new(input->str+strlen("$NickList "));
		str=g_string_truncate(str,str->len-1);	/* remove trailing | */
		ret=-3;
		if(strlen(str->str)>0)
		{
			GString *output;
			char *oplist;
			int i=FALSE;

			output=g_string_new("");
			t=str->str;
			SKIP_SPACE(t);
			oplist=t;
			while(strchr(oplist,'$'))
			{
				t=strchr(oplist,'$');
				*t++='\0';
				if(i==TRUE)
				{
					output=g_string_append(output," ");
					output=g_string_append(output,oplist);
				}
				else
				{
					output=g_string_append(output,oplist);
					i=TRUE;
				}
				t+=1;
				oplist=t;
				/* output=g_string_append(output,oplist); */
			}
			ret=(fncs->nicklist_fnc)(xtra_param,output->str);
			g_string_free(output,TRUE);
		}
		g_string_free(str,TRUE);

		return ret;
	}

	else    if(!STRNCMP(input->str,"$OpList "))
	{       /* OpList handler */
		if(fncs->oplist_fnc==NULL)
			return 0;	/* no function to call, why wasting time to decode */

		str=g_string_new(input->str+strlen("$OpList "));
		str=g_string_truncate(str,str->len-1);	/* remove trailing | */
		ret=-3;
		if(strlen(str->str)>0)
		{
			GString *output;
			char *oplist;
			int i=FALSE;

			output=g_string_new("");
			t=str->str;
			SKIP_SPACE(t);
			oplist=t;
			printf("oplist: %s %i\n",t,strlen(t));
			
			while(strchr(oplist,'$'))
			{
				t=strchr(oplist,'$');
				*t++='\0';
				if(i==TRUE)
				{
					output=g_string_append(output," ");
					output=g_string_append(output,oplist);
				}
				else
				{
					output=g_string_append(output,oplist);
					i=TRUE;
				}
				t+=1;
				oplist=t;
				/* output=g_string_append(output,t); */
			}
			ret=(fncs->oplist_fnc)(xtra_param,output->str);
			g_string_free(output,TRUE);
		}
		g_string_free(str,TRUE);

		return ret;
	}

	if(fncs->other_fnc!=NULL)
		return (fncs->other_fnc)(xtra_param,input);
	return 0;
}

