/* Distributed Checksum Clearinghouse
 *
 * white-list lister
 *
 * Copyright (c) 2004 by Rhyolite Software
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND RHYOLITE SOFTWARE DISCLAIMS ALL
 * WARRANTIES WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL RHYOLITE SOFTWARE
 * BE LIABLE FOR ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES
 * OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 * WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 *
 * Rhyolite Software DCC 1.2.66-1.27 $Revision$
 */

#include "dcc_defs.h"
#include "dcc_ck.h"
#include <arpa/inet.h>

static DCC_EMSG dcc_emsg;
static const char *homedir;

static void NRATTRIB
usage(void)
{
	dcc_logbad(EX_USAGE, "usage: [-VQU] [-h homedir] [-t env_to] file");
}


int
main(int argc, char **argv)
{
	const char *to = 0;
	u_char print_version = 0;
	DCC_WHITE_INX inx, inx2;
	const char *nm;
	DCC_SUM sum;
	char tgts_buf[20], time_buf[26];
	struct tm tm;
	int col;
	u_char heading;
	int i;

	dcc_syslog_init(0, argv[0], 0);
	dcc_clnt_debug = 99;

	dup2(1, 2);			/* put error messages in context */

	dcc_wf_init(&cmn_wf, 0, 0);
	cmn_wf.flags |= DCC_WF_WLIST_CMD;

	while ((i = getopt(argc, argv, "VQUh:t:")) != EOF) {
		switch (i) {
		case 'V':
			fprintf(stderr, DCC_VERSION"\n");
			print_version = 1;
			break;

		case 'Q':		/* query;don't rebuild hash table */
			cmn_wf.flags |= DCC_WF_RO;
			break;

		case 'U':
			cmn_wf.flags |= DCC_WF_PER_USER;
			cmn_wf.flags &= ~DCC_WF_WLIST_CMD;
			break;

		case 'h':
			homedir = optarg;
			break;

		case 't':
			to = optarg;
			break;

		default:
			usage();
		}
	}
	argc -= optind;
	argv += optind;
	if (argc == 0) {
		if (print_version)
			exit(EX_OK);
		usage();
	}

	if (!dcc_cdhome(dcc_emsg, homedir))
		dcc_error_msg("%s", dcc_emsg);

	while ((nm = *argv++) != 0) {
		printf("\n%s\n", nm);
		if (!dcc_white_rdy(dcc_emsg, &cmn_wf, nm,
				   DCC_WHITE_RDY_LOCK_NEED_BOTH, 0)) {
			dcc_wf_unlock(&cmn_wf);
			dcc_error_msg("%s", dcc_emsg);
			exit(EX_DATAERR);
		}
		printf(" %s\n", cmn_wf.info->magic);

		if (to) {
			dcc_str2ck(sum, 0, 0, to);
			printf("%s\n%8s %s\n    ",
			       to,
			       dcc_type2str_err(DCC_CK_ENV_TO, 0, 0),
			       dcc_ck2str_err(DCC_CK_ENV_TO, sum));
			switch (dcc_white_sum(dcc_emsg, &cmn_wf,
					      DCC_CK_ENV_TO, sum, 1)) {
			case DCC_WHITE_ERROR:
				dcc_error_msg("%s", dcc_emsg);
				break;
			case DCC_WHITE_HALF_LISTED:
				printf("DCC_WHITE_HALF_LISTED\n");
				break;
			case DCC_WHITE_LISTED:
				printf("DCC_WHITE_LISTED\n");
				break;
			case DCC_WHITE_UNLISTED:
				printf("DCC_WHITE_UNLISTED\n");
				break;
			case DCC_WHITE_BLACK:
				printf("DCC_WHITE_BLACK\n");
				break;
			}
			continue;
		}

		printf("    %d entries  generation #%d\n",
		       cmn_wf.info->hdr.entries, cmn_wf.info->hdr.gen);
		if (cmn_wf.info_flags & DCC_WHITE_FG_PER_USER)
			printf("    per-user whitelist\n");
		switch (cmn_wf.info->hdr.resolve) {
		case DCC_WHITE_RESOLVE_BAD:
			printf("    hash table broken\n");
			break;
		case DCC_WHITE_RESOLVE_NO_IP:
			printf("    no host names\n");
			break;
		default:
			strftime(time_buf, sizeof(time_buf), "%b %d %X %Z",
				 dcc_localtime(cmn_wf.info->hdr.resolve, &tm));
			printf("    resolve hostnames after %s\n", time_buf);
			break;
		}
		for (i = 0; i < DIM(cmn_wf.info->hdr.white_incs); ++i) {
			if (cmn_wf.info->hdr.white_incs[i][0] == '\0')
				break;
			if (!i)
				printf("    includes\n");
			printf("      %s\n",
			       DCC_NM2PATH(cmn_wf.info->hdr.white_incs[i]));
		}
		if (cmn_wf.info_flags & DCC_WHITE_FG_DCC_ON)
			printf("    option DCC-on\n");
		if (cmn_wf.info_flags & DCC_WHITE_FG_DCC_OFF)
			printf("    option DCC-off\n");
		if (cmn_wf.info_flags & DCC_WHITE_FG_GREY_ON)
			printf("    option greylist-on\n");
		if (cmn_wf.info_flags & DCC_WHITE_FG_GREY_OFF)
			printf("    option greylist-off\n");
		if (cmn_wf.info_flags & DCC_WHITE_FG_LOG_ALL)
			printf("    option log-all\n");
		if (cmn_wf.info_flags & DCC_WHITE_FG_LOG_NORMAL)
			printf("    option log-normal\n");
		if (cmn_wf.info_flags & DCC_WHITE_FG_GREY_LOG_ON)
			printf("    option greylist-log-on\n");
		if (cmn_wf.info_flags & DCC_WHITE_FG_GREY_LOG_OFF)
			printf("    option greylist-log-off\n");

		heading = 0;
		for (i = 0; i < cmn_wf.info->hdr.cidr_cur.len; ++i) {
			char cidr_buf[INET6_ADDRSTRLEN];
			DCC_WHITE_CIDR_ENTRY *e=&cmn_wf.info->hdr.cidr_cur.e[i];
			if (!heading) {
				heading = 1;
				printf("\n CIDR blocks\n");
			}
			if (DCC_INET_NTOP(AF_INET6, &e->addr,
					  cidr_buf, sizeof(cidr_buf)))
				printf("%6s    %s/%d",
				       dcc_cnt2str(tgts_buf, sizeof(tgts_buf),
						   e->tgts,0),
				       cidr_buf, e->bits);
			else
				printf("%6s    %s/%d",
				       dcc_cnt2str(tgts_buf, sizeof(tgts_buf),
						   e->tgts,0),
				       "???", e->bits);
		}
		if (heading)
			putchar('\n');

		/* first the hash table */
		fputs("\n hash table\n", stdout);
		col = 0;
		for (inx = 0; inx < DIM(cmn_wf.info->bins); ++inx) {
			if (!cmn_wf.info->bins[inx]
			    && col == 0
			    && inx != 0) {
				inx2 = inx;
				while (inx2 < DIM(cmn_wf.info->bins)
				       && !cmn_wf.info->bins[inx2])
					++inx2;
				i = inx2 - inx;
				i -= i % 4;
				if (i != 0) {
					printf(" ...\n");
					inx += i;
				}
			}
			printf("%4d:", inx);
			if (cmn_wf.info->bins[inx]) {
				printf("%-4d", cmn_wf.info->bins[inx]);
			} else {
				printf("    ");
			}
			col = (col + 1) % 4;
			putchar(col == 0 ? '\n' : '\t');
		}

		/* then the entries */
		printf("\n\n%4s->%-4s  %12s  %6s\n",
		       "slot", "next", "type", "count");
		for (inx = 0; inx < cmn_wf.info_entries; ++inx) {
			DCC_WHITE_ENTRY *e = &cmn_wf.info->tbl[inx];
			if (e->type == DCC_CK_INVALID)
				continue;
			printf("%4d->%-4d  %12s  %6s  %s\n",
			       inx, e->fwd,
			       dcc_type2str_err(e->type, 0, 0),
			       dcc_cnt2str(tgts_buf, sizeof(tgts_buf),
					   e->tgts, 0),
			       dcc_ck2str_err(e->type, e->sum));
		}
	}
	exit(EX_OK);
}
