/* Distributed Checksum Clearinghouse
 *
 * convert a service name to a port number
 *
 * Copyright (c) 2004 by Rhyolite Software
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND RHYOLITE SOFTWARE DISCLAIMS ALL
 * WARRANTIES WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL RHYOLITE SOFTWARE
 * BE LIABLE FOR ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES
 * OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 * WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 *
 * Rhyolite Software DCC 1.2.66-1.48 $Revision$
 */

#include "dcc_clnt.h"
#ifndef DCC_WIN32
#include <arpa/inet.h>			/* for AIX */
#endif


#ifndef NO_IPV6
static struct hostent *dcc_hp;
#endif


/* get port number
 *	Note that this function uses dcc_host_lock() and dcc_host_unlock() */
u_int					/* DCC_GET_PORT_INVALID or port # */
dcc_get_port(DCC_EMSG emsg,
	     const char *portname,
	     u_int def_port,		/* DCC_GET_PORT_INVALID or default */
	     const char *fnm, int lineno)
{
	char *p;
	unsigned long l;
	struct servent *sp;
	u_int16_t port;


	if (portname[0] == '\0'
	    || !strcmp(portname, "-")) {
		if (def_port != DCC_GET_PORT_INVALID)
			return def_port;
		dcc_pemsg(EX_USAGE, emsg, "missing port%s",
			  fnm_lineno(fnm, lineno));
		return DCC_GET_PORT_INVALID;
	}

	/* first try a numeric port number, since that is common and
	 * the getservby* functions are so slow. */
	l = strtoul(portname, &p,0);
	if (*p == '\0' && l > 0 && l <= 65535)
		return htons((u_int16_t)l);

	dcc_host_lock();
	sp = getservbyname(portname, 0);
	if (sp) {
		port = sp->s_port;
		dcc_host_unlock();
		return port;
	}
	dcc_host_unlock();

	dcc_pemsg(EX_USAGE, emsg, "invalid port \"%s\"%s",
		  portname, fnm_lineno(fnm, lineno));
	return DCC_GET_PORT_INVALID;
}



/* This must be protected with dcc_host_lock()and dcc_host_unlock().
 *	On some platforms we need to use freehostent() and others we
 *	can't.  freehostent() is not guaranteed to be thread safe,
 *	so there's no profit in caring too much about it.
 */
void
dcc_freehostent(void)
{
#ifndef NO_IPV6
	if (dcc_hp) {
		freehostent(dcc_hp);
		dcc_hp = 0;
	}
#endif
	dcc_host_unlock();
}



/* This must be protected with dcc_host_lock()and dcc_freehostent(). */
static const struct hostent *
dcc_get_host_ipv4(const char *nm,	/* look for this name */
		  u_char use_ipv6,	/* 0=IPv4, 1=require IPv6, 2=guess */
		  int *ep,		/* put errno or herrno here */
		  struct hostent * (WSAAPI fnc)(const char *))
{
	const struct hostent *hp;
	static struct hostent num_hp;
	static struct in_addr ipv4;
	static struct in6_addr ipv6;
	static const char* addr_list[2] = {(char*)&ipv4, 0};
	static char num_nm[16];

#ifdef NO_IPV6
	if (use_ipv6 == 2)
		use_ipv6 = 0;
#endif
	if (!dcc_host_locked)
		dcc_logbad(EX_SOFTWARE, "dcc_get_host() not locked");

	/* First see if it is a number to avoid the MicroStupid stall
	 * when doing a gethostbyname() on a number */
	ipv4.s_addr = inet_addr(nm);
	if (ipv4.s_addr != INADDR_NONE) {
		strncpy(num_nm, inet_ntoa(ipv4), sizeof(num_nm));
		num_hp.h_name = num_nm;
		num_hp.h_addr_list = (char**)addr_list;
		if (use_ipv6 == 1) {
			dcc_toipv6(&ipv6, ipv4);
			addr_list[0] = (char *)&ipv6;
			num_hp.h_length = sizeof(ipv6);
			num_hp.h_addrtype = AF_INET6;
		} else {
			addr_list[0] = (char *)&ipv4;
			num_hp.h_length = sizeof(ipv4);
			num_hp.h_addrtype = AF_INET;
		}
		return &num_hp;
	}

	hp = fnc(nm);

	if (hp) {
		if (!use_ipv6)
			return hp;
		/* convert answer to IPv6 if required */
		num_hp.h_name = hp->h_name;
		num_hp.h_addr_list = (char**)addr_list;
		dcc_toipv6(&ipv6, *(struct in_addr *)hp->h_addr);
		addr_list[0] = (char *)&ipv6;
		num_hp.h_addrtype = AF_INET6;
		return &num_hp;
	}
	*ep = h_errno;
	return 0;
}



/* This must be protected with dcc_host_lock()and dcc_freehostent(). */
const struct hostent *
dcc_get_host_SOCKS(const char *nm,	/* look for this name */
		   u_char use_ipv6,	/* 0=IPv4, 1=require IPv6, 2=guess */
		   int *ep)		/* put errno or herrno here */
{
	return dcc_get_host_ipv4(nm, use_ipv6, ep, Rgethostbyname);
}



/* This must be protected with dcc_host_lock()and dcc_freehostent().
 *	It does not assme that gethostbyname() or whatever is thread safe.
 */
const struct hostent *
dcc_get_host(const char *nm,		/* look for this name */
	     u_char use_ipv6,		/* 0=IPv4, 1=require IPv6, 2=guess */
	     int *ep)			/* put errno or herrno here */
{
#ifndef NO_IPV6
	if (!dcc_host_locked)
		dcc_logbad(EX_SOFTWARE, "dcc_get_host() not locked");

	if (dcc_hp != 0)
		dcc_logbad(EX_SOFTWARE, "dcc_hp not freed");

	if (use_ipv6 == 2) {
		dcc_hp =  getipnodebyname(nm, AF_INET6,
					  AI_V4MAPPED | AI_ADDRCONFIG, ep);
		if (dcc_hp)
			return dcc_hp;
		use_ipv6 = 0;
	}
	if (use_ipv6)
		dcc_hp = getipnodebyname(nm, AF_INET6, AI_V4MAPPED|AI_ALL, ep);
	else
		dcc_hp = getipnodebyname(nm, AF_INET, 0, ep);
	return dcc_hp;
#else
	return dcc_get_host_ipv4(nm, use_ipv6, ep, gethostbyname);
#endif
}



/* make socket address from an IP address, a family, and a port number */
DCC_SOCKU *
dcc_mk_su(DCC_SOCKU *su,		/* put it here */
	  int family,			/* AF_INET or AF_INET6 */
	  const void *addrp,		/* this IP address; 0=INADDR_ANY */
	  u_short port)
{
	memset(su, 0, sizeof(*su));	/* assume INADDR_ANY=0 */
	su->sa.sa_family = family;
	if (family == AF_INET) {
#ifdef DCC_HAVE_SA_LEN
		su->sa.sa_len = sizeof(struct sockaddr_in);
#endif
		su->ipv4.sin_port = port;
		if (addrp)
			memcpy(&su->ipv4.sin_addr, addrp,
			       sizeof(su->ipv4.sin_addr));
	} else {
#ifdef DCC_HAVE_SA_LEN
		su->sa.sa_len = sizeof(struct sockaddr_in6);
#endif
		su->ipv6.sin6_port = port;
		if (addrp)
			memcpy(&su->ipv6.sin6_addr, addrp,
			       sizeof(su->ipv6.sin6_addr));
	}

	return su;
}



/* strip leading and trailing white space */
static const char *
dcc_strip_white(const char *str, u_int *lenp)
{
	const char *end;
	char c;

	str += strspn(str, DCC_WHITESPACE);
	end = str+strlen(str);
	while (end > str) {
		c = *(end-1);
		if (c != ' ' && c != '\t' && c != '\r' && c != '\n')
			break;
		--end;
	}
	*lenp = end-str;
	return str;
}



/* get a socket address from a dotted quad or IPv6 string */
u_char
dcc_str2ip(DCC_SOCKU *su, const char *str)
{
#ifndef NO_IPV6
	u_int len;
	char buf[INET6_ADDRSTRLEN];
#endif

#ifdef HAVE_INET_ATON
	if (0 < inet_aton(str, &su->ipv4.sin_addr)) {
		su->sa.sa_family = AF_INET;
		return 1;
	}
#else
	u_int addr = inet_addr(str);
	if (su->ipv4.sin_addr.s_addr != INADDR_NONE) {
		su->ipv4.sin_addr.s_addr = addr;
		su->sa.sa_family = AF_INET;
		return 1;
	}
#endif

#ifndef NO_IPV6
	/* Try IPv6 only after failing to understand the address as IPv4.
	 *
	 * inet_pton() does not like blanks or terminal '\n'
	 * It is also too smart by half and assumes that it is
	 * given a pointer to struct sockaddr.  When it decodes
	 * an IPv4 address, it sticks it 4 bytes before the
	 * start of the IPv6 buffer it is given. */
	str = dcc_strip_white(str, &len);
	if (len == 0 || len >= sizeof(buf))
		return 0;
	memcpy(buf, str, len);
	buf[len] = '\0';
	if (0 < inet_pton(AF_INET6, buf, &su->ipv6.sin6_addr)) {
		su->sa.sa_family = AF_INET6;
		return 1;
	}
#endif
	return 0;
}



void
dcc_bits2mask(struct in6_addr *mask, int bits)
{
	int wordno, i;

	for (wordno = 0; wordno < 4; ++wordno) {
		i = bits - wordno*32;
		if (i >= 32) {
			mask->s6_addr32[wordno] = 0xffffffff;
			continue;
		}
		if (i <= 0) {
			mask->s6_addr32[wordno] = 0;
		} else {
			mask->s6_addr32[wordno] = 0xffffffff << (32-i);
		}
		mask->s6_addr32[wordno] = htonl(mask->s6_addr32[wordno]);
	}
}



/* get an IPv6 address and netmask */
int					/* # of bits, 0=not address, -1 error */
dcc_str2cidr(DCC_EMSG emsg,
	     struct in6_addr *addr6,
	     struct in6_addr *mask6,
	     const char *str,
	     const char *fnm, int lineno)
{
	char addrstr[INET6_ADDRSTRLEN];
	DCC_SOCKU su;
	struct in6_addr mask6_loc;
	const char *mp;
	char *p;
	u_int str_len;
	int bits, wordno;

	str = dcc_strip_white(str, &str_len);
	mp = strchr(str, '/');

	if (!mp) {
		if (str_len >= ISZ(addrstr))
			return 0;	/* not an IP address */
		memcpy(addrstr, str, str_len);
		addrstr[str_len] = '\0';
	} else if (mp == str
		   || mp >= str+sizeof(addrstr)) {
		dcc_pemsg(EX_NOHOST, emsg,
			  "invalid IP address range \"%s\"%s",
			  str, fnm_lineno(fnm, lineno));
		return -1;
	} else {
		memcpy(addrstr, str, mp-str);
		addrstr[mp-str] = '\0';
	}
	if (!dcc_str2ip(&su, addrstr))
		return 0;		/* not an IP address */

	if (su.sa.sa_family == AF_INET6) {
		*addr6 = su.ipv6.sin6_addr;
		bits = mp ? strtoul(++mp, &p, 10) : 128;
	} else {
		dcc_toipv6(addr6, su.ipv4.sin_addr);
		bits = mp ? (strtoul(++mp, &p, 10) + 128-32) : 128;
	}
	if ((mp && *p != '\0' && p < str+str_len)
	    || bits > 128 || bits <= 0) {
		dcc_pemsg(EX_NOHOST, emsg,
			  "invalid netmask length \"%s\"%s",
			  str, fnm_lineno(fnm, lineno));
		return -1;
	}

	if (!mask6)
		mask6 = &mask6_loc;
	dcc_bits2mask(mask6, bits);
	for (wordno = 0; wordno < 4; ++wordno) {
		if ((addr6->s6_addr32[wordno]
		     & ~mask6->s6_addr32[wordno]) != 0) {
			dcc_pemsg(EX_NOHOST, emsg,
				  "%s does not start on %s-bit CIDR boundary%s",
				  str, mp, fnm_lineno(fnm, lineno));
			return -1;
		}
	}

	return bits;
}



/* Create and bind a UDP socket.
 *	The client library uses this function to determine whether
 *	IPv6 works. */
u_char					/* 0=fatal error, 1=done */
dcc_udp_bind(DCC_EMSG emsg,
	     SOCKET *fdp,		/* INVALID_SOCKET or socket */
	     DCC_SOCKU *su,
	     int *retry_secs)
{
	int tenths, i;
#ifdef DCC_WIN32
	u_long on;
#endif


#ifdef NO_IPV6
	if (su->sa.sa_family == AF_INET6) {
		*fdp = INVALID_SOCKET;
		return 1;
	}
#endif
	*fdp = socket(su->sa.sa_family, SOCK_DGRAM, 0);
	if (*fdp == INVALID_SOCKET) {
		dcc_pemsg(EX_OSERR, emsg, "socket(UDP): %s", ERROR_STR());
#ifndef NO_IPV6
		/* let the caller try again if this system does not do IPv6 */
		if (su->sa.sa_family == AF_INET6
		    && (errno == EPFNOSUPPORT
			|| errno == EPROTONOSUPPORT))
			return 1;
#endif
		return 0;
	}

#ifdef DCC_WIN32
	on = 1;
	if (SOCKET_ERROR == ioctlsocket(*fdp, FIONBIO, &on)) {
		dcc_pemsg(EX_OSERR, emsg, "fcntl(UDP, F_SETFD, FD_CLOEXEC): %s",
			  ERROR_STR());
		closesocket(*fdp);
		*fdp = INVALID_SOCKET;
		return 0;
	}
#else
	if (0 > fcntl(*fdp, F_SETFD, FD_CLOEXEC)) {
		dcc_pemsg(EX_OSERR, emsg, "fcntl(UDP, F_SETFD, FD_CLOEXEC): %s",
			  ERROR_STR());
		closesocket(*fdp);
		*fdp = INVALID_SOCKET;
		return 0;
	}
	if (-1 == fcntl(*fdp, F_SETFL,
			fcntl(*fdp, F_GETFL, 0) | O_NONBLOCK)) {
		dcc_pemsg(EX_OSERR, emsg, "fcntl(socket O_NONBLOCK): %s",
			  ERROR_STR());
		closesocket(*fdp);
		*fdp = INVALID_SOCKET;
		return 0;
	}
#endif

	tenths = 10;
	for (;;) {
		if (SOCKET_ERROR != bind(*fdp, &su->sa, DCC_SU_LEN(su)))
			return 1;

		if (errno != EADDRINUSE
		    || !retry_secs || *retry_secs <= 0) {
#ifndef NO_IPV6
			if (su->sa.sa_family == AF_INET6
			    && (errno == EPFNOSUPPORT
				|| errno == EPROTONOSUPPORT))
				i = 1;
			else
#endif
				i = 0;
			dcc_pemsg(EX_OSERR, emsg, "bind(UDP %s): %s",
				  dcc_su2str(su), ERROR_STR());
			closesocket(*fdp);
			*fdp = INVALID_SOCKET;
			return i;
		}

		usleep(100*1000);
		if (!--tenths) {
			--*retry_secs;
			tenths = 10;
		}
	}
}
