/* Distributed Checksum Clearinghouse
 *
 * DCC interface daemon
 *
 * Copyright (c) 2004 by Rhyolite Software
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND RHYOLITE SOFTWARE DISCLAIMS ALL
 * WARRANTIES WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL RHYOLITE SOFTWARE
 * BE LIABLE FOR ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES
 * OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 * WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 *
 * Rhyolite Software DCC 1.2.66-1.52 $Revision$
 */

#include "dccif.h"
#include "cmn_defs.h"
#include "dcc_paths.h"
#include <signal.h>


static int stopping;

static const char *conn_addr;
static u_char conn_family;
static struct sockaddr_un conn_sun;
static char conn_host[MAXHOSTNAMELEN+1];
static u_int16_t conn_port;
static DCC_SOCKU conn_su;
static const char *raddr_rmask;
static struct in6_addr raddr, rmask;
static SOCKET main_sock = -1;

static const char *homedir = 0;
static const char *rundir = DCC_RUNDIR;
static DCC_PATH pidpath;

static const char *tmpdir;

static u_char background = 1;

static const char *userdirs;
static DCC_PATH userdirs_path;

static struct {
    int	    tgts;			/* total addressees */
    int	    tgts_discarded;		/* discarded for this many addressess */
    int	    tgts_rejected;
    int	    tgts_embargoed;
    time_t  prev;
    time_t  now;
    time_t  next;
} totals;

static enum {
    DCCIFD_SETHDR, DCCIFD_ADDHDR, DCCIFD_NOHDR
} chghdr = DCCIFD_SETHDR;

/* message state or context */
typedef struct work {
    int		mta_fd;			/* connection to MTA */
    DCC_SOCKU	mta_addr;
    char	env_from[DCC_HDR_CK_MAX+1];
    int		tmp_fd;			/* to send a copy back to the MTA */
    DCC_PATH	tmp_nm;
    char	srbuf[DCC_HDR_CK_MAX*8];    /* >=DCC_HDR_CK_MAX*2 & MAX_RCPTS */
    char	swbuf[DCC_HDR_CK_MAX*8];
    CMN_WORK	cw;
    /* from here down is zeroed when the structure is allocated */
#define WORK_ZERO fwd
    struct work *fwd;
    char	*srbuf_out;
    char	*srbuf_in;
    char	*srbuf_next_line;
    u_int	swbuf_len;
    int		total_hdrs, cr_hdrs;
    u_int	flags;
#    define	 FG_WORK_LOCK	    0x0001  /* hold the lock */
#    define	 FG_MISSING_BODY    0x0002  /* missing message body */
#    define	 FG_MTA_BODY	    0x0004  /* MTA wants the body */
#    define	 FG_MTA_HEADER	    0x0008  /* MTA wants the X-DCC header */
#    define	 FG_MTA_DCC_QUERY   0x0010  /* MTA only wants to query DCC */
#    define	 FG_MTA_GREY_QUERY  0x0020  /* MTA wants query DCC greylist */
#    define	 FG_MTA_NO_REJECT   0x0040  /* MTA wants only greylisting */
#    define	 FG_SEEN_HDR	    0x0080  /* have at least 1 header */
#    define	 FG_CK_RCVD	    0x0100  /* check 1st Received header */
} WORK;

/* use a free list to avoid malloc() overhead */
static WORK *work_free;

/* Every job involves
 *	a socket connected to the MTA,
 *	a log file,
 *	and a socket to talk to the DCC server.
 * While processing per-user whitelists there are
 *	another log file
 *	and an extra file descriptor for the main log file.
 * The file descriptors for the whitelists are accounted for in EXTRA_FILES */
#define FILES_PER_JOB	5


#define MAX_SELECT_WORK ((FD_SETSIZE-EXTRA_FILES)/FILES_PER_JOB)
#define DEF_MAX_WORK	200
#define MIN_MAX_WORK	2
static int max_max_work = MAX_SELECT_WORK;
static int max_work = DEF_MAX_WORK;

#define MAX_MTA_DELAY	5		/* max seconds to wait for MTA */


static void sigterm(int);
static void totals_msg(void);
static u_char set_sock(DCC_EMSG, int, const char *);
static void open_unix_sock(void);
static void open_tcp_sock(void);
static void close_main_sock(void);
static void unlink_conn_sun(void);
static void *job_start(void *);
static void job_close(WORK *);
static void NRATTRIB job_exit(WORK *);


static void
usage(u_char die)
{
	const char str[] = {
	    "usage: [-VdbxANQW] [-G on | off | noIP | IPmask/xx] [-h homedir]\n"
	    "    [-p /socket/name | host,port] [-m map] [-w whiteclnt]\n"
	    "    [-U userdirs] [-t type,[log-thold,][spam-thold]]"
	    " [-g [not-]type]\n"
	    "    [-S header] [-l logdir] [-R rundir] [-T tmpdir]\n"
	    "    [-j maxjobs] [-L ltype,facility.level]"
	};
	static u_char complained;

	/* its important to try to run, so don't give up unless necessary */
	if (die) {
		dcc_logbad(EX_USAGE, complained ? "giving up" : str);
	} else if (!complained) {
		dcc_error_msg("%s\ncontinuing", str);
		complained = 1;
	}
}


int NRATTRIB
main(int argc, char **argv)
{
	DCC_EMSG emsg;
#ifdef HAVE_RLIMIT_NOFILE
	struct rlimit nofile;
	int old_rlim_cur;
#endif
	FILE *f;
	long l;
	u_char log_tgts_set = 0;
	const char *logdir = 0;
	WORK *wp;
	char *p;
	time_t now;
	struct timeval delay;
	int secs;
	pthread_t tid;
	int namelen;
	int i;

	emsg[0] = '\0';
	dcc_syslog_init(1, argv[0], 0);
	dcc_init_tholds();

#ifdef HAVE_RLIMIT_NOFILE
	if (0 > getrlimit(RLIMIT_NOFILE, &nofile)) {
		dcc_error_msg("getrlimit(RLIMIT_NOFILE): %s", ERROR_STR());
		old_rlim_cur = 1000*1000;
	} else {
		old_rlim_cur = nofile.rlim_cur;
		if (nofile.rlim_max < 1000*1000) {
			i = nofile.rlim_max;
#ifndef USE_POLL
			if (i > FD_SETSIZE)
				i = FD_SETSIZE;
#endif
			max_max_work = (i - EXTRA_FILES)/FILES_PER_JOB;
			if (max_max_work <= 0) {
				dcc_error_msg("only %d open files allowed",
					      (int)nofile.rlim_max);
				max_max_work = 1;
			}
		}
	}
#else /* HAVE_RLIMIT_NOFILE */
	if (max_work <= 0) {
		dcc_error_msg(EX_OSERR, "too few open files allowed");
		max_max_work = 1;
	}
#endif /* HAVE_RLIMIT_NOFILE */
	max_work = min(DEF_MAX_WORK, max_max_work);

	while (EOF != (i = getopt(argc, argv,
				  "VdbxANQW"
				  "G:h:p:m:w:U:t:g:S:l:R:T:j:L:"))) {
		switch (i) {
		case 'V':
			fprintf(stderr, DCC_VERSION"\n");
			exit(EX_OK);
			break;

		case 'd':
			++dcc_clnt_debug;
			break;

		case 'b':
			background = 0;
			break;

		case 'x':
			try_extra_hard = DCC_CLNT_FG_NO_FAIL;
			break;

		case 'A':
			chghdr = DCCIFD_ADDHDR;
			break;

		case 'N':
			chghdr = DCCIFD_NOHDR;
			break;

		case 'Q':
			dcc_query_only = 1;
			break;

		case 'G':
			if (!dcc_parse_client_grey(optarg))
				usage(0);
			break;

		case 'W':
			to_white_only = 1;
			break;

		case 'h':
			homedir = optarg;
			break;

		case 'p':
			conn_addr = optarg;
			break;

		case 'm':
			mapfile_nm = optarg;
			break;

		case 'w':
			main_white_nm = optarg;
			break;

		case 'U':
			if (*optarg == '\0') {
				userdirs = optarg;
			} else {
				snprintf(userdirs_path, sizeof(userdirs_path),
					 "%s/", optarg);
				userdirs = userdirs_path;
			}
			break;

		case 't':
			if (dcc_parse_tholds('t', optarg))
				log_tgts_set = 1;
			break;

		case 'g':		/* honor not-spam "counts" */
			dcc_parse_honor(optarg);
			break;

		case 'S':
			dcc_add_sub_hdr(0, optarg);
			break;

		case 'l':		/* log rejected mail here */
			logdir = optarg;
			break;

		case 'R':
			rundir = optarg;
			break;

		case 'T':
			tmpdir = optarg;
			break;

		case 'j':		/* maximum simultaneous jobs */
			l = strtoul(optarg, &p, 0);
			if (*p != '\0' || l < MIN_MAX_WORK) {
				dcc_error_msg("invalid queue length %s",
					      optarg);
			} else if (l > max_max_work) {
				dcc_error_msg("queue length %s"
					      " larger than limit %d",
					      optarg, max_max_work);
				max_work = max_max_work;
			} else {
				max_work = l;
			}
			break;

		case 'L':
			dcc_parse_log_opt(optarg);
			break;

		default:
			usage(0);
		}
	}
	argc -= optind;
	argv += optind;
	if (argc != 0)
		usage(0);

	if (!dcc_cdhome(emsg, homedir))
		dcc_error_msg("%s", emsg);

	/* Open the socket before a dccif() function that is starting us
	 * looks for it, and so before our backgrounding fork(). */
	if (!conn_addr) {
		/* use default UNIX domain socket */
		snprintf(conn_sun.sun_path, sizeof(conn_sun.sun_path),
			 "%s/"DCC_DCCIF_UDS, dcc_homedir);
		conn_addr = conn_sun.sun_path;
	}
	p = strchr(conn_addr, ',');
	if (p && strchr(p, ',')) {
		open_tcp_sock();
	} else {
		open_unix_sock();
	}
	if (!set_sock(emsg, main_sock, "main socket"))
		dcc_logbad(dcc_ex_code, "%s", emsg);
	if (0 > listen(main_sock, 10))
		dcc_logbad(EX_IOERR, "listen(): %s", ERROR_STR());

	if (logdir)
		dcc_log_init(0, logdir);
	if (!dcc_have_logdir
	    && log_tgts_set)
		dcc_error_msg("log thresholds set with -t but no -l directory");

	if (!tmpdir)
		tmpdir = dcc_have_logdir ? dcc_logdir : _PATH_TMP;

#ifdef HAVE_RLIMIT_NOFILE
	if (old_rlim_cur < (i = max_work*FILES_PER_JOB+EXTRA_FILES)) {
		nofile.rlim_cur = i;
		if (0 > setrlimit(RLIMIT_NOFILE, &nofile)) {
			dcc_error_msg("setrlimit(RLIMIT_NOFILE,%d): %s",
				      i, ERROR_STR());
			max_work = old_rlim_cur/FILES_PER_JOB - EXTRA_FILES;
			if (max_work <= 0) {
				dcc_error_msg("only %d open files allowed",
					      old_rlim_cur);
				max_work = MIN_MAX_WORK;
			}
		}
	}
#endif /* HAVE_RLIMIT_NOFILE */

	/* Create the contexts. */
	i = max_work;
	wp = dcc_malloc(sizeof(*wp)*i);
	work_free = wp;
	memset(wp, 0, sizeof(*wp)*i);
	while (--i > 0) {
		wp->mta_fd = -1;
		wp->tmp_fd = -1;
		cmn_create(&wp->cw);
		wp->fwd = wp+1;
		++wp;
	}
	wp->mta_fd = -1;
	wp->tmp_fd = -1;
	cmn_create(&wp->cw);

	rcpts_create(max_work*2);

	if (background) {
		if (daemon(1, 0) < 0)
			dcc_logbad(EX_OSERR, "daemon(): %s", ERROR_STR());
		snprintf(pidpath, sizeof(pidpath), "%s/%s.pid",
			 rundir, dcc_progname);
		unlink(pidpath);
		f = fopen(pidpath, "w");
		if (!f) {
			dcc_error_msg("fopen(%s): %s",
				      pidpath, ERROR_STR());
		} else {
			fprintf(f, "%d\n", (u_int)getpid());
			fclose(f);
		}
	}

	signal(SIGPIPE, SIG_IGN);
	signal(SIGHUP, sigterm);
	signal(SIGTERM, sigterm);
	signal(SIGINT, sigterm);

	/* Be careful to start all threads only after the fork() in daemon(),
	 * because some POSIX threads packages (e.g. FreeBSD) get confused
	 * about threads in the parent.  */

	cmn_init(emsg);

	if (conn_family != AF_UNIX) {
		dcc_trace_msg(DCC_VERSION" listening to %s from %s",
			      conn_addr, raddr_rmask);
	} else {
		dcc_trace_msg(DCC_VERSION" listening to %s",
			      conn_addr);
	}
	if (dcc_clnt_debug)
		dcc_trace_msg("max_work=%d max_max_work=%d",
			      max_work, max_max_work);

	totals.prev  = time(0);
	totals.next = totals.prev+24*60*60;
	while (!stopping) {
		/* do nothing for a second if we are too busy */
		if (work_free == 0) {
			delay.tv_usec = 0;
			delay.tv_sec = 1;
			if (0 > select(0, 0, 0, 0, &delay)
			    && !DCC_SELECT_NERROR())
				dcc_logbad(EX_OSERR, "select():%s",
					   ERROR_STR());
			continue;
		}

		/* Recompute delay until we publish totals or quit at
		 * most once per second.  This is cheap enough and deals
		 * with clock changes. */
		now = time(0);
		if (totals.now != now) {
			struct tm tm;

			totals.now = now;
			if (now >= totals.next)
				totals_msg();

			dcc_localtime(now, &tm);
			tm.tm_sec = 0;
			tm.tm_min = 0;
			tm.tm_hour = 0;
			++tm.tm_mday;
			totals.next = mktime(&tm);
			if (totals.next == -1) {
				dcc_error_msg("mktime() failed");
				totals.next = now + 60*60;
			}
		}
		secs = totals.next;
		if (secs <= now) {
			secs = 0;
		} else {
			secs -= now;
			/* wake up frequently to notice signals */
			if (secs > 2)
				secs = 2;
		}

		i = dcc_select_poll(emsg, main_sock, 1, secs*DCC_USECS);
		if (i < 0)
			dcc_logbad(EX_OSERR, "%s", emsg);
		if (i == 0)
			continue;

		/* A new connection is ready.  Allocate a context
		 * block and create a thread */
		lock_work();
		wp = work_free;
		if (!wp) {
			/* pretend we weren't listening if we
			 * are out of context blocks */
			unlock_work();
			continue;
		}
		work_free = wp->fwd;
		unlock_work();

		/* clear most of it */
		memset(&wp->WORK_ZERO, 0,
		       sizeof(*wp) - ((char*)&wp->WORK_ZERO - (char*)wp));
		cmn_clear(&wp->cw, wp);

		namelen = sizeof(wp->mta_addr);
		wp->mta_fd = accept(main_sock, &wp->mta_addr.sa, &namelen);
		if (wp->mta_fd < 0) {
			dcc_error_msg("accept(): %s", ERROR_STR());
			job_close(wp);
			continue;
		}
		if (conn_family != AF_UNIX) {
			struct in6_addr addr6, *ap;

			if (wp->mta_addr.sa.sa_family == AF_INET6) {
				ap = &wp->mta_addr.ipv6.sin6_addr;
			} else {
				ap = &addr6;
				dcc_toipv6(ap, wp->mta_addr.ipv4.sin_addr);
			}
			if (!DCC_IN_BLOCK(*ap, raddr, rmask)) {
				dcc_error_msg("unauthorized client address %s",
					      dcc_su2str(&wp->mta_addr));
				job_close(wp);
				continue;
			}
		}
		if (!set_sock(emsg, wp->mta_fd, "MTA")) {
			dcc_error_msg("%s", emsg);
			job_close(wp);
			continue;
		}
		pthread_create(&tid, 0, job_start, wp);
		i = pthread_detach(tid);
		if (i != 0) {
		   if (i != ESRCH)
			   dcc_error_msg("pthread_detach(): %s", ERROR_STR1(i));
		   else if (dcc_clnt_debug)
			   dcc_trace_msg("pthread_detach(): %s", ERROR_STR1(i));
		}
	}
	totals_msg();

	close_main_sock();
	if (pidpath[0] != '\0')
		unlink(pidpath);

	exit(-stopping);
}



/* brag about our accomplishments */
static void
totals_msg(void)
{
	struct timeval now;
	struct tm tm;
	char tbuf[20];

	lock_work();
	gettimeofday(&now, 0);
	strftime(tbuf, sizeof(tbuf), "%x %X",
		 dcc_localtime(totals.prev, &tm));
	if (grey_on) {
		dcc_trace_msg(DCC_VERSION
			      " greylisted %d messages,"
			      " rejected messages to %d targets and"
			      " discarded messages to %d targets among"
			      " %d total since %s",
			      totals.tgts_embargoed,
			      totals.tgts_rejected,
			      totals.tgts_discarded,
			      totals.tgts, tbuf);
	} else {
		dcc_trace_msg(DCC_VERSION
			      " rejected messages to %d targets and"
			      " discarded messages to %d targets among"
			      " %d total since %s",
			      totals.tgts_rejected,
			      totals.tgts_discarded,
			      totals.tgts, tbuf);
	}
	memset(&totals, 0, sizeof(totals));
	totals.prev = now.tv_sec;
	unlock_work();
}



static void
unlink_conn_sun(void)
{
	if (conn_family != AF_UNIX)
		return;

	/* there is an unavoidable race here */
	if (0 > unlink(conn_sun.sun_path)
	    && errno != ENOENT)
		dcc_error_msg("unlink(%s): %s",
			      conn_sun.sun_path, ERROR_STR());
}



static void
open_unix_sock(void)
{
	DCC_EMSG emsg;
	struct stat sb;

	emsg[0] = '\0';
	conn_family = AF_UNIX;

	if (conn_addr != conn_sun.sun_path) {
		if (strlen(conn_addr) >= ISZ(conn_sun.sun_path))
			dcc_logbad(EX_USAGE, "invalid UNIX domain socket: %s",
				   conn_addr);
		strcpy(conn_sun.sun_path, conn_addr);
	}
#ifdef DCC_HAVE_SA_LEN
	conn_sun.sun_len = SUN_LEN(&conn_sun);
#endif
	conn_sun.sun_family = AF_UNIX;

	if (0 <= stat(conn_sun.sun_path, &sb)
	    && !(S_ISSOCK(sb.st_mode) || S_ISFIFO(sb.st_mode)))
		dcc_logbad(EX_UNAVAILABLE, "non-socket present at %s",
			   conn_sun.sun_path);

	/* look for a daemon already using our socket */
	main_sock = socket(AF_UNIX, SOCK_STREAM, 0);
	if (main_sock < 0)
		dcc_logbad(EX_OSERR, "socket(AF_UNIX): %s", ERROR_STR());
	/* unlink it only if it looks like a dead socket */
	if (0 > connect(main_sock, (struct sockaddr *)&conn_sun,
			sizeof(conn_sun))) {
		if (errno == ECONNREFUSED || errno == ECONNRESET
		    || errno == EACCES) {
			unlink_conn_sun();
		} else if (dcc_clnt_debug > 2
			   && errno != ENOENT) {
			dcc_trace_msg("connect(old server %s): %s",
				      conn_sun.sun_path, ERROR_STR());
		}
	} else {
		/* connect() worked so the socket is alive */
		dcc_logbad(EX_UNAVAILABLE,
			   "something already or still running with socket at"
			   " %s",
			   conn_sun.sun_path);
	}
	close(main_sock);

	main_sock = socket(AF_UNIX, SOCK_STREAM, 0);
	if (main_sock < 0)
		dcc_logbad(EX_OSERR, "socket(AF_UNIX): %s", ERROR_STR());
	if (0 > bind(main_sock, (struct sockaddr *)&conn_sun,
		     sizeof(conn_sun)))
		dcc_logbad(EX_IOERR, "bind(%s) %s",
			   conn_sun.sun_path, ERROR_STR());
	if (0 >	chmod(conn_sun.sun_path, 0666))
		dcc_error_msg("chmod(%s, 0666): %s",
			      conn_sun.sun_path, ERROR_STR());
}



static void
open_tcp_sock(void)
{
	DCC_EMSG emsg;
	const char *cp;
	const struct hostent *hp;
	char *rhost;
	int error;

	emsg[0] = '\0';

	rhost = strchr(conn_addr, ',');
	if (!rhost)
		dcc_logbad(EX_USAGE, "missing port number: \"%s\"",
			   conn_addr);
	rhost = strchr(rhost+1, ',');
	if (!rhost) {
		dcc_logbad(EX_USAGE, "missing remote host name: \"%s\"",
			   conn_addr);
	}
	*rhost++ = '\0';
	error = dcc_str2cidr(emsg, &raddr, &rmask, rhost, 0, 0);
	if (error <= 0) {
		dcc_logbad(EX_USAGE, "invalid remote host and mask: \"%s\"",
			   rhost);
	}
	raddr_rmask = rhost;

	cp = dcc_parse_nm_port(emsg, conn_addr, DCC_GET_PORT_INVALID,
			       conn_host, sizeof(conn_host),
			       &conn_port, 0, 0,
			       0, 0);
	if (!cp)
		dcc_logbad(dcc_ex_code, "%s", emsg);
	if (*cp != '\0')
		dcc_logbad(EX_USAGE, "invalid IP address: \"%s\"", conn_addr);

	if (conn_host[0] == '\0' || !strcmp(conn_host, "@")) {
			/* null or "@" means INADDR_ANY */
		dcc_mk_su(&conn_su, AF_INET, 0, conn_port);
	} else {
		hp = dcc_get_host(conn_host, 2, &error);
		if (!hp)
			dcc_logbad(EX_NOHOST, "%s: %s",
				   conn_host, hstrerror(error));
		dcc_mk_su(&conn_su, hp->h_addrtype, hp->h_addr, conn_port);
		dcc_freehostent();
	}

	main_sock = socket(conn_su.sa.sa_family, SOCK_STREAM, 0);
	if (main_sock < 0)
		dcc_logbad(EX_OSERR, "socket(): %s", ERROR_STR());
	if (0 > bind(main_sock, &conn_su.sa, DCC_SU_LEN(&conn_su)))
		dcc_logbad(EX_UNAVAILABLE, "bind(%s) %s",
			   conn_addr, ERROR_STR());

	conn_family = conn_su.sa.sa_family;
}



static u_char
set_sock(DCC_EMSG emsg, int s, const char *sname)
{
	int on;

	if (0 > fcntl(s, F_SETFD, FD_CLOEXEC)) {
		dcc_pemsg(EX_IOERR, emsg,
			  "fcntl(%s, F_SETFD, FD_CLOEXEC): %s",
			  sname, ERROR_STR());
		return 0;
	}

	if (conn_family != AF_UNIX) {
		on = 1;
		if (0 > setsockopt(s, SOL_SOCKET, SO_KEEPALIVE,
				   &on, sizeof(on))) {
			dcc_pemsg(EX_IOERR, emsg,
				  "setsockopt(%s, SO_KEEPALIVE): %s",
				  sname, ERROR_STR());
			return 0;
		}
	}

	if (-1 == fcntl(s, F_SETFL,
			fcntl(s, F_GETFL, 0) | O_NONBLOCK)) {
		dcc_pemsg(EX_OSERR, emsg, "fcntl(%s, O_NONBLOCK): %s",
			  sname, ERROR_STR());
		return 0;
	}

	return 1;
}



static u_char				/* 0=EOF, 1=read something */
sock_read(WORK *wp)
{
	int fspace, len, total, i;

	if (wp->srbuf_out >= wp->srbuf_in)
		wp->srbuf_out = wp->srbuf_in = wp->srbuf;

	fspace = &wp->srbuf[sizeof(wp->srbuf)] - wp->srbuf_in;
	if (fspace < DCC_HDR_CK_MAX) {
		if (wp->srbuf_out == wp->srbuf) {
			cmn_error_msg(&wp->cw, "buffer overrun; in=%d",
				      wp->srbuf_in - wp->srbuf);
			job_exit(wp);
		}
		len = wp->srbuf_in - wp->srbuf_out;
		memmove(wp->srbuf, wp->srbuf_out, len);
		wp->srbuf_out = wp->srbuf;
		wp->srbuf_in = wp->srbuf_out+len;
		fspace = sizeof(wp->srbuf) - len;
	}

	if (wp->flags & FG_WORK_LOCK)
		unlock_work();

	for (;;) {
		i = dcc_select_poll(wp->cw.emsg, wp->mta_fd, 1,
				    MAX_MTA_DELAY*DCC_USECS);
		if (i > 0)
			break;
		if (i < 0) {
			cmn_error_msg(&wp->cw, "%s", wp->cw.emsg);
		} else {
			cmn_error_msg(&wp->cw, "MTA read timeout");
		}
		job_exit(wp);
	}
	total = read(wp->mta_fd, wp->srbuf_in, fspace);
	if (total < 0) {
		cmn_error_msg(&wp->cw, "read(sock): %s", ERROR_STR());
		job_exit(wp);
	}
	wp->srbuf_in += total;

	if (wp->flags & FG_WORK_LOCK)
		lock_work();
	return total != 0;
}



/* ensure there is another line in wp->srbuf and add a terminal '\0' */
static int				/* bytes available */
sock_read_line(WORK *wp)
{
	int i;
	char *p;

	for (;;) {
		if (0 != (i = wp->srbuf_in - wp->srbuf_out)
		    && 0 != (p = memchr(wp->srbuf_out, '\n', i))) {
			wp->srbuf_next_line = p+1;
			for (;;) {
				*p-- = '\0';
				if (p < wp->srbuf_out)
					return 0;
				if (*p != '\r')
					return p+1 - wp->srbuf_out;
			}
		}

		if (!sock_read(wp)) {
			cmn_error_msg(&wp->cw, "truncated request");
			job_exit(wp);
		}
	}
}



/* Look for a string from the MTA */
static u_char
sock_ck_str(WORK *wp, const char *st, int stlen)
{
	while (wp->srbuf_out < wp->srbuf_next_line
	       && (*wp->srbuf_out == '\t' || *wp->srbuf_out == ' '))
		++wp->srbuf_out;

	if (stlen <= wp->srbuf_next_line - wp->srbuf_out
	    && !strncasecmp(wp->srbuf_out, st, stlen)) {
		if (wp->srbuf_out[stlen] == '\0') {
			wp->srbuf_out += stlen;
			return 1;
		}
		if (wp->srbuf_out[stlen] == '\t'
		    || wp->srbuf_out[stlen] == ' ') {
			do {
				++stlen;
			} while (wp->srbuf_out[stlen] == '\t'
				 || wp->srbuf_out[stlen] == ' ');
			wp->srbuf_out += stlen;
			return 1;
		}
	}

	return 0;
}



static void
sock_write_flush(WORK *wp)
{
	const char *buf;
	int len, i;

	len = wp->swbuf_len;
	if (!len)
		return;
	wp->swbuf_len = 0;

	if (wp->mta_fd < 0)
		dcc_logbad(EX_SOFTWARE, "attempt to write closed socket");

	buf = wp->swbuf;
	do {
		i = dcc_select_poll(wp->cw.emsg, wp->mta_fd, 0,
				    MAX_MTA_DELAY*DCC_USECS);
		if (i < 0) {
			cmn_error_msg(&wp->cw, "%s", wp->cw.emsg);
			job_exit(wp);
		}
		if (i == 0) {
			cmn_error_msg(&wp->cw, "MTA write timeout");
			job_exit(wp);
		}
		i = write(wp->mta_fd, buf, len);
		if (i < 0) {
			if (DCC_BLOCK_ERROR())
				continue;
			cmn_error_msg(&wp->cw, "write(MTA socket,%d): %s",
				      len, ERROR_STR());
			job_exit(wp);
		}
		if (i == 0) {
			cmn_error_msg(&wp->cw, "write(MTA socket,%d)=%d",
				      len, i);
			job_exit(wp);
		}
		buf += i;
		len -= i;
	} while (len > 0);
}



static void
sock_write_buf(WORK *wp, const void *buf, u_int len)
{
	u_int n;

	for (;;) {
		n = sizeof(wp->swbuf) - wp->swbuf_len;
		if (n > len)
			n = len;
		memcpy(&wp->swbuf[wp->swbuf_len], buf, n);
		if ((wp->swbuf_len += n) >= sizeof(wp->swbuf))
			sock_write_flush(wp);
		if ((len -= n) == 0)
			return;
		buf = (void *)((char *)buf + n);
	}
}



static void
tmp_write(WORK *wp, const void *buf, int len)
{
	int i;

	if (wp->tmp_fd < 0)
		return;

	i = write(wp->tmp_fd, buf, len);
	if (i != len) {
		if (i < 0)
			cmn_error_msg(&wp->cw, "write(%s,%d): %s",
				      wp->tmp_nm, len, ERROR_STR());
		else
			cmn_error_msg(&wp->cw, "write(%s,%d)=%d",
				      wp->tmp_nm, len, i);
		job_exit(wp);
	}
}



static int
tmp_read(WORK *wp, void *buf, int len)
{
	int i;

	if (wp->tmp_fd < 0)
		return 0;

	i = read(wp->tmp_fd, buf, len);
	if (i < 0) {
		cmn_error_msg(&wp->cw, "read(%s,%d): %s",
			      wp->tmp_nm, len, ERROR_STR());
		job_exit(wp);
	}
	return i;
}



/* fill wp->srbuf */
static int
tmp_read_srbuf(WORK *wp)
{
	int i;

	/* preserving what is already in it */
	i = wp->srbuf_in - wp->srbuf_out;
	if (i > 0)
		memmove(wp->srbuf, wp->srbuf_out, i);
	wp->srbuf_out = wp->srbuf;
	wp->srbuf_in = wp->srbuf_out+i;

	i = tmp_read(wp, wp->srbuf_in, ISZ(wp->srbuf) - i);
	if (i == 0)
		return 0;
	wp->srbuf_in += i;
	return 1;
}



static void
job_close(WORK *wp)
{
	if (wp->tmp_fd >= 0) {
		if (0 > close(wp->tmp_fd))
			cmn_error_msg(&wp->cw, "close(%s): %s",
				      wp->tmp_nm, ERROR_STR());
		wp->tmp_fd = -1;
	}
	if (wp->mta_fd >= 0) {
		sock_write_flush(wp);
		if (0 > close(wp->mta_fd))
			cmn_error_msg(&wp->cw, "close(socket): %s",
				      ERROR_STR());
		wp->mta_fd = -1;
	}
	log_stop(&wp->cw);

	if (!(wp->flags & FG_WORK_LOCK))
		lock_work();

	rcpts_free(&wp->cw, 0);

	wp->fwd = work_free;
	work_free = wp;
	wp->flags &= ~FG_WORK_LOCK;
	unlock_work();
}



static void NRATTRIB
job_exit(WORK *wp)
{
	job_close(wp);
	pthread_exit(0);
	/* mostly to suppress warning */
	dcc_logbad(EX_SOFTWARE, "pthread_exit() returned");
}



/* write headers lines from the buffer */
static void
hdr_write(WORK *wp, char *bol)
{
	int i;

	i = bol - wp->srbuf_out;
	if (i > 0) {
		sock_write_buf(wp, wp->srbuf_out, i);
		wp->srbuf_out = bol;
	}
}



static void
hdrs_copy(WORK *wp)
{
	enum {START_HDR, SKIP_HDR, COPY_HDR} cmode;
	char *bol, *nl, *eol;
	int i;

	if (-1 == lseek(wp->tmp_fd, 0, SEEK_SET)) {
		cmn_error_msg(&wp->cw, "rewind %s: %s",
			      wp->tmp_nm, ERROR_STR());
		job_exit(wp);
	}

	cmode = START_HDR;
	wp->srbuf_in = wp->srbuf_out = wp->srbuf;
	for (;;) {
		/* fill wp->srbuf while keeping anything already present */
		if (!tmp_read_srbuf(wp))
			return;

		bol = wp->srbuf_out;
		while ((i = wp->srbuf_in - bol) > 0) {
			/* Find the end of the next line. */
			eol = memchr(bol, '\n', i);
			if (!eol) {
				/* Fill the buffer if we can't find '\n''
				 * and the buffer has room */
				if (i < ISZ(wp->srbuf))
					break;
				eol = wp->srbuf_in-1;
				nl = 0;
			} else if (eol+1 >= wp->srbuf_in) {
				/* insist on having the character after '\n'
				 * or pretend we could not find it */
				if (i < ISZ(wp->srbuf))
					break;
				--eol;
				nl = 0;
			} else {
				nl = eol+1;
			}

			if (cmode == START_HDR) {
				/* We are at start of a header or the body.
				 * Quit before line of "\n" or "\r\n" */
				if (eol == bol
				    || (eol == bol+1 && *bol == '\r')) {
					/* write any preceding lines */
					hdr_write(wp, bol);
					return;
				}

				/* Look for our header
				 * Assume the buffer is larger than the
				 * largest possible X-DCC header. */
				if (bol[wp->cw.xhdr_len] == ':'
				    && !strncasecmp(bol, wp->cw.xhdr,
						    wp->cw.xhdr_len)
				    && chghdr == DCCIFD_SETHDR) {
					/* Found it and we want to skip it.
					 * Copy any preceding headers. */
					hdr_write(wp, bol);
					cmode = SKIP_HDR;
				} else {
					cmode = COPY_HDR;
				}
			}
			if (cmode == SKIP_HDR)
				wp->srbuf_out = eol+1;

			/* Check the character after '\n' for
			 * whitespace indicating continuation */
			if (nl && *nl  != ' ' && *nl != '\t')
				cmode = START_HDR;
			bol = eol+1;
		}

		hdr_write(wp, bol);
	}
}



/* We are finished with one SMTP message
 *	Send the result to the MTA and end this thread */
static void NRATTRIB
job_done(WORK *wp, char result_char, const char *result_str)
{
	RCPT_ST *rcpt_st;
	char *p;
	int i;

	LOG_CAPTION(wp, "result: ");
	log_write(&wp->cw, result_str, 0);
	LOG_EOL(wp);

	/* tell MTA the overall result */
	p = wp->srbuf;
	*p++ = result_char;
	*p++ = '\n';

	/* output list of recipients */
	for (rcpt_st = wp->cw.rcpt_st; rcpt_st; rcpt_st = rcpt_st->fwd) {
		if (p >= &wp->srbuf[sizeof(wp->srbuf)-1]) {
			sock_write_buf(wp, wp->srbuf, sizeof(wp->srbuf)-1);
			p = wp->srbuf;
		}
		if (result_char == DCCIF_RESULT_GREY) {
			*p++ = DCCIF_RCPT_GREY;
		} else if (rcpt_st->flags & RCPT_ST_BLACK ) {
			*p++ = DCCIF_RCPT_REJECT;
			/* tell greylist to restore embargo for those targets
			 * that believe the message was spam and did not
			 * white- or blacklist it */
			if (rcpt_st->grey_result != DCC_GREY_ASK_OFF
			    && rcpt_st->grey_result != DCC_GREY_ASK_FAIL
			    && !dcc_grey_spam(wp->cw.emsg, wp->cw.dcc_ctxt,
					      &wp->cw.cks, rcpt_st->triple_sum))
				cmn_error_msg(&wp->cw, "%s", wp->cw.emsg);
		} else {
			*p++ = DCCIF_RCPT_ACCEPT;
		}
	}
	*p++ = '\n';
	sock_write_buf(wp, wp->srbuf, p-wp->srbuf);

	if (wp->flags & FG_MTA_BODY) {
		hdrs_copy(wp);
		if (wp->cw.header.buf[0] != '\0'
		    && chghdr != DCCIFD_NOHDR) {
			/* write X-DCC */
			sock_write_buf(wp, wp->cw.header.buf,
				       wp->cw.header.used);
			if (wp->cr_hdrs > wp->total_hdrs/2) {
				/* end the X-DCC header with "\r\n" if at least
				 * half of the header lines ended that way */
				sock_write_buf(wp, "\r\n", 2);
			} else {
				/* otherwise use only "\n" */
				sock_write_buf(wp, "\n", 1);
			}
		}
		do {			/* copy body */
			i = wp->srbuf_in - wp->srbuf_out;
			if (i > 0) {
				sock_write_buf(wp, wp->srbuf_out, i);
				wp->srbuf_out = wp->srbuf_in;
			}
		} while (tmp_read_srbuf(wp));

	} else if (wp->flags & FG_MTA_HEADER) {
		/* MTA wants only the header, if we have it */
		if (wp->cw.header.used != 0)
			sock_write_buf(wp, wp->cw.header.buf,
				       wp->cw.header.used);
		sock_write_buf(wp, "\n", 1);
	}

	job_exit(wp);
}



static void
close_main_sock(void)
{
	if (main_sock < 0)
		return;

	unlink_conn_sun();
	if (0 > close(main_sock))
		dcc_error_msg("close(main socket): %s",
			      ERROR_STR());
	main_sock = -1;
}



/* watch for fatal signals */
static void
sigterm(int sig)
{
	stopping = sig;
	unlink_conn_sun();
	dcc_clnt_stop_resolve();
}



void
user_reject(CMN_WORK *cwp, UATTRIB RCPT_ST *rcpt_st,
	    u_char rej_type)		/* 0=rcpt only, 1=all, 2=grey */
{
	if (rej_type == 0) {
		++totals.tgts_discarded;
		--cwp->reject_tgts;
		return;
	}
}



/* read lines of pairs if (env_To, username) values from the MTA */
static void
do_rcpts(WORK *wp)
{
	RCPT_ST *rcpt_st;
	char *rcpt, *user, c, *p;
	int rcpt_len;

	lock_work();
	wp->flags |= FG_WORK_LOCK;
	for (;;) {
		sock_read_line(wp);

		/* stop after the empty line */
		rcpt = wp->srbuf_out;
		if (*rcpt == '\0') {
			wp->srbuf_out = wp->srbuf_next_line;
			break;
		}

		++wp->cw.tgts;

		/* bytes up to '\r' are the env_To value
		 * and bytes after are the local recipient */
		user = strchr(rcpt, '\r');
		if (!user) {
			rcpt_len = wp->srbuf_next_line - wp->srbuf_out;
		} else {
			*user++ = '\0';
			rcpt_len  = user - rcpt;
			if (*user == '\0'
			    || !strcmp(user, "..") || strchr(user, '/'))
				user = 0;
		}
		if (rcpt_len > ISZ(rcpt_st->env_to)) {
			unlock_work();
			cmn_error_msg(&wp->cw, "recipient \"%s\" is too long",
				      rcpt);
			job_exit(wp);
		} else if (rcpt_len == 0) {
			unlock_work();
			cmn_error_msg(&wp->cw, "null recipient");
			job_exit(wp);
		}

		rcpt_st = rcpt_st_alloc(&wp->cw, 0);
		if (!rcpt_st) {
			unlock_work();
			job_exit(wp);
		}

		memcpy(rcpt_st->env_to, rcpt, rcpt_len);

		if (user) {
			/* Convert ASCII upper to lower case.
			 * Be simplistic about international character
			 * sets and avoid locale and portability
			 * complications. */
			p = rcpt_st->user;
			do {
				c = *user++;
				if (c >= 'A' && c <= 'Z')
					c -= 'A' - 'a';
				*p++ = c;
			} while (c != '\0' && p < LAST(rcpt_st->user));
		}

		if (userdirs && user && dcc_have_logdir)
			snprintf(rcpt_st->dir, sizeof(rcpt_st->dir), "%s%s",
				 userdirs, rcpt_st->user);

		wp->srbuf_out = wp->srbuf_next_line;
	}
	unlock_work();
	wp->flags &= ~FG_WORK_LOCK;
}



/* Get the next header.
 *	The line is copied to te temporary file and then null terminated
 *	in the buffer */
static u_char				/* 1=have one, 0=end of headers */
get_hdr(WORK *wp)
{
	int log_len, hdr_len, llen;
	char *bol, *eol;
	u_char at_eol;

	log_len = 0;
	hdr_len = 0;
	at_eol = 0;
	for (;;) {
		/* get another line of the header */
		bol = wp->srbuf_out + hdr_len;
		llen = wp->srbuf_in - bol;
		if (llen < 2) {
			if (hdr_len > DCC_HDR_CK_MAX) {
				if (log_len < hdr_len)
					tmp_write(wp, wp->srbuf_out+log_len,
						  hdr_len - log_len);
				hdr_len = DCC_HDR_CK_MAX;
				log_len = DCC_HDR_CK_MAX;
				if (llen > 0)
					memmove(wp->srbuf_out + DCC_HDR_CK_MAX,
						bol, llen);
				wp->srbuf_in = (wp->srbuf_out + DCC_HDR_CK_MAX
						+ llen);
			}
			if (!sock_read(wp)) {
				/* EOF implies the message body is missing.
				 * Fake newlines until we get out of here
				 * so that we will log whatever we got */
				wp->flags |= FG_MISSING_BODY;
				*wp->srbuf_in++ = '\n';
			}
			continue;
		}

		/* stop before the next line if it is not a continuation
		 * of the current header */
		if (at_eol
		    && *bol != ' '
		    && *bol != '\t') {
			if (log_len < hdr_len)
				tmp_write(wp, wp->srbuf_out+log_len,
					  hdr_len - log_len);
			wp->srbuf_out[hdr_len-1] = '\0';
			wp->srbuf_next_line = &wp->srbuf_out[hdr_len];
			return 1;
		}

		/* find the end of the next line */
		eol = memchr(bol, '\n', llen);
		if (eol) {
			hdr_len += ++eol - bol;
			if (hdr_len == 1
			    || (hdr_len == 2 && *wp->srbuf_out == '\r'))
				return 0;   /* quit at the end of headers */
			at_eol = 1;
		} else {
			hdr_len += llen;
			at_eol = 0;
		}
	}
}



static void
do_hdrs(WORK *wp)
{
	char *p, *p2;
	int i;

	for (;;) {
		/* stop at the separator between the body and headers */
		if (!get_hdr(wp))
			break;

#define GET_HDR_CK(h,t) {						\
			if (!CSTRCMP(wp->srbuf_out, h)) {		\
				dcc_get_cks(&wp->cw.cks, DCC_CK_##t,	\
					    &wp->srbuf_out[STRZ(h)], 1); \
				wp->flags |= FG_SEEN_HDR;		\
				wp->srbuf_out = wp->srbuf_next_line;	\
				continue;}}
		GET_HDR_CK(DCC_XHDR_TYPE_FROM":", FROM);
		GET_HDR_CK(DCC_XHDR_TYPE_MESSAGE_ID":", MESSAGE_ID);
#undef GET_HDR_CK

		if (!CSTRCMP(wp->srbuf_out, "Return-Path:")) {
			if (wp->cw.cks.sums[DCC_CK_ENV_FROM].type
			    == DCC_CK_INVALID) {
				BUFCPY(wp->env_from,
				       &wp->srbuf_out[STRZ("Return-Path:")]);
				dcc_get_cks(&wp->cw.cks, DCC_CK_ENV_FROM,
					    wp->env_from, 1);
			}
			wp->flags |= FG_SEEN_HDR;
			wp->srbuf_out = wp->srbuf_next_line;
			continue;
		}

		/* notice UNIX From_ line */
		if (!(wp->flags & FG_SEEN_HDR)
		    && !strncmp(wp->srbuf_out, "From ", STRZ("From "))) {
			p = &wp->srbuf_out[STRZ("From ")];
			p += strspn(p, " ");
			p2 = strchr(p, ' ');
			if (p2 != 0) {
				if (wp->cw.cks.sums[DCC_CK_ENV_FROM].type
				    == DCC_CK_INVALID) {
					if (p2 > p+sizeof(wp->env_from))
					    p2 = p+sizeof(wp->env_from);
					memcpy(wp->env_from, p, p2-p);
					wp->env_from[p2-p] = '\0';
				}
				wp->flags |= FG_SEEN_HDR;
				wp->srbuf_out = wp->srbuf_next_line;
				continue;
			}
		}

		if (!CSTRCMP(wp->srbuf_out, DCC_XHDR_TYPE_RECEIVED":")) {
			p = &wp->srbuf_out[STRZ(DCC_XHDR_TYPE_RECEIVED":")];

			/* compute checksum of the last Received: header */
			dcc_get_cks(&wp->cw.cks, DCC_CK_RECEIVED, p, 1);

			/* pick IP address out of first Received: header */
			if ((wp->flags & FG_CK_RCVD)
			    && wp->cw.cks.sums[DCC_CK_IP].type == DCC_CK_INVALID
			    && 0 != (p2 = strchr(p, '('))
			    && 0 != (p = strchr(++p2, '['))
			    && (i = strspn(++p, ".:abcdefABCDEF0123456789")) > 6
			    && i < INET6_ADDRSTRLEN
			    && p[i] == ']') {
				p[i] = '\0';
				if (dcc_get_str_ip_ck(&wp->cw.cks, p,
						      &wp->cw.clnt_addr)) {
					if (!DCC_INET_NTOP(AF_INET6,
						    &wp->cw.clnt_addr,
						    wp->cw.clnt_str,
						    sizeof(wp->cw.clnt_str)))
					    strcpy(wp->cw.clnt_str, p);

					i = p-p2-1;
					if (i > ISZ(wp->cw.clnt_name)-1)
					    i = ISZ(wp->cw.clnt_name)-1;
					while (i > 0
					       && (p2[i-1] == ' '
						   || p2[i-1] == '\t'))
					    --i;
					if (i > 0) {
					    memcpy(wp->cw.clnt_name, p2, i);
					    wp->cw.clnt_name[i] = '\0';
					} else {
					    snprintf(wp->cw.clnt_name,
						     sizeof(wp->cw.clnt_name),
						     "[%s]", p);
					}
				}
			}
			wp->flags &= ~FG_CK_RCVD;
			wp->flags |= FG_SEEN_HDR;
			wp->srbuf_out = wp->srbuf_next_line;
			continue;
		}

		/* Notice MIME multipart boundary definitions */
		dcc_ck_mime_hdr(&wp->cw.cks, wp->srbuf_out, 0);

		if (dcc_ck_get_sub(&wp->cw.cks, wp->srbuf_out, 0))
			wp->flags |= FG_SEEN_HDR;

		/* notice any sort of header */
		if (!(wp->flags & FG_SEEN_HDR)) {
			for (p = wp->srbuf_out; ; ++p) {
				if (*p == ':') {
					wp->flags |= FG_SEEN_HDR;
					break;
				}
				if (*p <= ' ' || *p >= 0x7f)
					break;
			}
		}

		wp->srbuf_out = wp->srbuf_next_line;
	}

	/* Create a checksum for a null Message-ID header if there
	 * was no Message-ID header.  */
	if (wp->cw.cks.sums[DCC_CK_MESSAGE_ID].type != DCC_CK_MESSAGE_ID)
		dcc_get_cks(&wp->cw.cks, DCC_CK_MESSAGE_ID, "", 0);
}



/* start a new connection to an SMTP client */
static void *
job_start(void *wp0)
{
	WORK *wp = wp0;
	sigset_t sigsoff;
	RCPT_ST *rcpt_st;
	char *p, *p2, *lim;
	char buf[1024];
	int buflen, i;

	i = pthread_sigmask(SIG_BLOCK, &sigsoff, 0);
	if (i)
		dcc_logbad(EX_SOFTWARE, "pthread_sigmask(): %s",
			   ERROR_STR1(i));
	log_start(&wp->cw);

	/* get any options */
	sock_read_line(wp);
	while (*wp->srbuf_out != '\0') {
		if (sock_ck_str(wp, DCCIF_OPT_SPAM,
				sizeof(DCCIF_OPT_SPAM)-1)) {
			wp->cw.honor |= DCC_HONOR_MTA_ISSPAM;
			continue;
		}
		if (sock_ck_str(wp, DCCIF_OPT_BODY,
				sizeof(DCCIF_OPT_BODY)-1)) {
			wp->flags |= FG_MTA_BODY;
			continue;
		}
		if (sock_ck_str(wp, DCCIF_OPT_HEADER,
				sizeof(DCCIF_OPT_HEADER)-1)) {
			wp->flags |= FG_MTA_HEADER;
			continue;
		}
		if (sock_ck_str(wp, DCCIF_OPT_QUERY,
				sizeof(DCCIF_OPT_QUERY)-1)) {
			wp->flags |= FG_MTA_DCC_QUERY;
			continue;
		}
		if (sock_ck_str(wp, DCCIF_OPT_GREY_QUERY,
				sizeof(DCCIF_OPT_GREY_QUERY)-1)) {
			wp->flags |= FG_MTA_GREY_QUERY;
			continue;
		}
		if (sock_ck_str(wp, DCCIF_OPT_NO_REJECT,
				sizeof(DCCIF_OPT_NO_REJECT)-1)) {
			wp->flags |= FG_MTA_NO_REJECT;
			continue;
		}

		cmn_error_msg(&wp->cw, "unrecognized option value: \"%s\"",
			      wp->srbuf_out);
		job_exit(wp);
	}
	if ((wp->cw.honor & DCC_HONOR_MTA_ISSPAM)
	    && (wp->flags & FG_MTA_DCC_QUERY)) {
		cmn_error_msg(&wp->cw, DCCIF_OPT_SPAM" and "DCCIF_OPT_QUERY
			      " are incompatible");
		wp->cw.honor &= ~DCC_HONOR_MTA_ISSPAM;
	}
	if ((wp->cw.honor & DCC_HONOR_MTA_ISSPAM)
	    && (wp->flags & FG_MTA_GREY_QUERY)) {
		cmn_error_msg(&wp->cw, DCCIF_OPT_SPAM" and "DCCIF_OPT_GREY_QUERY
			      " are incompatible");
		wp->cw.honor &= ~DCC_HONOR_MTA_ISSPAM;
	}
	wp->srbuf_out = wp->srbuf_next_line;

	/* deal with the SMTP client IP address and host name */
	i = sock_read_line(wp);
	if (i == 0) {
		wp->cw.clnt_name[0] = '\0';
		wp->cw.clnt_str[0] = '\0';
		wp->flags |= FG_CK_RCVD;    /* try first Received header */
	} else {
		/* the host name follows the IP address */
		p = strchr(wp->srbuf_out, '\r');
		if (p) {
			*p++ = '\0';
			BUFCPY(wp->cw.clnt_name, p);
		} else {
			wp->cw.clnt_name[0] = '\0';
		}
		/* convert IP address to an IPv6 number */
		if (!dcc_get_str_ip_ck(&wp->cw.cks, wp->srbuf_out,
				       &wp->cw.clnt_addr)) {
			cmn_error_msg(&wp->cw, "unrecognized IP address \"%s\"",
				      wp->srbuf_out);
		} else {
			/* convert IPv6 address to a canonical string */
			if (!DCC_INET_NTOP(AF_INET6, &wp->cw.clnt_addr,
					   wp->cw.clnt_str,
					   sizeof(wp->cw.clnt_str))) {
				cmn_error_msg(&wp->cw,
					      "inet_ntop(addr of %s) failed",
					      wp->srbuf_out);
				strcpy(wp->cw.clnt_str, "(inet_ntop() ?)");
			}
		}
	}
	wp->srbuf_out = wp->srbuf_next_line;

	/* deal with the HELO value */
	i = sock_read_line(wp);
	if (i > DCC_HELO_MAX-1) {
		i = DCC_HELO_MAX-1;
		strcpy(&wp->srbuf_out[DCC_HELO_MAX-ISZ(DCC_HELO_CONT)],
		       DCC_HELO_CONT);
	}
	memcpy(wp->cw.helo, wp->srbuf_out, i+1);
	wp->srbuf_out = wp->srbuf_next_line;

	/* get the envelope mail_from value */
	i = sock_read_line(wp);
	if (i > ISZ(wp->env_from)-1) {
		i = ISZ(wp->env_from)-1;
		wp->srbuf_out[ISZ(wp->env_from)] = '\0';
	}
	memcpy(wp->env_from, wp->srbuf_out, i+1);
	if (wp->env_from[0] != '\0')
		dcc_get_cks(&wp->cw.cks, DCC_CK_ENV_FROM,
			    wp->env_from, 1);
	wp->srbuf_out = wp->srbuf_next_line;

	/* get the recipients */
	do_rcpts(wp);

	/* We must make a copy of the message in a temporary file
	 * if the MTA wants a copy with the X-DCC header added
	 * Copy the headers to a temporary file because the official
	 * log file needs the SMTP client IP address and envelope information
	 * before the header lines.  The log file needs all of the header
	 * lines including stray X-DCC lines, but those lines must be removed
	 * from the output file. */
	wp->tmp_fd = dcc_mkstemp(wp->cw.emsg, wp->tmp_nm, sizeof(wp->tmp_nm),
				 tmpdir, "/tmp.", 1, 0, 0);
	if (wp->tmp_fd < 0) {
		dcc_error_msg("%s", wp->cw.emsg);
		job_close(wp);
	}

	/* get ready for the body checksums before the headers so that
	 * we can notice the MIME separator */
	dcc_ck_body_init(&wp->cw.cks);

	/* open the connection to the nearest DCC server
	 * and figure out our X-DCC header */
	if (!ck_dcc_ctxt(&wp->cw)) {
		/* failed to create context */
		wp->cw.dcc_ctxt_sn = 0;
		job_done(wp, DCCIF_RESULT_TEMP, "temporary failure");
	}

	/* get the headers */
	do_hdrs(wp);

	/* log IP address, env_From, and env_To values now
	 * that we (may) have collected from the headers */
	if (wp->cw.cks.sums[DCC_CK_IP].type != DCC_CK_INVALID) {
		log_print(&wp->cw, DCC_XHDR_TYPE_IP": %s %s\n",
			  wp->cw.clnt_name, wp->cw.clnt_str);
	}
	if (wp->cw.helo[0] != '\0')
		log_print(&wp->cw, "HELO: %s\n", wp->cw.helo);
	dcc_ck_get_sub(&wp->cw.cks, "helo", wp->cw.helo);
	if (wp->env_from[0] != '\0') {
		LOG_CAPTION(wp, DCC_XHDR_TYPE_ENV_FROM": ");
		log_write(&wp->cw, wp->env_from, 0);
		dcc_get_cks(&wp->cw.cks, DCC_CK_ENV_FROM, wp->env_from, 1);

		LOG_CAPTION(wp, "  mail_host=");
		p = strchr(wp->env_from, '@');
		if (p) {
			++p;
			p2 = strchr(p, '>');
			if (p2)
				*p2 = '\0';
			log_write(&wp->cw, p, 0);
			dcc_ck_get_sub(&wp->cw.cks, "mail_host", p);
		}
		LOG_EOL(wp);
	}
	wp->cw.log_pos_to_first = log_lseek(&wp->cw, SEEK_END);
	wp->cw.log_pos_to_end = wp->cw.log_pos_to_first;
	for (rcpt_st = wp->cw.rcpt_st; rcpt_st; rcpt_st = rcpt_st->fwd) {
		rcpt_st->log_pos_to = wp->cw.log_pos_to_end;
		LOG_CAPTION(wp, DCC_XHDR_TYPE_ENV_TO": ");
		log_write(&wp->cw, rcpt_st->env_to, 0);
		LOG_CAPTION(wp, "  addr=");
		log_write(&wp->cw, rcpt_st->user, 0);
		LOG_CAPTION(wp, "  dir=");
		log_write(&wp->cw, rcpt_st->dir, 0);
		LOG_EOL(wp);
		wp->cw.log_pos_to_end = log_lseek(&wp->cw, SEEK_END);
	}

	/* log the blank line between the header and the body */
	LOG_EOL(wp);

	/* copy headers from tmp file to log file */
	if (wp->cw.log_fd >= 0) {
		if (0 > lseek(wp->tmp_fd, 0, SEEK_SET)) {
			cmn_error_msg(&wp->cw, "rewind %s: %s",
				      wp->tmp_nm, ERROR_STR());
			job_exit(wp);
		}
		for (;;) {
			buflen = tmp_read(wp, buf, sizeof(buf));
			if (buflen <= 0)
				break;
			log_write(&wp->cw, buf, buflen);
		}
	}
	if (!(wp->flags & FG_MTA_BODY)) {
		if (0 > close(wp->tmp_fd))
			cmn_error_msg(&wp->cw, "close(%s): %s",
				      wp->tmp_nm, ERROR_STR());
		wp->tmp_fd = -1;
	}

	if (wp->flags & FG_MISSING_BODY) {
		cmn_error_msg(&wp->cw, "missing message body");
		job_exit(wp);
	}

	/* collect the body */
	for (;;) {
		buflen = wp->srbuf_in - wp->srbuf_out;
		if (buflen <= 0) {
			if (!sock_read(wp))
				break;
			buflen = wp->srbuf_in - wp->srbuf_out;
		}

		/* Log the body block
		 * avoid filling the disk by truncating the body */
		if (wp->cw.log_fd >= 0) {
#if MAX_LOG_SIZE > 0
			i = MAX_LOG_SIZE - wp->cw.log_size;
			if (i >= 0) {
				if (i >= buflen) {
					log_write(&wp->cw,
						  wp->srbuf_out, buflen);
				} else {
					p = wp->srbuf_out+i;
					lim = wp->srbuf_out;
					p = lim+i;
					if (i > 80)
					    lim += i-80;
					while (--p > lim) {
					    if (*p == '\n') {
						i = p-wp->srbuf_out+1;
						break;
					    }
					}
					log_write(&wp->cw, wp->srbuf_out, i);
					if (wp->srbuf_out[i-1] != '\n')
					    LOG_EOL(wp);
					LOG_CAPTION(wp, DCC_LOG_TRN_MSG_CR);
					wp->cw.log_size = MAX_LOG_SIZE+1;
				}
			}
#else
			log_write(&wp->cw, wp->srbuf_out, buflen);
#endif
		}

		if (wp->flags & FG_MTA_BODY)
			tmp_write(wp, wp->srbuf_out, buflen);

		dcc_ck_body(&wp->cw.cks, wp->srbuf_out, buflen);
		wp->srbuf_out = wp->srbuf_in;
	}
	dcc_ck_body_fin(&wp->cw.cks);

	LOG_CAPTION(wp, DCC_LOG_MSG_SEP);

	if (wp->cw.honor & DCC_HONOR_MTA_ISSPAM)
		LOG_CAPTION(wp, "MTA-->spam\n");

	/* check the grey and white lists */
	cmn_ask_white(&wp->cw,
		      (wp->flags & FG_MTA_GREY_QUERY) != 0);

	LOG_CAPTION(wp, "\n");
	wp->cw.header.buf[0] = '\0';
	wp->cw.header.used = 0;
	if (wp->cw.tgts != wp->cw.white_tgts || wp->cw.tgts == 0) {
		/* Report to the DCC and add our header if allowed.
		 * After serious errors, act as if DCC server said not-spam */
		if (!cmn_ask_dcc(&wp->cw,
				 (wp->flags & FG_MTA_DCC_QUERY) != 0)
		    && try_extra_hard)
			job_done(wp, DCCIF_RESULT_TEMP, "temporary failure");

	} else {
		/* if not ok to ask, remove our X-DCC header */
		wp->cw.cks.flags &= ~DCC_CKS_HAVE_SUM;
		/* reject and/or log it if the target count is high enough */
		dcc_honor_log_cnts(&wp->cw.honor, &wp->cw.cks,
				   ((wp->cw.honor & DCC_HONOR_LOCAL_ISSPAM)
				    ? DCC_TGTS_TOO_MANY : wp->cw.tgts));
	}

	totals.tgts += wp->cw.tgts;
	users_process(&wp->cw);

	/* deliver it if all remaining targets want it */
	totals.tgts_rejected += wp->cw.reject_tgts;

	if ((wp->cw.reject_tgts != 0
	     || (wp->cw.tgts == 0 && (wp->cw.honor & DCC_HONOR_SRVR_ISSPAM)))
	    && !(wp->flags & FG_MTA_NO_REJECT))
		job_done(wp, DCCIF_RESULT_REJECT, "reject");

	if (wp->cw.honor & DCC_HONOR_GREY_EMBARGO) {
		totals.tgts_embargoed += wp->cw.tgts;
		job_done(wp, DCCIF_RESULT_GREY, "temporary greylist embargo");
	}

	if (wp->cw.deliver_tgts == wp->cw.tgts)
		job_done(wp, DCCIF_RESULT_OK, "accept");
	job_done(wp, DCCIF_RESULT_SOME, "accept for some");
}
