/* Copyright (C) 2004 - 2006  db4objects Inc.  http://www.db4o.com

This file is part of the db4o open source object database.

db4o is free software; you can redistribute it and/or modify it under
the terms of version 2 of the GNU General Public License as published
by the Free Software Foundation and as clarified by db4objects' GPL 
interpretation policy, available at
http://www.db4o.com/about/company/legalpolicies/gplinterpretation/
Alternatively you can write to db4objects, Inc., 1900 S Norfolk Street,
Suite 350, San Mateo, CA 94403, USA.

db4o is distributed in the hope that it will be useful, but WITHOUT ANY
WARRANTY; without even the implied warranty of MERCHANTABILITY or
FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License along
with this program; if not, write to the Free Software Foundation, Inc.,
59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. */
namespace Db4objects.Db4o.Config
{
	/// <summary>
	/// Wildcard Alias functionality to create aliases for packages,
	/// namespaces or multiple similar named classes.
	/// </summary>
	/// <remarks>
	/// Wildcard Alias functionality to create aliases for packages,
	/// namespaces or multiple similar named classes. One single '*'
	/// wildcard character is supported in the names.
	/// <br /><br />See
	/// <see cref="Db4objects.Db4o.Config.IAlias">Db4objects.Db4o.Config.IAlias</see>
	/// for concrete examples.
	/// </remarks>
	public class WildcardAlias : Db4objects.Db4o.Config.IAlias
	{
		private readonly Db4objects.Db4o.Config.WildcardAlias.WildcardPattern _storedPattern;

		private readonly Db4objects.Db4o.Config.WildcardAlias.WildcardPattern _runtimePattern;

		public WildcardAlias(string storedPattern, string runtimePattern)
		{
			if (null == storedPattern)
			{
				throw new System.ArgumentNullException("storedPattern");
			}
			if (null == runtimePattern)
			{
				throw new System.ArgumentNullException("runtimePattern");
			}
			_storedPattern = new Db4objects.Db4o.Config.WildcardAlias.WildcardPattern(storedPattern
				);
			_runtimePattern = new Db4objects.Db4o.Config.WildcardAlias.WildcardPattern(runtimePattern
				);
		}

		/// <summary>resolving is done through simple pattern matching</summary>
		public virtual string ResolveRuntimeName(string runtimeTypeName)
		{
			string match = _runtimePattern.Matches(runtimeTypeName);
			return match != null ? _storedPattern.Inject(match) : null;
		}

		/// <summary>resolving is done through simple pattern matching</summary>
		public virtual string ResolveStoredName(string storedTypeName)
		{
			string match = _storedPattern.Matches(storedTypeName);
			return match != null ? _runtimePattern.Inject(match) : null;
		}

		internal class WildcardPattern
		{
			private string _head;

			private string _tail;

			public WildcardPattern(string pattern)
			{
				string[] parts = Split(pattern);
				_head = parts[0];
				_tail = parts[1];
			}

			public virtual string Inject(string s)
			{
				return _head + s + _tail;
			}

			public virtual string Matches(string s)
			{
				if (!s.StartsWith(_head) || !s.EndsWith(_tail))
				{
					return null;
				}
				return Sharpen.Runtime.Substring(s, _head.Length, s.Length - _tail.Length);
			}

			private void InvalidPattern()
			{
				throw new System.ArgumentException("only one '*' character");
			}

			internal virtual string[] Split(string pattern)
			{
				int index = pattern.IndexOf('*');
				if (-1 == index || index != pattern.LastIndexOf('*'))
				{
					InvalidPattern();
				}
				return new string[] { Sharpen.Runtime.Substring(pattern, 0, index), Sharpen.Runtime.Substring
					(pattern, index + 1) };
			}
		}
	}
}
