/*
 * Copyright (c) 2008 Cyrille Berger <cberger@cberger.net>
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either 
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public 
 * License along with this library.  If not, see <http://www.gnu.org/licenses/>. */

#include "ListOptionsModel.h"

#include <QPixmap>
#include <QSignalMapper>

#include <KConfigGroup>
#include <kdebug.h>
#include <KLocale>
#include <KIcon>

#include "ProcessingOptions.h"
#include "ProcessingOptionsInfo.h"

struct ListOptionsModel::Private {
  QList< ProcessingOptionsInfo* > infos;
  QSignalMapper* dataMapper;
};


ListOptionsModel::ListOptionsModel( ) : d(new Private) 
{
  d->dataMapper = 0;
}

ListOptionsModel::~ListOptionsModel()
{
  delete d;
}

int ListOptionsModel::rowCount ( const QModelIndex & parent) const
{
  Q_UNUSED(parent);
  return d->infos.size();
}

QVariant ListOptionsModel::data ( const QModelIndex & index, int role ) const
{
  switch( role )
  {
    case Qt::DisplayRole:
    {
      return d->infos[ index.row() ]->name();
    }
    case Qt::EditRole:
    {
      return d->infos[ index.row() ]->name();
    }
    case Qt::DecorationRole:
    {
      return d->infos[ index.row() ]->preview();
    }
    case 100:
    {
      return qVariantFromValue(d->infos[ index.row() ]->processingOptions() );
    }
  }
  return QVariant();
}

bool ListOptionsModel::setData ( const QModelIndex & index, const QVariant & value, int role )
{
  switch( role )
  {
    case Qt::EditRole:
    {
      QString s = value.toString();
      if( s != "" and not hasOptionNamed( s ) )
      {
        d->infos[ index.row() ]->setName( s );
        emit(dataChanged( index, index));
        emit(optionsChanged());
        return true;
      } else {
        return false;
      }
      break;
    }
    default:
      break;
  }
  return true;
}

Qt::ItemFlags ListOptionsModel::flags ( const QModelIndex & /*index*/ ) const
{
  return Qt::ItemIsSelectable | Qt::ItemIsEditable | Qt::ItemIsEnabled;
}

void ListOptionsModel::saveOption( const QString& _name, const ProcessingOptions& _options, const QPixmap& _preview)
{
  beginInsertRows( QModelIndex(), d->infos.size(), d->infos.size());
  QString name = _name;
  if( hasOptionNamed( name ) ) name = "";
  if( name == "" )
  {
    for(int i = 1;true; ++i)
    {
      name = i18n("Unnamed %1", i);
      if( not hasOptionNamed( name ) )
      {
        break;
      }
    }
  }
  d->infos.push_back( new ProcessingOptionsInfo( name, _options, _preview ) );
  endInsertRows();
  emit(optionsChanged());
}

void ListOptionsModel::removeBookmarkAt( const QModelIndex & index )
{
  beginRemoveRows( index.parent(), index.row(), index.row() );
  delete d->infos[ index.row() ];
  d->infos.removeAt( index.row() );
  endRemoveRows();
  emit(optionsChanged());
}

void ListOptionsModel::changeRawImageInfo( RawImageInfoSP _info )
{
  updateMapper();
  foreach( ProcessingOptionsInfo* info, d->infos )
  {
    info->changeRawImageInfo( _info );
  }
}

void ListOptionsModel::updateMapper()
{
  delete d->dataMapper;
  d->dataMapper = new QSignalMapper( this );
  for(int i = 0; i < d->infos.size(); ++i)
  {
    connect(d->infos[i], SIGNAL(previewChanged()), d->dataMapper, SLOT(map()));
    d->dataMapper->setMapping( d->infos[i], i );
  }
  connect(d->dataMapper, SIGNAL(mapped(int)), SLOT(dataUpdated(int)));
}

void ListOptionsModel::dataUpdated(int _index)
{
  kDebug() << "dataUpdated";
  QModelIndex mi = createIndex(_index, 0);
  emit( dataChanged(mi, mi));
}

void ListOptionsModel::saveToConfigGroup( KConfigGroup& group ) const
{
  group.deleteGroup();
  foreach( const ProcessingOptionsInfo* info, d->infos )
  {
    group.writeEntry( info->name(), QVariant(info->processingOptions().toXml()));
  }
}

void ListOptionsModel::loadFromConfigGroup( const KConfigGroup& group )
{
  if( not d->infos.empty() )
  {
    beginRemoveRows( QModelIndex(), 0, d->infos.size() - 1 );
    qDeleteAll( d->infos );
    d->infos.clear();
    endRemoveRows();
  }
  if( not group.keyList().empty() )
  {
    beginInsertRows( QModelIndex(), 0, group.keyList().size() - 1 );
    foreach( const QString& key, group.keyList() )
    {
      ProcessingOptions options;
      QString xml = group.readEntry( key, "");
      options.fromXml( xml );
      d->infos.push_back( new ProcessingOptionsInfo( key, options, KIcon("image-x-dcraw").pixmap(100)));
    }
    endInsertRows();
  }
}

bool ListOptionsModel::hasOptionNamed( const QString& name )
{
  foreach( ProcessingOptionsInfo* info, d->infos )
  {
    if(info->name() == name) return true;
  }
  return false;
}

#include "ListOptionsModel.moc"
