#!/usr/bin/env perl

# Some tests for proper handling of filepaths

use lib 'lib/perl';
use Test::More 'no_plan';
use Test::Darcs;
use Shell::Command;
use strict;
use warnings;
use Cwd;
use File::Temp 'tempdir';
my $work_dir = tempdir( CLEANUP => 1 );
chdir $work_dir;

mkpath 'temp1';

my $repo_flag = '--repodir=temp1';
my $test_name = 'Make sure that init works with --repodir';
darcs "init $repo_flag";
ok((-d 'temp1/_darcs'), '_darcs directory was created');

# add some meat to that repository 
chdir 'temp1';
touch 'baz';
darcs qw( add baz ) ;
darcs qw( record  -m moo -a ) ;
chdir '../';

# ----------------------------------------------------------------------
# local vs remote filepaths
# ----------------------------------------------------------------------
like( darcs("get temp1 temp2"), qr/Finished getting/i );
chdir 'temp2';
mkpath 'dir';
darcs "add dir";
chdir 'dir';
`touch foo:bar`;
unlike( darcs("add foo:bar"), qr/failed/i );
chdir '../..';
cleanup 'temp2';

# ----------------------------------------------------------------------
# repodir stuff
# ----------------------------------------------------------------------
mkpath 'temp1/non-darcs';
$test_name = '--repodir is not recursive';
like( darcs("get temp1/non-darcs"), qr/Not a repository/i, $test_name );
cleanup 'temp1/non-darcs';
cleanup 'non-darcs';

$test_name = 'get accepts --repodir.';
like( darcs("get --repodir=temp2 temp1"), qr/Finished getting/i, $test_name );
ok((-d 'temp2/_darcs'), '_darcs directory was created');
cleanup 'temp2';
$test_name = 'get accepts absolute --repodir.';
like( darcs(q{get --repodir="}.cwd().q{/temp2" temp1}), qr/Finished getting/i, $test_name );
ok((-d 'temp2/_darcs'), '_darcs directory was created');

$test_name = 'changes accepts --repodir.';
like( darcs("changes $repo_flag"), qr/moo/i, $test_name );
$test_name = 'changes accepts absolute --repo.';
like( darcs(q{changes --repo="}.cwd().q{/temp1"}), qr/moo/i, $test_name );
$test_name = 'changes accepts relative --repo.';
like( darcs("changes --repo=temp1"), qr/moo/i, $test_name );
$test_name = "[issue467] context --repodir";
like ( darcs("changes --context --repodir=temp1"), qr/Context:/, $test_name );

$test_name = 'dist accepts --repodir.';
like( darcs("dist $repo_flag"), qr/Created dist/i, $test_name );

$test_name = 'optimize accepts --repodir.';
like( darcs("optimize --reorder-patches $repo_flag"), qr/done optimizing/i, $test_name );

$test_name = 'repair accepts --repodir.';
like( darcs("repair $repo_flag"), qr/already consistent/i, $test_name );

$test_name = 'replace accepts --repodir.';
like( darcs("replace $repo_flag foo bar"), qr//i, $test_name );

$test_name = 'setpref accepts --repodir.';
like( darcs("setpref $repo_flag test echo"), qr/Changing value of test/i, $test_name );

$test_name = 'trackdown accepts --repodir.';
like( darcs("trackdown $repo_flag"), qr/Success!/i, $test_name );

# ----------------------------------------------------------------------
# converting between absolute and relative paths
# ----------------------------------------------------------------------

like( darcs("get temp1 temp3"), qr/Finished getting/i );
chdir 'temp3';
mkpath 'a/b';
chdir '..';

sub change_something_record {
  my $hello = shift;
  open BAZ, ">>baz";
  print BAZ "hello $hello";
  close BAZ;
  darcs("record  -m hello$hello -a");
}
chdir 'temp2';
change_something_record 1;
change_something_record 2;
chdir '..';

$test_name = 'can handle .. path';
chdir 'temp3';
like( darcs("pull ../temp2 -p1 --all"), qr/Finished pulling/, $test_name );
like( darcs("pull --dry-run"), qr/hello2/, "$test_name with defaultrepo" );
chdir 'a/b';
like( darcs("pull --dry-run"), qr/hello2/, "[issue268] $test_name with subdir" );

chdir '/';
