-- Copyright (C) 2004 David Roundy
--
-- This program is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2, or (at your option)
-- any later version.
--
-- This program is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
-- GNU General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software Foundation,
-- Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

module Main (main) where

import Repository ( read_repo )
import PatchInfo ( pi_author )

import Data.List ( sort, group )
import System ( getProgName, getArgs )
import Monad ( liftM )

intro :: IO String
intro = do
  prog_name <- getProgName
  return $
    "This is a list of ALL contributors to darcs, sorted according to the\n"++
    "number of patches they have contributed.  This list is automatically\n"++
    "created from the darcs repository, so if there are problems with it,\n"++
    "changes should be made to list_authors.hs.\n"++
    "Say \"" ++ prog_name ++ " stats\" for more detailed information.\n"++
    "\n"

main :: IO ()
main = do darcs_history <- read_repo "."
          intro >>= putStr
          use_statistics <- elem "stats" `liftM` getArgs
          mapM_ putStrLn $ sort_authors use_statistics $ map (pi_author.fst)
                         $ concat darcs_history

droundy :: String
droundy = "David Roundy <droundy@abridgegame.org>"

canonize_author :: String -> String
canonize_author "David" = droundy
canonize_author "brian@brianweb.net" = "Brian Alliet <brian@brianweb.net>"
canonize_author "jan@informatik.unibw-muenchen.de"
    = "Jan Scheffczyk <jan@informatik.unibw-muenchen.de>"
canonize_author "egli@apache.org" = "Christian Egli <egli@apache.org>"
canonize_author "simon@joyful.com" = "Simon Michael <simon@joyful.com>"
canonize_author "ralph@inputplus.co.uk"
    = "Ralph Corderoy <ralph@inputplus.co.uk>"
canonize_author "jch@pps.jussieu.fr"
    = "Juliusz Chroboczek <jch@pps.jussieu.fr>"
canonize_author "mark@summersault.com"
    = "Mark Stosberg <mark@summersault.com>"
canonize_author "Tomasz Zielonka <t.zielonka@students.mimuw.edu.pl>"
    = "Tomasz Zielonka <tomasz.zielonka@gmail.com>"
canonize_author "Tomasz Zielonka <tomekz@gemius.pl>"
    = "Tomasz Zielonka <tomasz.zielonka@gmail.com>"
canonize_author "aj@azure.humbug.org.au"
    = "Anthony Towns <aj@azure.humbug.org.au>"
canonize_author "peter@zarquon.se" = "Peter Strand <peter@zarquon.se>"
canonize_author "schnetter@aei.mpg.de"
    = "Erik Schnetter <schnetter@aei.mpg.de>"
canonize_author "fw@deneb.enyo.de" = "Florian Weimer <fw@deneb.enyo.de>"
canonize_author "wnoise@ofb.net" = "Aaron Denney <wnoise@ofb.net>"
canonize_author "kannan@cakoose.com" = "Kannan Goundan <kannan@cakoose.com>"
canonize_author "zander@kde.org" = "Thomas Zander <zander@kde.org>"
canonize_author "rho@swiftdsl.com.au" = "Nigel Rowe <rho@swiftdsl.com.au>"
canonize_author "ptp@lysator.liu.se" = "Tommy Pettersson <ptp@lysator.liu.se>"
canonize_author s
    | s `contains` "roundy" = droundy
    | s `contains` "igloo" = "Ian Lynagh <igloo@earth.li>"
    | s `contains` "Thomas Zander" = "Thomas Zander <zander@kde.org>"
    | s `contains` "marnix.klooster@" = "Marnix Klooster <marnix.klooster@gmail.com>"
    | s `contains` "mklooster@" = "Marnix Klooster <marnix.klooster@gmail.com>"
    | s `contains` "mirian@" = "Mirian Crzig Lennox <mirian@cosmic.com>"
    | s `contains` "zbrown" = "Zack Brown <zbrown@tumblerings.org>"
    | otherwise = s

sort_authors :: Bool -> [String] -> [String]
sort_authors use_stats as = reverse $ map shownames $ sort $
                            map (\s -> (length s,head s)) $
                            group $ sort $ map canonize_author as
        where shownames (n, a) = if use_stats
                                 then show n ++ "\t" ++ a
                                 else a

contains :: String -> String -> Bool
a `contains` b | length a < length b = False
               | take (length b) a == b = True
               | otherwise = tail a `contains` b
