%  Copyright (C) 2003-2004 David Roundy
%
%  This program is free software; you can redistribute it and/or modify
%  it under the terms of the GNU General Public License as published by
%  the Free Software Foundation; either version 2, or (at your option)
%  any later version.
%
%  This program is distributed in the hope that it will be useful,
%  but WITHOUT ANY WARRANTY; without even the implied warranty of
%  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%  GNU General Public License for more details.
%
%  You should have received a copy of the GNU General Public License
%  along with this program; if not, write to the Free Software Foundation,
%  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
\subsection{darcs unrevert}\label{unrevert}
\begin{code}
module Unrevert ( unrevert, remove_from_unrevert_context, write_unrevert ) where
import System ( ExitCode(..), exitWith )
import Monad ( liftM )
import List ( (\\) )

import DarcsCommands ( DarcsCommand(..), nodefaults )
import DarcsArguments ( DarcsFlag( Unified ),
                        verbose, ignoretimes, working_repo_dir,
                        all_gui_interactive,
                      )
import Directory ( removeFile )
import Repository ( PatchSet, read_repo, am_in_repo, slurp_recorded,
                    read_pending, write_pending,
                    sync_repo, slurp_recorded_and_unrecorded,
                  )
import Patch ( Patch, join_patches, apply_to_slurpy, readPatchPS,
               patch2patchinfo, commute, namepatch, flatten_to_primitives,
             )
import SelectChanges ( with_selected_changes_to_files )
import PatchInfo ( PatchInfo, patchinfo, readPatchInfoPS )
import SlurpDirectory ( Slurpy, slurp_write_dirty )
import FastPackedString ( PackedString, readFilePS, nilPS, nullPS )
import Lock ( withRepoLock, writeDocBinFile )
import Pull ( merge_with_us_and_pending )
import Depends ( get_common_and_uncommon )
import Resolution ( standard_resolution )
import DarcsUtils ( askUser, catchall )
import PatchBundle ( scan_bundle, make_bundle )
import SelectChanges ( promptChar )
import IsoDate ( getIsoDateTime )
import SignalHandler ( withSignalsBlocked )
#include "impossible.h"
\end{code}
\begin{code}
unrevert_description :: String
unrevert_description =
 "Undo the last revert (may fail if changes after the revert)."
\end{code}

\options{unrevert}

\haskell{unrevert_help}
\begin{code}
unrevert_help :: String
unrevert_help =
 "Unrevert is used to undo the results of a revert command. It is only\n"++
 "guaranteed to work properly if you haven't made any changes since the\n"++
 "revert was performed.\n"
\end{code}
The command makes a best effort to merge the unreversion with any changes
you have since made.  In fact, unrevert should even work if you've recorded
changes since reverting.
\begin{code}
unrevert :: DarcsCommand
unrevert = DarcsCommand {command_name = "unrevert",
                         command_help = unrevert_help,
                         command_description = unrevert_description,
                         command_extra_args = 0,
                         command_extra_arg_help = [],
                         command_command = unrevert_cmd,
                         command_prereq = am_in_repo,
                         command_get_arg_possibilities = return [],
                         command_argdefaults = nodefaults,
                         command_darcsoptions = [verbose, ignoretimes,
                                                 all_gui_interactive,
                                                 working_repo_dir]}
\end{code}
\begin{code}
unrevert_cmd :: [DarcsFlag] -> [String] -> IO ()
unrevert_cmd opts [] = withRepoLock $ do
  us <- read_repo "."
  them <- unrevert_patch_bundle
  (rec, working) <- slurp_recorded_and_unrecorded "."
  case get_common_and_uncommon (us, them) of
    (_, us', them') -> do
      (_, work_patch) <- merge_with_us_and_pending opts
                         (map (fromJust.snd) $ reverse $ head us',
                          map (fromJust.snd) $ reverse $ head them')
      pw_resolved <- join_patches `liftM` standard_resolution work_patch
      with_selected_changes_to_files "unrevert" opts working
                            [] (flatten_to_primitives pw_resolved) $
                            \ (skipped, p) ->
        case apply_to_slurpy (join_patches p) working of
          Nothing -> fail "Error applying unrevert to working directory..."
          Just work' -> do
              pend <- read_pending
              let pend_and_p = case pend of
                               Nothing -> join_patches p
                               Just pending -> join_patches (pending : p)
              withSignalsBlocked $
                do write_unrevert pend_and_p skipped rec
                   slurp_write_dirty [] work'
                   write_pending pend_and_p
              sync_repo
              putStrLn "Finished unreverting."
unrevert_cmd _ _ = impossible
\end{code}

\begin{code}
write_unrevert :: Patch -> [Patch] -> Slurpy -> IO ()
write_unrevert _ [] _ =
    do removeFile "_darcs/patches/unrevert" `catchall` return ()
       removeFile "_darcs/patches/unrevert_context" `catchall` return ()
write_unrevert pend ps rec =
    case commute (join_patches ps, pend) of
    Nothing -> do really <- askUser "This operation will not be unrevertable! Proceed? "
                  case really of ('y':_) -> return ()
                                 _ -> exitWith $ ExitSuccess
                  removeFile "_darcs/patches/unrevert_context"
                                 `catchall` return ()
                  removeFile "_darcs/patches/unrevert" `catchall` return()
    Just (_, p') -> do
        rep <- read_repo "."
        case get_common_and_uncommon (rep,rep) of
            (common,_,_) -> do
                removeFile "_darcs/patches/unrevert_context"
                               `catchall` return ()
                date <- getIsoDateTime
                writeDocBinFile "_darcs/patches/unrevert" $
                             make_bundle [Unified] rec common
                            [namepatch date "unrevert" "anonymous" [] p']
\end{code}

\begin{code}
unrevert_patch_bundle :: IO PatchSet
unrevert_patch_bundle = do
  c <- readFilePS "_darcs/patches/unrevert_context"
       `catchall` return nilPS
  pf <- readFilePS "_darcs/patches/unrevert"
        `catchall` fail "There's nothing to unrevert!"
  if nullPS c
     then case scan_bundle pf of
          Right ps -> return ps
          Left err -> fail $ "Couldn't parse unrevert patch:\n" ++ err
     else return [(patchinfo "now" "unreversion" "me" [],
                   fst `liftM` readPatchPS pf) : zip (get_context c) nothings]

nothings :: [Maybe a]
nothings = Nothing : nothings
get_context :: PackedString -> [PatchInfo]
get_context ps =
    case readPatchInfoPS ps of
    Just (pinfo,r') ->
        case get_context r' of
        pis -> pinfo:pis
    Nothing -> []
\end{code}

\begin{code}
remove_from_unrevert_context :: [Patch] -> IO ()
remove_from_unrevert_context ps = do
  bundle <- unrevert_patch_bundle `catchall` return [[]]
  case bundle of
    [[]] -> return ()
    _ -> do
    ref <- read_repo "."
    case get_common_and_uncommon (bundle, ref) of
        (common,[[(_, Just us)]],[[]]) ->
            case commute (us, join_patches ps) of
            Nothing -> unrevert_impossible
            Just (_, us') -> do
                s <- slurp_recorded "."
                writeDocBinFile "_darcs/patches/unrevert" $
                             make_bundle [] s
                             (common \\ pis) [us']
        (common,[[(_, Just _)]],_)
            | any (`elem` common) pis -> unrevert_impossible
            | otherwise -> return ()
        _ -> unrevert_impossible
    where unrevert_impossible =
              do yorn <- promptChar
                         "This operation will make unrevert impossible!\nProceed?"
                         "yn"
                 case yorn of
                     'n' -> fail "Cancelled to avoid unrevert catastrophe!"
                     'y' -> do removeFile "_darcs/patches/unrevert_context"
                                 `catchall` return ()
                               removeFile "_darcs/patches/unrevert"
                                 `catchall` return ()
                     _ -> impossible
          pis = map (fromJust . patch2patchinfo) ps
\end{code}
