/* TableViewController.m - this file is part of Cynthiune
 *
 * Copyright (C) 2002, 2003 Wolfgang Sourdeau
 *
 * Author: Wolfgang Sourdeau <wolfgang@contre.com>
 *
 * This file is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This file is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#import <AppKit/AppKit.h>

#import "TableViewController.h"
#import "Playlist.h"
#import "Song.h"
#import "FormatTester.h"
#import "utils.h"

#if defined (NeXT_PDO) || defined (__APPLE__)
#define _(X) NSLocalizedString (X, nil)
#define NSDebugLog
#endif

static NSString *CynthiunePlaylistDragType = @"CynthiunePlaylistDragType";

NSString *TableDoubleClickNotification = @"TableDoubleClickNotification";
NSString *TableSelectionDidChangeNotification = @"TableSelectionDidChangeNotification";
NSString *TableSelectionIsChangingNotification = @"TableSelectionIsChangingNotification";

@implementation TableViewController : NSObject

- (id) init
{ 
  if ((self = [super init]))
    {
      nc = [NSNotificationCenter defaultCenter];
    }

  return self;
}

/* nib and gui init stuff... */

- (void) awakeFromNib
{
#ifndef NeXT_PDO
  [self deleteAllColumns];
  [self addNeededColumns];
  [playlistView setAllowsColumnResizing: NO];
  [playlistView setAutoresizesAllColumnsToFit: NO];
  [playlistView setVerticalMotionCanBeginDrag: YES];
#endif /* NeXT_PDO */
  [playlistView registerForDraggedTypes:
                  [NSArray arrayWithObjects: NSFilenamesPboardType,
                           CynthiunePlaylistDragType, nil]];
  [playlistView setDataSource: self];
  [playlistView setTarget: self];
  [playlistView setDelegate: self];
  [playlistView setDoubleAction: @selector(doubleClick:)];
  [playlistView sizeToFit];

  scrollView = [[playlistView superview] superview];
  [nc addObserver: self
      selector: @selector(scrollViewDidResize:)
      name: NSViewFrameDidChangeNotification
      object: scrollView];
}

/* delegator */
- (void) setDelegate: (id) anObject
{
  if (_delegate)
    [nc removeObserver: _delegate name: nil object: self];

  _delegate = anObject;

  if ([_delegate respondsToSelector: @selector(tableDoubleClick:)])
    [nc addObserver: _delegate
	selector: @selector (tableDoubleClick:)
	name: TableDoubleClickNotification
	object: self];
  if ([_delegate respondsToSelector: @selector (tableSelectionDidChange:)])
    [nc addObserver: _delegate
	selector: @selector (tableSelectionDidChange:)
	name: TableSelectionDidChangeNotification
	object: self];
  if ([_delegate respondsToSelector: @selector (tableSelectionIsChanging:)])
    [nc addObserver: _delegate
	selector: @selector (tableSelectionIsChanging:)
	name: TableSelectionIsChangingNotification
	object: self];
}

- (id) delegate
{
  return _delegate;
}

- (void) _postTableDoubleClickNotification
{
  [nc postNotificationName: TableDoubleClickNotification object: self];
}

- (void) _postTableSelectionDidChangeNotification
{
  [nc postNotificationName: TableSelectionDidChangeNotification object: self];
}

- (void) _postTableSelectionIsChangingNotification
{
  [nc postNotificationName: TableSelectionIsChangingNotification object: self];
}

- (void) doubleClick: (id) sender
{
  [playlist selectSongNumber: [playlistView selectedRow]];
  if ([playlist shuffle])
    [playlist reShuffle];

  [self _postTableDoubleClickNotification];
}

- (void) deleteAllColumns
{
  NSArray *columns;
  int count, max;

  columns = [playlistView tableColumns];
  max = [playlistView numberOfColumns];
  for (count = 0; count < max; count++)
    [playlistView removeTableColumn: [columns objectAtIndex: 0]];
}

- (float) _timeColumnWidth
{
  NSAttributedString *anAString;
  float labelWidth, zeroesWidth;

  anAString = [[NSAttributedString alloc] initWithString: _(@"Time")];
  labelWidth = [anAString size].width;
  anAString = [[NSAttributedString alloc] initWithString: @"00:00"];
  zeroesWidth = [anAString size].width;

  return (18 + ((labelWidth > zeroesWidth)
                ? labelWidth
                : zeroesWidth));
}

- (void) addNeededColumns
{
  NSTableColumn *column;

  column = [[NSTableColumn alloc] initWithIdentifier: @"playlist"];
  [[column headerCell] setStringValue: _(@"Song")];
  [column setEditable: NO];
  [column setResizable: YES];
  [playlistView addTableColumn: column];

  column = [[NSTableColumn alloc] initWithIdentifier: @"time"];
  [[column headerCell] setStringValue: _(@"Time")];
  [[column dataCell] setAlignment: NSRightTextAlignment];
  [column setEditable: NO];
  [column setResizable: NO];
  [column setWidth: [self _timeColumnWidth]];
  [playlistView addTableColumn: column];
}

/* datasource protocol */
- (int) numberOfRowsInTableView: (NSTableView *) aTableView
{
  return [playlist count];
}

-            (id) tableView: (NSTableView *) aTableView
  objectValueForTableColumn: (NSTableColumn *) aTableColumn
                        row: (int) rowIndex
{
  NSString *cellContent, *colId;
  Song *song;

  colId = [aTableColumn identifier];
  song = [playlist songAtIndex: rowIndex];
  if (!song)
    return @"*** nil ***";

  if ([colId isEqualToString: @"playlist"])
    cellContent = [song displayRepresentation];
  else if ([colId isEqualToString: @"time"])
    cellContent = timeStringFromSeconds ([song duration]);
  else
    cellContent = @"in which kind of column are we?";

  return cellContent;
}

- (NSArray *) _songsInRows: (NSArray *) rows
{
  NSMutableArray *songs;
  Song *song;
  unsigned int count, max;

  max = [rows count];
  songs = [NSMutableArray arrayWithCapacity: max];
  for (count = 0; count < max; count++)
    {
      song = [playlist songAtNSIndex: [rows objectAtIndex: count]];
      [songs addObject: [song fullFilename]];
    }

  return songs;
}

- (BOOL) tableView: (NSTableView *) tv
	 writeRows: (NSArray *) rows
      toPasteboard: (NSPasteboard*) pboard
{
  NSArray *typeArray, *filesList;
  BOOL accept;
  unsigned int count;

  count = [rows count];

  if (count > 0)
    {
      accept = YES;
#ifdef __MACOSX__
      typeArray = [NSArray arrayWithObject: CynthiunePlaylistDragType];
      [pboard declareTypes: typeArray owner: self];
      [pboard setPropertyList: rows forType: CynthiunePlaylistDragType];
#else
      typeArray = [NSArray arrayWithObjects: CynthiunePlaylistDragType,
                           NSFilenamesPboardType, nil];
      [pboard declareTypes: typeArray owner: self];

      filesList = [self _songsInRows: rows];
      [pboard setPropertyList: rows forType: CynthiunePlaylistDragType];
      [pboard setPropertyList: filesList forType: NSFilenamesPboardType];
#endif
    }
  else
    accept = NO;

  return accept;
}

/* drag and drop */
- (BOOL) _acceptFilesInPasteboard: (NSPasteboard *) pboard
{
  NSArray *filesList;
  NSString *filename;
  FormatTester *formatTester;
  BOOL accept;
  unsigned int count, max;

  filesList = [pboard propertyListForType: NSFilenamesPboardType];

  accept = NO;
  count = 0;
  max = [filesList count];
  formatTester = [FormatTester formatTester];

  while (count < max && !accept)
    {
      filename = [filesList objectAtIndex: count];
      accept = ([formatTester fileType: filename] != -1);
      count++;
    }

  return accept;
}

- (NSDragOperation) tableView: (NSTableView *) tv
		 validateDrop: (id <NSDraggingInfo>) info
		  proposedRow: (int) row
	proposedDropOperation: (NSTableViewDropOperation) dropOperation
{
  NSArray *typeArray;
  NSString *availableType;
  NSPasteboard *pboard;
  NSDragOperation dragOperation;

  pboard = [info draggingPasteboard];
  typeArray = [NSArray arrayWithObjects: NSFilenamesPboardType, CynthiunePlaylistDragType, nil];
  availableType = [pboard availableTypeFromArray: typeArray];
  dragOperation = NSDragOperationNone;

  if (availableType)
    {
      [tv setDropRow: row dropOperation: NSTableViewDropAbove];
      if ([availableType isEqualToString: CynthiunePlaylistDragType])
        dragOperation = NSDragOperationMove;
      else /* NSFilenamesPboardType */
        if ([info draggingSource] != playlistView
            && [self _acceptFilesInPasteboard: pboard])
          dragOperation = NSDragOperationCopy;
    }

  return dragOperation;
}

- (void) _acceptDroppedFiles: (NSArray *) aFilesList
                       atRow: (int) row
{
  NSString *aFile;
  FormatTester *formatTester;
  unsigned int count, max;

  formatTester = [FormatTester formatTester];
  max = [aFilesList count];

  for (count = 0; count < max; count++)
    {
      aFile = [aFilesList objectAtIndex: count];
      
      if ([formatTester fileType: aFile] != -1)
        [playlist insertSong: [Song songWithFilename: aFile]
                  atIndex: row];
    }
}

- (void) _acceptDroppedRows: (NSArray *) aRowsList
                      atRow: (int) row
{
  unsigned int firstRow, max, count;

  firstRow = [playlist moveSongsWithRows: aRowsList to: row];
  [playlistView selectRow: firstRow byExtendingSelection: NO];
  max = [aRowsList count];
  for (count = 1; count < max; count++)
    [playlistView selectRow: (firstRow + count) byExtendingSelection: YES];
}

- (BOOL) tableView: (NSTableView *) tv
	acceptDrop: (id <NSDraggingInfo>) info
	       row: (int) row
     dropOperation: (NSTableViewDropOperation) op
{
  NSPasteboard *pboard;
  NSArray *objectsList, *typeArray;
  NSString *availableType;
  BOOL accept;

  pboard = [info draggingPasteboard];
  typeArray = ([info draggingSource] == playlistView)
    ? [NSArray arrayWithObject: CynthiunePlaylistDragType]
    : [NSArray arrayWithObject: NSFilenamesPboardType];
  availableType = [pboard availableTypeFromArray: typeArray];
  objectsList = [pboard propertyListForType: availableType];

  if ([objectsList count])
    {
      if ([availableType isEqualToString: CynthiunePlaylistDragType])
        [self _acceptDroppedRows: objectsList atRow: row];
      else
        [self _acceptDroppedFiles: objectsList atRow: row];
      [playlistView reloadData];
      accept = YES;
    }
  else
    accept = NO;

  return accept;
}

/* delegate */

- (void) tableViewSelectionIsChanging: (NSNotification *) aNotif
{
  [self _postTableSelectionIsChangingNotification];
}

- (void) tableViewSelectionDidChange: (NSNotification *) aNotif
{
  [self _postTableSelectionDidChangeNotification];
}

/* scrollView delegate */
- (void) scrollViewDidResize: (NSNotification *) aNotif
{
  [playlistView sizeToFit];
}

/* real methods */
- (unsigned int) getSelectedRow
{
  return [playlistView selectedRow];
}

- (Song *) selectedSong
{
  Song *aSong;
  int nbr;

  aSong = nil;
  nbr = [self getSelectedRow];
  if (nbr > -1)
    aSong = [playlist songAtIndex: nbr];

  return aSong;
}

- (void) selectRow: (int) aRow
{
  if (aRow > -1)
    [playlistView selectRow: aRow byExtendingSelection: NO];
  else
    [playlistView deselectAll: nil];
}

- (NSArray *) getSelection
{
  NSMutableArray *selection;
  int count, max;

  max = [playlistView numberOfRows];
  selection = [NSMutableArray arrayWithCapacity: max];

  for (count = 0; count < max; count++)
    if ([playlistView isRowSelected: count])
      [selection addObject: [playlist songAtIndex: count]];

  return selection;
}

- (NSArray *) getSelectionAsFilenames
{
  NSMutableArray *selection;
  Song *song;
  int count, max;

  max = [playlistView numberOfRows];
  selection = [NSMutableArray arrayWithCapacity: max];
  for (count = 0; count < max; count++)
    if ([playlistView isRowSelected: count])
      {
        song = [playlist songAtIndex: count];
        [selection addObject: [song fullFilename]];
      }

  return selection;
  
}

- (BOOL) isSongInSelection: (Song *) aSong
{
  BOOL result;
  int count, max;

  result = NO;
  count = 0;
  max = [playlistView numberOfRows];

  while (result == NO && count < max)
    if ([playlistView isRowSelected: count]
        && ([playlist songAtIndex: count] == aSong))
      result = YES;
    else
      count++;

  return result;
}

- (void) updateView
{
  [self selectRow: [playlist currentSongNumber]];
  [playlistView reloadData];
}

- (void) setPlaylist: (Playlist *) aPlaylist
{
  if (playlist)
    [playlist release];
  playlist = aPlaylist;
  if (playlist)
    [playlist retain];
  [self updateView];
}

@end
