/* Ogg.m - this file is part of Cynthiune
 *
 * Copyright (C) 2002, 2003 Wolfgang Sourdeau
 *
 * Author: Wolfgang Sourdeau <wolfgang@contre.com>
 *
 * This file is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This file is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#import <Foundation/Foundation.h>

#import <stdio.h>
#import <string.h>
#import <vorbis/vorbisfile.h>

#import <CynthiuneBundle.h>
#import <Format.h>

#import "Ogg.h"

#if defined (NeXT_PDO) || defined (__APPLE__)
#define _(X) NSLocalizedString (X, nil)
#define NSDebugLog
#endif

@implementation Ogg : NSObject

+ (NSArray *) bundleClasses
{
  return [NSArray arrayWithObject: [self class]];
}

- (BOOL) streamOpen: (NSString *) fileName;
{
  FILE *_fileHandle;

#if defined(GNUSTEP) || defined(NeXT_PDO)
  _fileHandle = fopen ([fileName cString], "rb");
#else
  _fileHandle = fopen ([fileName UTF8String], "r");
#endif
  bitStream = 0;

  if (_fileHandle)
    {
      _ov_file = calloc (sizeof (OggVorbis_File), 1); 
      lastError = ov_open (_fileHandle, _ov_file, NULL, 0L);
      if (lastError)
	fclose (_fileHandle);
    }
  else
    {
      printf ("No native handle...\n");
      lastError = OV_EREAD;
    }

  return (lastError == 0);
}

+ (BOOL) streamTestOpen: (NSString *) fileName
{
  FILE *_fileHandle;
  BOOL result = NO;
  int vorbisError;
  OggVorbis_File *_ov_test_file;

#if defined(GNUSTEP) || defined(NeXT_PDO)
  _fileHandle = fopen ([fileName cString], "rb");
#else
  _fileHandle = fopen ([fileName UTF8String], "r");
#endif

  if (_fileHandle)
    {
      _ov_test_file = calloc (sizeof (OggVorbis_File), 1); 
      vorbisError = ov_open (_fileHandle, _ov_test_file, NULL, 0L);
      if (vorbisError)
	{
          if (vorbisError != OV_ENOTVORBIS)
            NSLog (@"Ogg: streamTestOpen: %@", [self errorText: vorbisError]);
	  fclose (_fileHandle);
	}
      else
	{
	  result = YES;
	  ov_clear (_ov_test_file);
	}
    }
  else
    NSLog (@"Ogg: NULL _fileHandle");

  return result;
}

+ (NSString *) errorText: (int) error
{
  NSString *errMsg;
  char *cerrmsg;

  switch (error)
    {
    case OV_EREAD:
      cerrmsg = "A read from media returned an error";
      break;
    case OV_ENOTVORBIS:
      cerrmsg = "Bitstream is not Vorbis data";
      break;
    case OV_EVERSION:
      cerrmsg = "Vorbis version mismatch";
      break;
    case OV_EBADHEADER:
      cerrmsg = "Invalid Vorbis bitstream header";
      break;
    case OV_EFAULT:
      cerrmsg = "Internal logic fault";
      break;
    case OV_ENOSEEK:
      cerrmsg = "Bitstream is not seekable";
      break;
    case OV_EINVAL:
      cerrmsg = "Invalid argument value";
      break;
    case OV_EBADLINK:
      cerrmsg = "Invalid stream section supplied to libvorbisfile, or"
	" the requested link is corrupt";
      break;
    case 0:
      cerrmsg = "(No error)";
      break;
    default:
      cerrmsg = "(undefined error)";
    }

  errMsg = [[NSString alloc] initWithCString: cerrmsg];  

  return errMsg;
}

- (int) readNextChunk: (unsigned char*) buffer
	     withSize: (unsigned int) bufferSize
{
  int bytes_read;

  if (_ov_file)
    {
      bytes_read = ov_read (_ov_file, buffer, bufferSize,
			    0, 2, 1, &bitStream);
      if (bytes_read < 0)
	lastError = bytes_read;
    }
  else
    {
      lastError = OV_EFAULT;
      bytes_read = -1;
    }

  return bytes_read;
}

- (int) lastError
{
  return lastError;
}

- (unsigned int) readChannels
{
  return (_ov_file->vi->channels);
}

- (unsigned long) readRate
{
  return (_ov_file->vi->rate);
}

- (unsigned int) readDuration
{
  return (ov_time_total (_ov_file, -1));
}

- (NSString *) readComment: (char*) cString
{
  NSString *comment;
  const char *cComment;
  char **cPointer;
  BOOL found = NO;
  int counter = 0;
  
  cPointer = ov_comment (_ov_file, -1)->user_comments;
  cComment = *cPointer;

  while (cComment && !found)
    {
      int len = strlen (cString);
      counter++;

      if (!strncmp (cString, cComment, len))
	{
	  found = YES;
	  cComment += len + 1;
	}
      else
	cComment = *(cPointer + counter);
    }

  if (cComment)
#ifdef NeXT_PDO
    comment = [NSString stringWithCString: cComment];
#else
    comment = [NSString stringWithUTF8String: cComment];
#endif
  else
    comment = @"";

  return comment;
}

- (NSString *) readTitle
{
  return [self readComment: "title"];
}

- (NSString *) readGenre
{
  return [self readComment: "genre"];
}

- (NSString *) readArtist
{
  return [self readComment: "artist"];
}

- (NSString *) readAlbum
{
  return [self readComment: "album"];
}

- (NSString *) readTrackNumber
{
  return [self readComment: "tracknumber"];
}

- (void) streamClose
{
  ov_clear (_ov_file);
  _ov_file = NULL;
}

// Player Protocol
+ (NSArray *) acceptedFileExtensions
{
  return [NSArray arrayWithObjects: @"ogg", nil];
}

- (BOOL) isSeekable
{
  return (BOOL) ov_seekable (_ov_file);
}

- (void) seek: (unsigned int) aPos
{
  ogg_int64_t pcmPos;
  pcmPos = aPos * [self readChannels] * [self readRate] / 2;
#ifdef __MACOSX__
  ov_pcm_seek (_ov_file, pcmPos);
#else
  NSDebugLog (@"Ogg: seeking to %d/%d (test)", aPos, pcmPos);
  ov_pcm_seek_lap (_ov_file, pcmPos);
#endif
}

@end
