/***************************************************************************
                          fall.cpp  -  description
                             -------------------
    begin                : Sat Aug 18 2001
    copyright            : (C) 2001 by Immi
    email                : cuyo@karimmi.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "fall.h"
#include "spielfeld.h"
#include "aufnahme.h"
#include "fehler.h"


/* Anzahl der Pixel: Verschiebung einen Blops beim drehen... */
#define am_drehen_diff 16
/* Anz. d. angezeigten Zwischenbildchen beim Drehen. (Das erste
   Zwischenbild sieht man allerdings nicht...) */
#define am_drehen_start 3


/* Anzahl der Pixel: Verschiebung des Falls beim verschieben... */
#define am_schieben_diff 8
/* Restverschiebung direkt nach dem Tastendruck. (Das erste
   Zwischenbild sieht man allerdings nicht...) */
#define am_schieben_start 16



/* Return-Werte fr testBelegt() */
#define belegt_keins 0
#define belegt_0 1
#define belegt_1 2
#define belegt_beide (belegt_0 | belegt_1)




/** Konstruktor... */
Fall::Fall(Spielfeld * sp, bool re): mSp(sp) {
  mBlop[0].setBesitzer(this, re, 0);
  mBlop[1].setBesitzer(this, re, 1);
}

	
/** Erzeugt ein neues Fall. Liefert false, wenn dafr kein Platz ist */
bool Fall::erzeug() {
  mPos.x = grx / 2 - 1;
  mPos.yy = mSp->getHetzrandYPix() - gric + fall_langsampix;
  mPos.r = richtung_waag;
	
  /* Platz? */
  if (testBelegt(mPos)) {
    mPos.r = richtung_keins;
    return false;
  }

  mBlop[0] = Blop(Aufnahme::rnd(ld->mAnzFarben));
  mBlop[1] = Blop(Aufnahme::rnd(ld->mAnzFarben));

  mExtraDreh = 0;
  mExtraX = 0;
  mSchnell = false;
		
  setUpdate();
  return true;
}
	
	
/** Liefert true, wenn das Fallende senkrecht ist */
bool Fall::istSenkrecht() const {
  CASSERT(mPos.r == richtung_waag || mPos.r == richtung_senk);
  return mPos.r == richtung_senk;
}

	
/** Lsst nur noch Blop a brig */
void Fall::halbiere(int a) {
  CASSERT(mPos.r == richtung_waag);
  CASSERT(a == 0 || a == 1);
  mPos.x += a;
  mPos.r = richtung_einzel;
  mExtraDreh = 0;
  mExtraX = 0;
}
	
/** Entfernt das Fall ganz */
void Fall::zerstoere() {
  mPos.r = richtung_keins;
}


/** Bewegt das Fall ggf. nach unten und animiert es ggf.
 */
void Fall::spielSchritt() {
  /* Bei nicht-Existenz nichts tun. */
  if (mPos.r == richtung_keins)
    return;

  /* Fall an alter Pos. lschen. */
  setUpdate();
 		
  /* Wenn der Stein von einem frheren Tastendruck noch nicht fertig
     gedreht ist, dann weiterdrehen */
  if (mExtraDreh > 0)
    mExtraDreh--;
  /* Wenn der Stein noch nicht fertig X-verschoben ist, dann jetzt fertig
     X-verschieben */
  if (mExtraX > 0) mExtraX -= am_schieben_diff;
  if (mExtraX < 0) mExtraX += am_schieben_diff;


 	
  /* Wenn am Platzen, dann nicht mehr nach unten bewegen */
  if (getAmPlatzen())
    return;
 		
  /* Neue y-Koordinate... */
  FallPos fp2 = mPos;
  if (mSchnell || mPos.r == richtung_einzel)
    fp2.yy += gric;
  else
    fp2.yy += fall_langsampix;

  /* Gibt's Platz um weiterzufallen? */
  int beleg = testBelegt(fp2);
  if (!beleg) {
    mPos = fp2;
    setUpdate();
    return;
  }
 	
  /* OK, wir kommen irgendwo auf. */
 	
  /* Einzelblop? */
  if (mPos.r == richtung_einzel) {
    //Blop::beginGleichzeitig();
    festige(0);
    //Blop::endGleichzeitig();
    zerstoere();
    return;
  }

  /* Senkrecht? */
  if (istSenkrecht()) {
    /* Wichtig: Erst den unteren Blop festigen; es knnte sein, dass die Blops
       weiter oben gefestigt werden mssen, als geplant (wg. Hochverschiebung
       oder sogar wegen richtiger Spielfeldnderung am Ende von einer Zeilen-
       bergabe)... */
    //Blop::beginGleichzeitig();
    festige(1);
    festige(0);
    //Blop::endGleichzeitig();
    zerstoere();
    return;
  }

  /* OK, waagrecht. Welche Teile? */
  //Blop::beginGleichzeitig();
  if (beleg & belegt_0) festige(0);
  if (beleg & belegt_1) festige(1);
  //Blop::endGleichzeitig();


  /* Beide Hlften aufgekommen? */ 		
  if (beleg == belegt_beide) {
    zerstoere();
    return;
  }
 			
  /* Nur eine Hlfte aufgekommen. */
  mPos = fp2;
  if (beleg == belegt_0) {
    // linke Hlfte aufgekommen =>
    // briges rechtes nach links schieben
    mBlop[0] = mBlop[1];
  }
  /* Nur noch die Hlfte vom Fall ist brig; wenn das linke fest ist,
     dann die rechte Hlfte. */
  halbiere(beleg == belegt_0);
  return;
}


/** Fhrt die Animationen durch. Innerhalb einer Gleichzeit aufrufen. */
void Fall::animiere() {
  //CASSERT(gGleichZeit);
  for (int i = 0; i < getAnz(); i++)
    mBlop[i].animiere();
}


/** kopiert einen fallenden Blop nach mDaten. Und sendet
    ein land-event. Muss innerhalb einer Gleichzeit aufgerufen
    werden. */
void Fall::festige(int n) {
  int x = getX(n);
  int y = getY(n);

  /* Evtl. ist dieses Feld schon belegt. Dann so lange weiter oben probieren,
     bis wir ein freies Feld finden. */
  while (mSp->getDatenPtr()->getFeldArt(x, y) != blopart_keins) {
    y--;
    /* Sollte tatschlich (auf welche Art auch immer) pltzlich die ganze Spalte
       voll sein, dann verschwindet der Blop halt. */
    if (y < 0)
      return;
  }
  Blop & dst = mSp->getDatenPtr()->getFeld(x, y);
  dst = mBlop[n];
  dst.execEvent(event_land);
}



/** Bewegt das Fall eins nach links */
void Fall::tasteLinks() {
  if (steuerbar()) {
    FallPos fp2 = mPos;
    fp2.x--;
    if (!testBelegt(fp2)) {
      setUpdate();	
      mPos = fp2;
      /* Auf dem Bildschirm soll der Fall noch nicht fertigverschoben
	 erscheinen */
      mExtraX = am_schieben_start;
      setUpdate();	
    }
  }
}

/** Bewegt das Fall eins nach rechts */
void Fall::tasteRechts(){
  if (steuerbar()) {
    FallPos fp2 = mPos;
    fp2.x++;
    if (!testBelegt(fp2)) {
      setUpdate();	
      mPos = fp2;
      /* Auf dem Bildschirm soll der Fall noch nicht fertigverschoben
	 erscheinen */
      mExtraX = -am_schieben_start;
      setUpdate();	
    }
  }
}

/** Dreht das Fall */
void Fall::tasteDreh(){
  if (steuerbar()) {
    FallPos fp2 = mPos;
    fp2.r = fp2.r == richtung_waag ? richtung_senk : richtung_waag;
    if (!testBelegt(fp2)) {
      setUpdate();	
      mPos = fp2;
      /* Drehrichtung bei senkrecht gespiegeltem Level andersrum, damit
	 es fr den Spieler gleich erscheint */
      if (ld->mSpiegeln ? fp2.r == richtung_senk : fp2.r == richtung_waag) {
	Blop b = mBlop[0];
	mBlop[0] = mBlop[1];
	mBlop[1] = b;
      }
      /* Fr Level, bei denen sich die Teile beim Drehen verndern... */
      Blop::beginGleichzeitig();
      mBlop[0].execEvent(event_turn);
      mBlop[1].execEvent(event_turn);
      Blop::endGleichzeitig();

      /* Auf dem Bildschirm soll der Fall noch nicht fertiggedreht
	 erscheinen */
      mExtraDreh = am_drehen_start;
      setUpdate();	
    }
  }
}

/** ndert die Fallgeschwindigkeit vom Fall */
void Fall::tasteFall(){
  if (steuerbar())
    mSchnell = !mSchnell;
}


/** Setzt die Grafik vom Fall auf upzudaten */
void Fall::setUpdate() {
  mSp->setUpdateFall();
}

/** Liefert true, wenn das Fall (noch) am Platzen ist
    (wg. Spielende) */
bool Fall::getAmPlatzen() const {
  if (mPos.r == richtung_keins)
    return false;

  /* Wenn einer platzt, dann beide. */
  CASSERT(mPos.r == richtung_einzel ||
         mBlop[0].getAmPlatzen() == mBlop[1].getAmPlatzen());

  return mBlop[0].getAmPlatzen();
}

/** Lsst alle Blops vom Fall platzen (Spielende). */
void Fall::lassPlatzen() {
  CASSERT(mPos.r != richtung_keins);
  for (int i = 0; i < getAnz(); i++)
    mBlop[i].lassPlatzen();
  setUpdate();
}

/** Liefert einen Pointer auf die Blops zurck. Wird vom
    KIPlayer bentigt. */
const Blop * Fall::getBlopPtr() const {
  return mBlop;
}

/** Malt das Fall. */
void Fall::malen(QPainter & p) const {
  /* Wenn kein Fallendes unterwegs ist, liefert getAnz() 0 */
  for (int i = 0; i < getAnz(); i++) {
    int x = getXX(i) + mExtraX; // nicht fertig verschoben?
    int y = getYY(i);
  		
    /* Ist das fallende noch dabei, sich zu drehen? */
    if (mExtraDreh != 0) {

      /* 3 bedeutet eigentlich: Noch gar nicht gedreht. Normalerweise wird
	 nach einem Tastendruck erst mal ein spielSchritt() aufgerufen (mit
	 mExtraDreh--) und dann das Fall erst neu gemalt. Bei einem auer-
	 ordentlichen Update knnte allerdings auch schon frher neu gemalt
	 werden. Der Einfachheit halber malen wir da schon angedreht. */
      int ed = mExtraDreh;
      if (ed == 3) ed = 2;

      /* Tabelle mit den Verschiebungen der Blobs */
      /* Bit 8: Spiegel? | Bit 4: wirdSenk? | Bit 2: Blob1? | Bit 1: Schritt1? */
      int offset = 10 * ld->mSpiegeln + 5 * istSenkrecht() + 2 * i + (ed == 1);
      /*           Normal    Spiegel    */
      /*           Waag Senk Waag Senk  */
      int drehx = "2212 2243 2212 4322"[offset] - '1';
      int drehy = "4322 2212 2243 2212"[offset] - '1';
      /*           1*>01>0* 0*>01 10>0* */
      /*           0  *  1  1  *  *  1  */
      /* 1: -gric + amDrehenDiff   2: 0   3: +amDrehenDiff 4: +gric */
      int wandel[] = {-gric + am_drehen_diff, 0, am_drehen_diff, gric};
      x += wandel[drehx];
      y += wandel[drehy];
    }
  		
    mBlop[i].malen(p, x, y);
  }
}


/** Prueft, ob an Position p schon was im Weg ist oder nicht.
    Wenn irgendwo drber in der Spalte was ist, zhlt das auch
    als im Weg.
    @return eine Konstante belegt_... */
int Fall::testBelegt(FallPos p) const {
  int i = 0;
  int ret = belegt_keins;
	
  /* Sonderfall: Zwei senkrechte Blops => Blop 0 braucht nicht getestet
     zu werden, da er in der Blop-1-Spalte liegt. Auerdem: Wenn im Senkrecht-
     Modus Blop 1 belegt ist, dann per Definition auch Blop 0. (Das wird
     zwar (im Moment) nirgends verwendet, macht aber Sinn.) */
  if (mPos.r == richtung_senk)
    i = 1;
		
  for (; i < p.getAnz(); i++)
    if (!mSp->getDatenPtr()->testPlatzSpalte(p.getX(i), p.getY(i, mSp->getHochVerschiebung())))
      ret |= (i == 0 ? belegt_0 : belegt_1);

  /* Wieder der Sonderfall */
  if (mPos.r == richtung_senk && ret) ret = belegt_beide;
	
  return ret;
}
/** Liefert true, wenn das Fall existiert. */
bool Fall::existiert() const {
  return mPos.r != richtung_keins;
}



#define ASSERT_EX(a) CASSERT(mPos.r != richtung_keins && !(mPos.r == richtung_einzel && a == 1))


int Fall::getX(int a) const {
  ASSERT_EX(a);
  return mPos.getX(a);
}

int Fall::getY(int a) const {
  ASSERT_EX(a);
  return mPos.getY(a, mSp->getHochVerschiebung());
}

int Fall::getXX(int a) const {
  ASSERT_EX(a);
  return mPos.x * gric + a * (mPos.r == richtung_waag) * gric;
}
int Fall::getYY(int a) const {
  ASSERT_EX(a);
  return mPos.yy + a * (mPos.r == richtung_senk) * gric;
}

#undef ASSERT_EX

/** Liefert true, wenn das Fall aus grade am zerfallen ist
    (d. h. existiert, aber aus nur noch einem Blop besteht).
    In dieser Zeit darf nmlich keine Explosion gezndet
    werden. (Erst warten, bis der andere Blop auch angekommen
    ist.) */
bool Fall::istEinzel(){
  return mPos.r == richtung_einzel;
}



int Fall::getSpezConst(int vnr) const {
  /* Wie nett: Man hat uns mitgeteilt, welcher der beiden
     Blops anfragt. */
  bool bin_1 = vnr <= spezconst_bin_1;
  if (bin_1) vnr -= spezconst_bin_1;
  
  switch (vnr) {
  case spezconst_loc_x:
    return getX(bin_1);
  case spezconst_loc_y:
    return getY(bin_1);
  case spezconst_turn:
    CASSERT(mExtraDreh < 3);
    return "021"[mExtraDreh] - '0';
  case spezconst_falling:
    return 1;
  };

  /* Wir wissen von nix; Blop::getSpezConst() soll den Default-Wert
     zurckliefern. */
  return spezconst_defaultwert;
}

