
/*
 * Copyright (c) 1999,2000 David Stes.
 *
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Library General Public License as published 
 * by the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: txtframe.m,v 1.37 2002/06/15 19:19:31 stes Exp $
 */

#include <Object.h>
#include <assert.h>
#include <curses.h>
#include <stdlib.h>
#include "cursel.h"
#include "frame.h"
#include "txtframe.h"
#include <ocstring.h>
#include <ordcltn.h>
#include <octext.h>
#include <txtstyle.h>
#include <txtattr.h>
#include <paragrph.h>
#include "slk.h"
#include "textline.h"
#include "charscan.h"
#include "compscan.h"
#include "dispscan.h"

id prettytext(char *s)
{
  id b,text;
  int i,j,k,l,n;
  unsigned curmode = A_NORMAL;
  unsigned lastmode = A_NORMAL;

  n = strlen(s);
  b = [String new:n];
  text = [Text fromString:b];

  for(i=0,j=0,k=0,l=0;i<n;i++) {
    char d = s[i];

    if (d == '\\' && i+4 <= n) {
      char *q = s+i+2;
      d = s[i+1];
      if (d=='+'||d=='-') {
        unsigned mode = A_STANDOUT|A_BOLD;
        if (!strncmp(q,"ul",2)) mode=A_UNDERLINE;
        if (!strncmp(q,"so",2)) mode=A_STANDOUT;
        if (!strncmp(q,"rv",2)) mode=A_REVERSE;
        if (!strncmp(q,"bk",2)) mode=A_BLINK; 
        if (!strncmp(q,"dm",2)) mode=A_DIM;
        if (!strncmp(q,"bd",2)) mode=A_BOLD;
        if (!strncmp(q,"ac",2)) mode=A_ALTCHARSET;
        if (!strncmp(q,"nm",2)) mode=A_NORMAL;
        if (mode != (A_STANDOUT|A_BOLD)) {
          if (j>0) {
             [b at:k insert:s+l count:j];
             if (curmode != lastmode) {
              id a = [TextAttribute emphasiscode:curmode];
              [text addAttribute:a from:k to:k+j-1];
              lastmode=curmode;
             }
             k+=j;j=0;
          }
          if (d=='+') {
             curmode=(curmode | mode);
          } else {
             curmode=(curmode & (~mode));
          }
          i+=3;
          l=i+1;
          continue;
        }
      }
    }
    j++;
  }

  if (j>0) {
    [b at:k insert:s+l count:j];
    if (curmode != lastmode) {
      id a = [TextAttribute emphasiscode:curmode];
      [text addAttribute:a from:k to:k+j-1];
    }
  }

  return text;
}

@implementation Textframe

- (STR)name
{
  return (title)?[title str]:"Text";
}

- sethome
{
  ycurs = 0;
  xcurs = 0;
  ytop  = 0;
  return self;
}

- setalignment:x
{
  if (x==nil) {
   style = [TextStyle named:"left"];
  } else {
   char *s;
   s = [expandstr(x) str];
   if ((!strcmp(s,"left"))||(!strcmp(s,"right"))||(!strcmp(s,"center"))||(!strcmp(s,"justify"))) {
     style = [TextStyle named:s];
   } else {
     [self notImplemented:_cmd];
   }
  }
  return self;
}

- setedit:x
{
  edit = expandstr(x);return self;
}

- settext:x
{
  if (x) {
    text = prettytext([expandstr(x) str]);
  }
  return self;
}

- setheader:x
{
  if (x) {
    header = prettytext([expandstr(x) str]);
  }
  return self;
}

- settitle:t
{
  title = expandstr(t);return self;
}

- setwrap:w
{
  wrap = expandstr(w);
  return self;
}

- headerparagraph
{
  if (headerparagraph) {
    return headerparagraph;
  } else {
    if (!header) header = [Text new];
    if (!style) style = [TextStyle named:"left"];
    return headerparagraph = [Paragraph withText:header style:style];
  }
}

- paragraph
{
  if (paragraph) {
    return paragraph;
  } else {
    if (!text) text = [Text new];
    if (!style) style = [TextStyle named:"left"];
    return paragraph = [Paragraph withText:text style:style];
  }
}

- compose
{
   int k;
   BOOL dowrap;
   int startindex,stopindex;

   if (!cScan) cScan = [CompositionScanner new];
   
   [cScan setleftmargin:1];
   [cScan setrightmargin:[self width]-2];
   dowrap = (wrap == nil) || (!strcasecmp([wrap str],"true"));
   [cScan setwrapping:dowrap];

   startindex = 0;
   stopindex = [text size];

   lines = [OrdCltn new];
  
   for(k=0;startindex<stopindex;k++) {
     id line = [cScan composeLine:k from:startindex in:[self paragraph]];
     startindex = [line last] + 1;
     dbg("textwrap: %i %i\n",k,startindex); 
     [lines add:line];
   }

   if (header) {
     startindex = 0;
     stopindex = [header size];

     headerlines = [OrdCltn new];
  
     for(k=0;startindex<stopindex;k++) {
       id line=[cScan composeLine:k from:startindex in:[self headerparagraph]];
       startindex = [line last] + 1;
       dbg("textwrap: %i %i\n",k,startindex); 
       [headerlines add:line];
     }

     hheader = [headerlines size] + 1;
     if (hheader > [self height] - 2) hheader = [self height] - 2;
   } else {
     hheader = 0;
   }

   hbody = [self height] - 2 - hheader;
   if (hbody > [lines size]) hbody = [lines size];
   assert(hbody >= 0);

   [self sethome]; /* reset cursor */
   return self;
}

- setframerect
{
  id r;
  int w,h;
  if (columns) {
    w = atoi([columns str]);
  } else {
    w = 10;
  }
  if (rows) {
    h = atoi([rows str]);
  } else {
    h = 10;
  }
  w+=4;
  h+=2;
  if (w > DISPLAYW) w=DISPLAYW;
  if (h > DISPLAYH) h=DISPLAYH;

  r=[self begframerect:w:h]; 
  if (r) {
    [self setframerect:r]; 
    return self;
  } else {
    return nil;
  }
}

- create
{
  int ok;
  if (eti_win) [self destroy];
  if (!framerect) {
    if (![self setframerect]) return nil;
  }
  [self compose];
  eti_win=subwin(stdscr,[self height],[self width],[self top],[self left]);
  assert(eti_win);
  eti_subwin=derwin(eti_win,[self height]-2,[self width]-2,1,1);
  assert(eti_subwin);
  ok = keypad(eti_subwin,TRUE);
  assert(ERR != ok);
  return self;
}

- drawheaderlines:(int)n
{
  int i;
  assert(n <= [headerlines size]);

  if (!dScan) dScan = [DisplayScanner new];
  [dScan setleftmargin:0];
  [dScan setrightmargin:[self width]-2];

  wattrset(eti_subwin,A_NORMAL);
  [dScan setetiwin:eti_subwin];

  for(i=0;i<n;i++) {
    id line = [headerlines at:i];
    [dScan displayLine:line num:i in:[self headerparagraph] at:i];
  }

  return self;
}

- drawlines:(int)y:(int)n at:(int)h
{
  int i;
  dbg("drawlines (%i,%i) at %i\n",y,n,h);
  assert(0 <= y && y+n <= [lines size]);

  if (!dScan) dScan = [DisplayScanner new];
  [dScan setleftmargin:0];
  [dScan setrightmargin:[self width]-2];

  wattrset(eti_subwin,A_NORMAL);
  [dScan setetiwin:eti_subwin];

  for(i=0;i<n;i++) {
    [dScan displayLine:[lines at:y+i] num:y+i in:[self paragraph] at:h+i];
  }

  return self;
}

- setcursor
{
  id l;
  int x;
  assert(0<=xcurs && xcurs<=[self width]);
  assert(0<=ycurs && ycurs<=hbody);
  assert(0<=ytop && ytop<=[lines size]);
  l = [lines at:ytop+ycurs];
  x = [l last] - [l first];
  if (xcurs > x) {
    wmove(eti_subwin,hheader+ycurs,x);
  } else {
    wmove(eti_subwin,hheader+ycurs,xcurs);
  }
  return self;
}

- showcursor
{
  [self setcursor];
  wrefresh(eti_subwin);
  return self;
}

- draw
{
  dbg("draw %s\n",[self name]);

  if (!eti_win) { if (![self create]) return nil; }

  assert(eti_win);
  if (has_colors()) {
    wbkgdset(eti_win,COLOR_PAIR(colorpairs[CP_TEXT])|' ');
  }

  [self drawbox:[self name]];

  werase(eti_subwin);

  if (header) [self drawheaderlines:hheader - 1];

  [self drawlines:ytop:hbody at:hheader];

  [self drawscrollbar:ytop:ytop+hbody:[lines size]];
  [self setcursor];
  needdisplay=NO;
  return self;
}

- getevent
{
  int c = [self getkey];
  dbg("txtframe wgetch: %i\n",c);
  return [self keydown:c];
}

- setslk
{
  int n = MAXFUNKEYS;
  while (n--) slktab[n] = nil;
  slktab[1] = k_help;
  slktab[2] = k_prev_page;
  slktab[3] = k_next_page;
  slktab[4] = k_prev_frm;
  slktab[5] = k_next_frm;
  slktab[6] = k_cancel;
  slktab[7] = k_cmd_menu;
  return [super setslk];
}

- updatecontents
{
  dontdrawbox++;
  return [self needsdisplay];
}

- pageup
{
  ytop-=hbody;
  if (ytop < 0) { beep(); ytop = 0; }
  return [self updatecontents];
}

- pagedown
{
  ytop+=hbody;
  if (ytop+hbody > [lines size]) { beep(); ytop = [lines size] - hbody; }
  if (ytop < 0) ytop = 0;
  return [self updatecontents];
}

- arrowup
{
  ycurs--;
  if (ycurs < 0) { ycurs = 0; ytop--;[self updatecontents]; }
  if (ytop < 0) { ytop = 0; beep(); }
  return [self showcursor];
}

- arrowdown
{
  ycurs++;
  if (ycurs >= [lines size]) { ycurs = [lines size]-1;beep(); }
  if (ycurs >= hbody) { ycurs = hbody-1; ytop++;[self updatecontents]; }
  if (ytop+hbody > [lines size]) { ytop = [lines size] - hbody; beep(); }
  return [self showcursor];
}

- arrowleft
{
  int x;
  id l = [lines at:ycurs+ytop];
  x = [l last] - [l first];
  if (xcurs > x) xcurs = x;
  xcurs--;
  if (xcurs < 0) {
    if (ycurs) {
       l = [lines at:ycurs - 1 + ytop];
       xcurs = [l last] - [l first];
       return [self arrowup];
    } else {
       beep();
       xcurs = 0;
    }
  }
  return [self showcursor];
}

- arrowright
{
  id l = [lines at:ycurs + ytop];
  xcurs++;
  if (xcurs > ([l last] - [l first])) {
    if (ycurs+ytop+1 == [lines size]) {
      beep();xcurs--;
    } else {
      xcurs = 0;
      return [self arrowdown];
    }
  }
  return [self showcursor];
}

- home
{
  [self sethome];
  return [self showcursor];
}

@end
 
