#ifndef _CONFIG_H
#define _CONFIG_H

// Various routines that deal with parsing the configuration file.

#include <arpa/inet.h>
#include <netinet/in.h>
#include <stddef.h>
#include <string>
#include <vector>

struct StreamConfig {
	std::string url;  // As seen by the client.
	std::string src;  // Can be empty.
	size_t backlog_size;
	size_t prebuffering_bytes;
	uint32_t pacing_rate;  // In bytes per second. Default is ~0U (no limit).
	enum Encoding { STREAM_ENCODING_RAW = 0, STREAM_ENCODING_METACUBE };
	Encoding encoding;
	Encoding src_encoding;
};

struct UDPStreamConfig {
	sockaddr_in6 dst;
	std::string src;  // Can be empty.
	uint32_t pacing_rate;  // In bytes per second. Default is ~0U (no limit).
	int ttl;  // Default is -1 (use operating system default).
	int multicast_iface_index;  // Default is -1 (use operating system default).
};

struct Gen204Config {
	std::string url;  // As seen by the client.
	std::string allow_origin;  // Can be empty.
};

struct AcceptorConfig {
	sockaddr_in6 addr;
};

struct LogConfig {
	enum { LOG_TYPE_FILE, LOG_TYPE_CONSOLE, LOG_TYPE_SYSLOG } type;
	std::string filename;
};

struct Config {
	bool daemonize;
	int num_servers;
	std::vector<StreamConfig> streams;
	std::vector<UDPStreamConfig> udpstreams;
	std::vector<Gen204Config> pings;
	std::vector<AcceptorConfig> acceptors;
	std::vector<LogConfig> log_destinations;

	std::string stats_file;  // Empty means no stats file.
	int stats_interval;

	std::string input_stats_file;  // Empty means no input stats file.
	int input_stats_interval;

	std::string access_log_file;  // Empty means no accses_log file.
};

// Parse and validate configuration. Returns false on error.
// <config> is taken to be empty (uninitialized) on entry.
bool parse_config(const std::string &filename, Config *config);

#endif  // !defined(_CONFIG_H)
