#ifndef _STREAM_H
#define _STREAM_H 1

// Representation of a single, muxed (we only really care about bytes/blocks) stream.
// Fed by Input, sent out by Server (to Client).

#include <stddef.h>
#include <stdint.h>
#include <sys/types.h>
#include <sys/uio.h>
#include <string>
#include <vector>

class StreamProto;
struct Client;

enum StreamStartSuitability {
	NOT_SUITABLE_FOR_STREAM_START,
	SUITABLE_FOR_STREAM_START,
};

struct Stream {
	// Must be in sync with StreamConfig::Encoding.
	enum Encoding { STREAM_ENCODING_RAW = 0, STREAM_ENCODING_METACUBE };

	Stream(const std::string &stream_id, size_t backlog_size, Encoding encoding);
	~Stream();

	// Serialization/deserialization.
	Stream(const StreamProto &serialized, int data_fd);
	StreamProto serialize();

	// Changes the backlog size, restructuring the data as needed.
	void set_backlog_size(size_t new_size);

	// Mutex protecting <queued_data> and <queued_data_last_starting_point>.
	// Note that if you want to hold both this and the owning server's
	// <mutex> you will need to take <mutex> before this one.
	mutable pthread_mutex_t queued_data_mutex;

	std::string url;

	// The HTTP response header, without the trailing double newline.
	std::string http_header;

	// The video stream header (if any).
	std::string stream_header;

	// What encoding we apply to the outgoing data (usually raw, but can also
	// be Metacube, for reflecting to another Cubemap instance).
	Encoding encoding;

	// The stream data itself, stored in a circular buffer.
	//
	// We store our data in a file, so that we can send the data to the
	// kernel only once (with write()). We then use sendfile() for each
	// client, which effectively zero-copies it out of the kernel's buffer
	// cache. This is significantly more efficient than doing write() from
	// a userspace memory buffer, since the latter makes the kernel copy
	// the same data from userspace many times.
	int data_fd;

	// How many bytes <data_fd> can hold (the buffer size).
	size_t backlog_size;

	// How many bytes this stream have received. Can very well be larger
	// than <backlog_size>, since the buffer wraps.
	size_t bytes_received;

	// The last point in the stream that is suitable to start new clients at
	// (after having sent the header). -1 if no such point exists yet.
	ssize_t last_suitable_starting_point;
	
	// Clients that are in SENDING_DATA, but that we don't listen on,
	// because we currently don't have any data for them.
	// See put_client_to_sleep() and wake_up_all_clients().
	std::vector<Client *> sleeping_clients;

	// Clients that we recently got data for (when they were in
	// <sleeping_clients>).
	std::vector<Client *> to_process;

	// Maximum pacing rate for the stream.
	uint32_t pacing_rate;

	// Queued data, if any. Protected by <queued_data_mutex>.
	// The data pointers in the iovec are owned by us.
	std::vector<iovec> queued_data;

	// Index of the last element in queued_data that is suitable to start streaming at.
	// -1 if none. Protected by <queued_data_mutex>.
	int queued_data_last_starting_point;

	// Put client to sleep, since there is no more data for it; we will on
	// longer listen on POLLOUT until we get more data. Also, it will be put
	// in the list of clients to wake up when we do.
	void put_client_to_sleep(Client *client);

	// Add more data to <queued_data>, adding Metacube headers if needed.
	// Does not take ownership of <data>.
	void add_data_deferred(const char *data, size_t bytes, StreamStartSuitability suitable_for_stream_start);

	// Add queued data to the stream, if any.
	// You should hold the owning Server's <mutex>.
	void process_queued_data();

private:
	Stream(const Stream& other);

	// Adds data directly to the stream file descriptor, without adding headers or
	// going through <queued_data>.
	// You should hold the owning Server's <mutex>.
	void add_data_raw(const std::vector<iovec> &data);
};

#endif  // !defined(_STREAM_H)
