/* 
   auto-generate self signed TLS certificates. Imported from Samba.

   Copyright (C) Andrew Tridgell 2005
   Copyright (C) Jelmer Vernooij 2006
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "ctrlproxy.h"

#include <glib.h>

#include "ssl.h"

#include <gnutls/gnutls.h>
#include <gnutls/x509.h>

#define ORGANISATION_NAME "CtrlProxy"
#define UNIT_NAME         "CtrlProxy - temporary autogenerated certificate"
#define COMMON_NAME       "CtrlProxy"
#define LIFETIME          700*24*60*60
#define DH_BITS 		  1024

/* 
   auto-generate a set of self signed certificates
*/
void ssl_cert_generate(const char *keyfile, const char *certfile,
		       const char *cafile)
{
	gnutls_x509_crt cacrt, crt;
	gnutls_x509_privkey key, cakey;
	guint32 serial = (guint32)time(NULL);
	unsigned char keyid[100];
	char buf[4096];
	size_t bufsize;
	size_t keyidsize = sizeof(keyid);
	time_t activation = time(NULL), expiry = activation + LIFETIME;
	int ret;

	if (g_file_test(keyfile, G_FILE_TEST_EXISTS) || 
		g_file_test(certfile, G_FILE_TEST_EXISTS) || 
		g_file_test(cafile, G_FILE_TEST_EXISTS)) {
		log_global(LOG_WARNING, "TLS autogeneration skipped - some TLS files already exist");
		return;
	}

#define TLSCHECK(call) do { \
	ret = call; \
	if (ret < 0) { \
		log_global(LOG_WARNING, "TLS %s - %s", #call, gnutls_strerror(ret)); \
		goto failed; \
	} \
} while (0)

	TLSCHECK(gnutls_global_init());

	log_global(LOG_INFO, 
			   "Attempting to autogenerate TLS self-signed keys");
	
	log_global(LOG_TRACE, "Generating private key");
	TLSCHECK(gnutls_x509_privkey_init(&key));
	TLSCHECK(gnutls_x509_privkey_generate(key,   GNUTLS_PK_RSA, DH_BITS, 0));

	log_global(LOG_TRACE, "Generating CA private key");
	TLSCHECK(gnutls_x509_privkey_init(&cakey));
	TLSCHECK(gnutls_x509_privkey_generate(cakey, GNUTLS_PK_RSA, DH_BITS, 0));

	log_global(LOG_TRACE, "Generating CA certificate");
	TLSCHECK(gnutls_x509_crt_init(&cacrt));
	TLSCHECK(gnutls_x509_crt_set_dn_by_oid(cacrt, 
				      GNUTLS_OID_X520_ORGANIZATION_NAME, 0,
				      ORGANISATION_NAME, strlen(ORGANISATION_NAME)));
	TLSCHECK(gnutls_x509_crt_set_dn_by_oid(cacrt, 
				      GNUTLS_OID_X520_ORGANIZATIONAL_UNIT_NAME, 0,
				      UNIT_NAME, strlen(UNIT_NAME)));
	TLSCHECK(gnutls_x509_crt_set_dn_by_oid(cacrt,
				      GNUTLS_OID_X520_COMMON_NAME, 0,
				      COMMON_NAME, strlen(COMMON_NAME)));
	TLSCHECK(gnutls_x509_crt_set_key(cacrt, cakey));
	TLSCHECK(gnutls_x509_crt_set_serial(cacrt, &serial, sizeof(serial)));
	TLSCHECK(gnutls_x509_crt_set_activation_time(cacrt, activation));
	TLSCHECK(gnutls_x509_crt_set_expiration_time(cacrt, expiry));
	TLSCHECK(gnutls_x509_crt_set_ca_status(cacrt, 0));
	TLSCHECK(gnutls_x509_crt_set_version(cacrt, 3));
	TLSCHECK(gnutls_x509_crt_get_key_id(cacrt, 0, keyid, &keyidsize));
	TLSCHECK(gnutls_x509_crt_set_subject_key_id(cacrt, keyid, keyidsize));
	TLSCHECK(gnutls_x509_crt_sign(cacrt, cacrt, cakey));

	log_global(LOG_TRACE, "Generating TLS certificaten");
	TLSCHECK(gnutls_x509_crt_init(&crt));
	TLSCHECK(gnutls_x509_crt_set_dn_by_oid(crt, 
				      GNUTLS_OID_X520_ORGANIZATION_NAME, 0,
				      ORGANISATION_NAME, strlen(ORGANISATION_NAME)));
	TLSCHECK(gnutls_x509_crt_set_dn_by_oid(crt, 
				      GNUTLS_OID_X520_ORGANIZATIONAL_UNIT_NAME, 0,
				      UNIT_NAME, strlen(UNIT_NAME)));
	TLSCHECK(gnutls_x509_crt_set_dn_by_oid(crt,
				      GNUTLS_OID_X520_COMMON_NAME, 0,
				      COMMON_NAME, strlen(COMMON_NAME)));
	TLSCHECK(gnutls_x509_crt_set_key(crt, key));
	TLSCHECK(gnutls_x509_crt_set_serial(crt, &serial, sizeof(serial)));
	TLSCHECK(gnutls_x509_crt_set_activation_time(crt, activation));
	TLSCHECK(gnutls_x509_crt_set_expiration_time(crt, expiry));
	TLSCHECK(gnutls_x509_crt_set_ca_status(crt, 0));
	TLSCHECK(gnutls_x509_crt_set_version(crt, 3));
	TLSCHECK(gnutls_x509_crt_get_key_id(crt, 0, keyid, &keyidsize));
	TLSCHECK(gnutls_x509_crt_set_subject_key_id(crt, keyid, keyidsize));
	TLSCHECK(gnutls_x509_crt_sign(crt, crt, key));

	log_global(LOG_TRACE, "Exporting TLS keys");

	bufsize = sizeof(buf);
	TLSCHECK(gnutls_x509_crt_export(crt, GNUTLS_X509_FMT_PEM, buf, &bufsize));
	g_file_set_contents(certfile, buf, bufsize, NULL);

	bufsize = sizeof(buf);
	TLSCHECK(gnutls_x509_crt_export(cacrt, GNUTLS_X509_FMT_PEM, buf, &bufsize));
	g_file_set_contents(cafile, buf, bufsize, NULL);

	bufsize = sizeof(buf);
	TLSCHECK(gnutls_x509_privkey_export(key, GNUTLS_X509_FMT_PEM, buf, &bufsize));
	g_file_set_contents(keyfile, buf, bufsize, NULL);

	gnutls_x509_privkey_deinit(key);
	gnutls_x509_privkey_deinit(cakey);
	gnutls_x509_crt_deinit(cacrt);
	gnutls_x509_crt_deinit(crt);

	log_global(LOG_INFO, "TLS self-signed keys generated OK");
	return;

failed:
	log_global(LOG_WARNING, "TLS certificate generation failed");
}

gpointer ssl_create_server_credentials(struct global *global, 
									   GKeyFile *kf, const char *group)
{
	if (!g_key_file_has_key(kf, group, "keyfile", NULL) &&
		!g_key_file_has_key(kf, group, "certfile", NULL)) {
		char *keyfile = g_build_filename(global->config->config_dir, "key.pem", NULL);
		char *certfile = g_build_filename(global->config->config_dir, "cert.pem", NULL);
		g_key_file_set_string(kf, group, "keyfile", keyfile);
		g_key_file_set_string(kf, group, "certfile", certfile);
		if (!g_file_test(keyfile, G_FILE_TEST_EXISTS) && 
			!g_file_test(certfile, G_FILE_TEST_EXISTS)) {
			char *cafile = g_build_filename(global->config->config_dir, "ca.pem", NULL);
			ssl_cert_generate(keyfile, certfile, cafile);
			g_free(cafile);
		}
		g_free(keyfile);
		g_free(certfile);
	}

	return ssl_get_server_credentials(
					g_key_file_get_string(kf, group, "certfile", NULL),
					g_key_file_get_string(kf, group, "keyfile", NULL));
}
