/* $Id: cpl_polynomial-test.c,v 1.112 2012/02/07 10:10:37 llundin Exp $
 *
 * This file is part of the ESO Common Pipeline Library
 * Copyright (C) 2001-2008 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/*
 * $Author: llundin $
 * $Date: 2012/02/07 10:10:37 $
 * $Revision: 1.112 $
 * $Name: cpl-6_1_1 $
 */

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <float.h>
#include <math.h>

#include "cpl_polynomial.h"
#include "cpl_error_impl.h"
#include "cpl_test.h"
#include "cpl_tools.h"
#include "cpl_memory.h"
#include "cpl_math_const.h"

/*-----------------------------------------------------------------------------
                                   Defines
 -----------------------------------------------------------------------------*/

#define     POLY_COEF_NB    5
#define     VECTOR_SIZE 1024
#define     POLY_SIZE 24
#define     POLY_DIM 11

/*-----------------------------------------------------------------------------
                                  Pricate functions
 -----------------------------------------------------------------------------*/

static void cpl_polynomial_fit_bench_2d(cpl_size, cpl_size, cpl_size);

static void cpl_polynomial_fit_test_1d(FILE *);
static cpl_polynomial * cpl_polynomial_fit_test_2d(FILE *, cpl_boolean);

static void cpl_vector_fill_polynomial_fit_residual_test(void);

static double cpl_vector_get_mse(const cpl_vector     *,
                                 const cpl_polynomial *,
                                 const cpl_matrix     *,
                                 double *);

static cpl_error_code cpl_polynomial_fit_cmp(cpl_polynomial    *,
                                             const cpl_matrix  *,
                                             const cpl_boolean *,
                                             const cpl_vector  *,
                                             const cpl_vector  *,
                                             cpl_boolean        ,
                                             const cpl_size    *,
                                             const cpl_size    *);

static cpl_flops  mse_flops = 0;
static double     mse_secs  = 0.0;

/**@{*/

/*-----------------------------------------------------------------------------
                                  Main
 -----------------------------------------------------------------------------*/
int main(void)
{
    cpl_polynomial  *   poly1;
    cpl_polynomial  *   poly2;
    cpl_polynomial  *   poly3;
    double              x, y, z1, z2;
    double              z, y_1, y2, x1, x2;
    cpl_vector      *   vec;
    cpl_vector      *   taylor;
    double          *   vec_data;
    /* Some binomial coefficients */
    double              p15[8] = {1,15,105,455,1365,3003,5005,6435};
    double              xmax = 0; /* Maximum rounding error on x */
    double              ymax = 0; /* Maximum rounding error on y */
    double              eps;
    cpl_size            expo[POLY_DIM];
    cpl_size            i, j;
    cpl_size            k;
    cpl_error_code      error;
    FILE              * stream;


    cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_WARNING);


    stream = cpl_msg_get_level() > CPL_MSG_INFO
        ? fopen("/dev/null", "a") : stdout;

    cpl_test_nonnull( stream );

    cpl_vector_fill_polynomial_fit_residual_test();

    cpl_polynomial_fit_test_1d(stream);

    x = 3.14;
    y = 2.72;
    
    cpl_test_null( cpl_polynomial_extract(NULL, 0, NULL) );

    cpl_test_error(CPL_ERROR_NULL_INPUT);

    cpl_test( cpl_polynomial_derivative(NULL, 0) );

    cpl_test_error(CPL_ERROR_NULL_INPUT);

    /* Create a polynomial */
        
    poly1 = cpl_polynomial_new(1);
    cpl_polynomial_dump(poly1, stream);

    cpl_test_zero( cpl_polynomial_get_degree(poly1));

    error = cpl_polynomial_multiply_scalar(poly1, poly1, 0.0);
    cpl_test_eq_error(error, CPL_ERROR_NONE);
    cpl_test_zero( cpl_polynomial_get_degree(poly1));

    /* Various tests regarding 0-degree polynomials */

    cpl_test_null( cpl_polynomial_extract(poly1, 0, poly1) );

    cpl_test_error(CPL_ERROR_INVALID_TYPE);

    error = cpl_polynomial_derivative(poly1, 1);

    cpl_test_eq_error(error, CPL_ERROR_ACCESS_OUT_OF_RANGE);

    cpl_test_zero( cpl_polynomial_derivative(poly1, 0) );

    cpl_test_abs( cpl_polynomial_eval_1d(poly1, 5, &x), 0.0, 0.0);

    cpl_test_abs(x, 0.0, 0.0 );

    cpl_test_abs( cpl_polynomial_solve_1d(poly1, 5, &x, 1), 0.0, 0.0 );

    cpl_test_abs(x, 0.0, 0.0 );

    cpl_test_abs( cpl_polynomial_eval_1d(poly1, 5, &x), 0.0, 0.0);

    cpl_test_abs(x, 0.0, 0.0 );

    /* Properly react to a 0-degree polynomial with no roots */
    i = 0;
    cpl_test_zero( cpl_polynomial_set_coeff(poly1, &i, 1) );
    cpl_test_abs( cpl_polynomial_get_coeff(poly1, &i), 1.0, 0.0 );

    error = cpl_polynomial_solve_1d(poly1, 1, &x, 1);

    cpl_test_eq_error(error, CPL_ERROR_DIVISION_BY_ZERO);

    /* Equivalent to setting all coefficients to zero */

    error = cpl_polynomial_derivative(poly1, 0);
    cpl_test_eq_error(error, CPL_ERROR_NONE);

    cpl_test_abs( cpl_polynomial_eval_1d(poly1, 5, &x), 0.0, 0.0);

    cpl_test_abs( x, 0.0, 0.0);

    i = VECTOR_SIZE;
    cpl_test_zero( cpl_polynomial_set_coeff(poly1, &i, 1) );
    cpl_test_abs( cpl_polynomial_get_coeff(poly1, &i), 1.0, 0.0 );

    cpl_test_zero( cpl_polynomial_set_coeff(poly1, &i, 0) );

    cpl_test_eq( cpl_polynomial_get_dimension(poly1), 1);
    cpl_test_zero( cpl_polynomial_get_degree(poly1));

    cpl_test( poly2 = cpl_polynomial_duplicate(poly1) );

    cpl_test_zero( cpl_polynomial_get_degree(poly2));

    cpl_polynomial_delete(poly2);

    cpl_test( poly2 = cpl_polynomial_new(VECTOR_SIZE) );

    cpl_test_eq( cpl_polynomial_get_dimension(poly2), VECTOR_SIZE );
    cpl_test_zero( cpl_polynomial_get_degree(poly2));

    cpl_test_zero( cpl_polynomial_copy(poly2, poly1));

    cpl_test_eq( cpl_polynomial_get_dimension(poly2), 1 );
    cpl_test_zero( cpl_polynomial_get_degree(poly2));

    cpl_polynomial_delete(poly2);

    /* Set the coefficients : 1 + 2.x + 3.x^2 + 4.x^3 + 5.x^4 */
    for (i=0; i < POLY_COEF_NB ; i++) {
        if (i % (POLY_COEF_NB-2) != 0) continue;
        cpl_test_zero( cpl_polynomial_set_coeff(poly1, &i, (double)(i+1)) );
            cpl_polynomial_dump(poly1, stream);
    }
    for (i=0; i < POLY_COEF_NB ; i++) {
        if (i % (POLY_COEF_NB-2) == 0) continue;
            cpl_polynomial_dump(poly1, stream);
            cpl_test_zero( cpl_polynomial_set_coeff(poly1, &i, (double)(i+1)));
    }
    cpl_polynomial_dump(poly1, stream);
    /* Test cpl_polynomial_get_degree() */
    cpl_test_eq( cpl_polynomial_get_degree(poly1), POLY_COEF_NB-1);

    /* Test cpl_polynomial_get_coeff() */
    for (i=0; i < POLY_COEF_NB ; i++)
        cpl_test_abs( cpl_polynomial_get_coeff(poly1, &i), (double)(i+1), 0.0);

    /* Test cpl_polynomial_eval() */
    vec = cpl_vector_new(1);
    vec_data = cpl_vector_get_data(vec);
    vec_data[0] = x;
    z1 = cpl_polynomial_eval(poly1, vec);
    cpl_vector_delete(vec);

    /* Dump polynomials */
    cpl_polynomial_dump(poly1, stream);

    /* Test cpl_polynomial_duplicate() and cpl_polynomial_copy() */
    poly2 = cpl_polynomial_duplicate(poly1);

    cpl_test_zero( cpl_polynomial_compare(poly1, poly2, 0) );

    cpl_polynomial_copy(poly1, poly2);

    cpl_test_zero( cpl_polynomial_compare(poly1, poly2, 0) );

    i = 10 + cpl_polynomial_get_degree(poly1);
    error = cpl_polynomial_set_coeff(poly1, &i, 42.0);
    cpl_test_eq_error(error, CPL_ERROR_NONE);

    error = cpl_polynomial_multiply_scalar(poly1, poly2, 0.5);
    cpl_test_eq_error(error, CPL_ERROR_NONE);
    cpl_test_eq(cpl_polynomial_get_degree(poly1),
                cpl_polynomial_get_degree(poly2));

    error = cpl_polynomial_multiply_scalar(poly1, poly1, 2.0);
    cpl_test_eq_error(error, CPL_ERROR_NONE);

    cpl_test_zero( cpl_polynomial_compare(poly1, poly2, 0) );

    error = cpl_polynomial_multiply_scalar(poly1, poly2, 0.0);
    cpl_test_eq_error(error, CPL_ERROR_NONE);
    cpl_test_zero( cpl_polynomial_get_degree(poly1));

    error = cpl_polynomial_multiply_scalar(poly2, poly2, 0.0);
    cpl_test_eq_error(error, CPL_ERROR_NONE);
    cpl_test_zero( cpl_polynomial_get_degree(poly2));

    /* Free */
    cpl_polynomial_delete(poly1);
    cpl_polynomial_delete(poly2);

    /* Properly react to a polynomial with one real root at x = 1
       and a proper (positive) 1st guess */
    poly1 = cpl_polynomial_new(1);

    i = 0;
    cpl_test_zero( cpl_polynomial_set_coeff(poly1, &i, -1.0) );
    i++;
    cpl_test_zero( cpl_polynomial_set_coeff(poly1, &i, 1.0) );
    i = 2 * POLY_SIZE;
    cpl_test_zero( cpl_polynomial_set_coeff(poly1, &i, -1.0) );
    i++;
    cpl_test_zero( cpl_polynomial_set_coeff(poly1, &i, 1.0) );

    cpl_test_zero( cpl_polynomial_solve_1d(poly1, 5, &x, 1) );

    cpl_test_abs( x,  1.0, 0.0 );

    cpl_polynomial_delete(poly1);

    poly1 = cpl_polynomial_new(1);

    /* Create a tmporary vector - with a small length */
    cpl_test_nonnull( vec = cpl_vector_new(POLY_SIZE) );
    
    /* Fill the vector with a exp() Taylor series */
    vec_data = cpl_vector_get_data(vec);
    cpl_test_nonnull( vec_data );
    i = 0;
    vec_data[i] = 1;
    for (i=1 ; i < POLY_SIZE ; i++) {
        vec_data[i] = vec_data[i-1] / (double)i;
    }

    for (i=POLY_SIZE-1 ; i >= 0; i--) {
        cpl_test_zero( cpl_polynomial_set_coeff(poly1, &i, vec_data[i]) );
    }

    cpl_test_eq( cpl_polynomial_get_degree(poly1), POLY_SIZE-1);

    cpl_test_abs(cpl_polynomial_eval_1d(poly1, 0, &y), 1.0, DBL_EPSILON);
    cpl_test_abs( y, 1.0,  DBL_EPSILON);

    /* See how far away from zero the approximation holds */
    x = DBL_EPSILON;
    while ( fabs(cpl_polynomial_eval_1d(poly1, x, NULL)-exp( x))
          < DBL_EPSILON * exp( x) &&
            fabs(cpl_polynomial_eval_1d(poly1,-x, &y)-exp(-x))
          < DBL_EPSILON * exp(-x) ) {
        /* Differentation of exp() does not change anything
           - but in the case of a Taylor approximation one term is
           lost and with it a bit of precision */
        cpl_test_rel( exp(-x), y, 7.5 * DBL_EPSILON);
        x *= 2;
    }
    x /= 2;
    /* FIXME: Verify the correctness of this test */
    cpl_test_leq( -x, -FLT_EPSILON); /* OK for POLY_SIZE >= 4 */

    z2 = 2 * x / VECTOR_SIZE;

    /* Evaluate a sequence of exp()-approximations */
    taylor = cpl_vector_new(VECTOR_SIZE);
    cpl_test_zero( cpl_vector_fill_polynomial(taylor, poly1, -x, z2));

    vec_data = cpl_vector_get_data(taylor);

    cpl_test_nonnull( vec_data );
    for (i=0 ; i < VECTOR_SIZE ; i++) {
        const double xapp = -x + (double)i * z2; 
        const double yapp = exp( xapp );

        /* cpl_vector_fill_polynomial() is just a wrapper */
        cpl_test_abs( vec_data[i], cpl_polynomial_eval_1d(poly1, xapp, &y),
                      2*DBL_EPSILON);

        if ( fabs(y - yapp ) > ymax * yapp )
            ymax = fabs(y - yapp ) / yapp;

        if ( fabs(vec_data[i] - yapp ) > xmax * yapp )
            xmax = fabs(vec_data[i] - yapp ) / yapp;
    }

    cpl_msg_info("","Rounding on %d-term Taylor-exp() in range [%g; %g] "
        "[DBL_EPSILON]: %7.5f %7.5f", POLY_SIZE, -x, x, xmax/DBL_EPSILON,
        ymax/DBL_EPSILON);

    cpl_test_leq( xmax, 2.0*DBL_EPSILON);
    cpl_test_leq( xmax, ymax);
    cpl_test_leq( ymax, 7.39*DBL_EPSILON);

    vec_data = cpl_vector_get_data(vec);
    eps = vec_data[0];

    /* Solve p(y) = exp(x/2) - i.e. compute a logarithm */
    i = 0;
    vec_data[i] -= exp(x/2);
    cpl_polynomial_set_coeff(poly1, &i, vec_data[i]);
    cpl_test_zero( cpl_polynomial_solve_1d(poly1, -x * x, &y, 1) );
    vec_data[0] = eps;
    cpl_polynomial_set_coeff(poly1, &i, vec_data[i]);

    /* Check solution - allow up to 2 * meps rounding */
    cpl_test_rel( y, x/2,  2.0*DBL_EPSILON);

    /* Check Residual - allow up to 2 * meps rounding */
    cpl_test_rel( cpl_polynomial_eval_1d(poly1, y, NULL), exp(x/2), 2.0 * DBL_EPSILON );

    /* Free */
    cpl_vector_delete(vec);
    cpl_vector_delete(taylor);

    cpl_polynomial_delete(poly1);

    poly1 = cpl_polynomial_new(1);

    i = 0;
    cpl_polynomial_set_coeff(poly1, &i, 2);
    i++;
    cpl_polynomial_set_coeff(poly1, &i, 2);
    i++;
    cpl_polynomial_set_coeff(poly1, &i, 1);

    cpl_test_eq( cpl_polynomial_get_degree(poly1), 2);

    /* Properly react on a polynomial with no real roots */
    cpl_test_eq( cpl_polynomial_solve_1d(poly1, 0, &x, 1),
                 CPL_ERROR_DIVISION_BY_ZERO );

    i = 0;
    cpl_polynomial_set_coeff(poly1, &i, 4);

    /* Properly react on a polynomial with no real roots */
    error = cpl_polynomial_solve_1d(poly1, 0, &x, 1);

    cpl_test_eq_error(error, CPL_ERROR_CONTINUE);

    cpl_polynomial_delete(poly1);

    poly1 = cpl_polynomial_new(1);

    /* The simplest 15-degree polynomial */
    i =15;
    cpl_polynomial_set_coeff(poly1, &i, 1);

    cpl_test_eq( cpl_polynomial_get_degree(poly1), i);

    cpl_test_zero( cpl_polynomial_solve_1d(poly1, 10, &x, i) );
    cpl_test_abs( x, 0.0, DBL_EPSILON);
    cpl_test_abs( cpl_polynomial_eval_1d(poly1, x, &y), 0.0, DBL_EPSILON);
    cpl_test_abs( y, 0.0, DBL_EPSILON);

    /* -1 is a root with multiplicity 15 */
    x1 = -1;

    poly2 = cpl_polynomial_duplicate(poly1);
    cpl_test_zero( cpl_polynomial_shift_1d(poly2, 0, -x1) );
    /* poly2 now holds the binomial coefficients for n = 15 */

    i = 0;
    for (i = 0; i < 8; i++) {
        cpl_polynomial_set_coeff(poly1, &i, p15[i]);
        j = 15 - i;
        cpl_polynomial_set_coeff(poly1, &j, p15[i]);
    }

    i = 15;

    cpl_test_eq( cpl_polynomial_get_degree(poly1), i);

    for (i = 0; i < 8; i++) {
        j = 15 - i;
        z1 = cpl_polynomial_get_coeff(poly1, &i);
        z2 = cpl_polynomial_get_coeff(poly2, &i);
        z  = cpl_polynomial_get_coeff(poly2, &j);
        cpl_test_rel( z1, z2, 1.0 * DBL_EPSILON);
        cpl_test_rel( z,  z2, 1.0 * DBL_EPSILON);
    }

    cpl_test_zero( cpl_polynomial_compare(poly1,poly2,0) );
    
    i = 15;
    cpl_polynomial_solve_1d(poly1, 10*x1, &x, i);

    z = cpl_polynomial_eval_1d(poly1, x, &y);
    cpl_msg_info("", "(X+1)^15 (%" CPL_SIZE_FORMAT "): %g %g %g", i,
                 (x-x1)/DBL_EPSILON, z/DBL_EPSILON, y/DBL_EPSILON);
    cpl_test_rel( x, x1, DBL_EPSILON );
    cpl_test_abs( z, 0.0, DBL_EPSILON );
    cpl_test_abs( y, 0.0, DBL_EPSILON);

    /* Lots of round-off here, which depends on the long double precision */
    i = 5;
    cpl_test_zero( cpl_polynomial_solve_1d(poly1, -10*x1, &x, i) );
    z = cpl_polynomial_eval_1d(poly1, x, &y);
    cpl_msg_info("", "(X+1)^15 (%" CPL_SIZE_FORMAT ") " CPL_LDBL_EPSILON_STR
                 ": %g %g %g", i, x-x1,
                 (double)(z/CPL_LDBL_EPSILON),
                 (double)(y/CPL_LDBL_EPSILON));
    cpl_test_rel( x, x1, 0.35 );  /* alphaev56 */
    cpl_test_abs( z, 0.0, 18202.0   * DBL_EPSILON);
    cpl_test_abs( y, 0.0, 1554616.0 * DBL_EPSILON); /* alphaev56 */

    i = 15;
    eps = 2 * DBL_EPSILON;
    cpl_polynomial_set_coeff(poly1, &i, 1 + eps);

    cpl_test( cpl_polynomial_compare(poly1,poly2,-eps/2) < 0);
    cpl_test_error(CPL_ERROR_ILLEGAL_INPUT);
    cpl_test_zero( cpl_polynomial_compare(poly1,poly2,eps) );
    cpl_test( cpl_polynomial_compare(poly1,poly2,eps/2) > 0);

    cpl_polynomial_delete(poly2);

    poly2 = cpl_polynomial_new(POLY_DIM);
    cpl_polynomial_dump(poly2, stream);

    cpl_test_eq( cpl_polynomial_get_dimension(poly2), POLY_DIM);
    cpl_test_zero( cpl_polynomial_get_degree(poly2));

    /* Set and reset some (outragous) coefficient */
    for (j = 0; j <POLY_DIM;j++) expo[j] = j*VECTOR_SIZE + POLY_SIZE;
    cpl_test_zero( cpl_polynomial_set_coeff(poly2, expo, 1));
    cpl_test_abs( cpl_polynomial_get_coeff(poly2, expo), 1.0, 0.0);
    cpl_test_zero( cpl_polynomial_set_coeff(poly2, expo, 0));

    cpl_test_zero( cpl_polynomial_get_degree(poly2));

    for (i = 0; i <= 15 ; i++) {
        for (j = 0; j <POLY_DIM;j++) expo[j] = 0;
        expo[POLY_DIM/3] = i;
        cpl_test_zero( cpl_polynomial_set_coeff(poly2, expo,
                cpl_polynomial_get_coeff(poly1, &i)) );
    }

    /* Scale polynomial (with integer to avoid rounding)  */
    i = 15;
    cpl_polynomial_set_coeff(poly1, &i, 1);
    for (i = 0; i <= 15;i++)
        cpl_test_zero( cpl_polynomial_set_coeff(poly1, &i,
                 5.0 * cpl_polynomial_get_coeff(poly1, &i)) );

    i = 15;
    j = i - 1;
    cpl_test_zero( cpl_polynomial_shift_1d(poly1, 0,
            -cpl_polynomial_get_coeff(poly1, &j)
           /(cpl_polynomial_get_coeff(poly1, &i)*(double)i)));

    i = 15;
    cpl_test_abs( cpl_polynomial_get_coeff(poly1, &i), 5.0, 0.0);

    for (i = 14; i >=0; i--)
        cpl_test_abs( cpl_polynomial_get_coeff(poly1, &i), 0.0, 0.0);

    cpl_test_zero( cpl_polynomial_copy(poly1, poly2) );

    cpl_test_eq( cpl_polynomial_get_dimension(poly2), POLY_DIM);
    cpl_test_zero( cpl_polynomial_compare(poly1, poly2, 0) );

    i = 12;
    for (j = 0; j <POLY_DIM;j++) expo[j] = 0;
    expo[POLY_DIM/3] = i;
    cpl_test_zero( cpl_polynomial_set_coeff(poly2, expo,
             cpl_polynomial_get_coeff(poly1, expo) * (1 + DBL_EPSILON)) );

    cpl_test( cpl_polynomial_compare(poly1, poly2, 0) > 0 );

    cpl_test_zero( cpl_polynomial_set_coeff(poly2, expo,
             cpl_polynomial_get_coeff(poly1, expo)) );

    cpl_test_zero( cpl_polynomial_compare(poly1, poly2, 0) );

    x = cpl_polynomial_get_coeff(poly1, expo);

    for (j = 0; j <POLY_DIM;j++) expo[j] = 0;
    expo[POLY_DIM-1] = i;

    cpl_test_zero( cpl_polynomial_set_coeff(poly2, expo, x) );

    cpl_test( cpl_polynomial_compare(poly1, poly2, 0) > 0 );

    cpl_polynomial_delete(poly2);

    poly3 = cpl_polynomial_new(POLY_DIM-1);
    for (j = 0; j <POLY_DIM;j++) expo[j] = 0;
    cpl_test_zero( cpl_polynomial_set_coeff(poly3, expo, -1.0) );

    poly2 = cpl_polynomial_extract(poly1, 1, poly3);

    cpl_polynomial_delete(poly3);

    cpl_test_eq( cpl_polynomial_get_dimension(poly2),
                 cpl_polynomial_get_dimension(poly1) - 1);

    cpl_polynomial_dump(poly2, stream);

    cpl_polynomial_delete(poly1);
    cpl_polynomial_delete(poly2);

    poly1 = cpl_polynomial_new(1);

    for (i = 0; i <=15;i++)
        cpl_polynomial_set_coeff(poly1, &i, 0);

    i = 0;
    cpl_polynomial_set_coeff(poly1, &i, 2);
    i++;
    cpl_polynomial_set_coeff(poly1, &i, -3);
    i = 3;
    cpl_polynomial_set_coeff(poly1, &i, 1);

    cpl_test_eq( cpl_polynomial_get_degree(poly1), i);

    /* 1 is a double root */
    x1 = 1;
    x2 = -2;
    y2 = 9;

    i = 2;
    cpl_test_zero( cpl_polynomial_solve_1d(poly1, 10*x1, &x, i) );
    z = cpl_polynomial_eval_1d(poly1, x, &y);
    cpl_msg_info("", "(X-1)^2(X+2) (%g) " CPL_LDBL_EPSILON_STR
                 ": %g %g %g", x1,
                 (double)((x-x1)/x1/CPL_LDBL_EPSILON), z/DBL_EPSILON,
                 (double)(y/CPL_LDBL_EPSILON));
     /* Lots of loss: */
    cpl_test_rel( x, x1, 250e8*CPL_LDBL_EPSILON );
    cpl_test_abs( z, 0.0, fabs(x1)*DBL_EPSILON );
    /* p`(x) should be zero */
    cpl_test_abs( y, 0.0, 15e10*CPL_LDBL_EPSILON );

    i = 1;
    cpl_test_zero( cpl_polynomial_solve_1d(poly1, 10*x2, &x, i) );
    z = cpl_polynomial_eval_1d(poly1, x, &y);
    cpl_msg_info("", "(X-1)^2(X+2) (%g) [DBL_EPSILON]: %g %g %g", x2,
                 (x-x2)/x2/DBL_EPSILON, z/DBL_EPSILON, (y-y2)/y2/DBL_EPSILON);
    cpl_test_abs( x, x2, fabs(x2)*DBL_EPSILON );
    cpl_test_abs( z, 0.0, fabs(x2)*DBL_EPSILON );
    cpl_test_abs( y, y2, DBL_EPSILON );

    /* When eps != 0 multiplicity should be 1 and 3 */
    eps = 0.5;
    x1 = 1 + eps;
    x2 = 1;
    y_1 = 0.125;

    i = 0;
    cpl_polynomial_set_coeff(poly1, &i, 1 + eps);
    i++;
    cpl_polynomial_set_coeff(poly1, &i, -4 - 3 * eps);
    i++;
    cpl_polynomial_set_coeff(poly1, &i, 6 + 3 * eps);
    i++;
    cpl_polynomial_set_coeff(poly1, &i, -4 - eps);
    i++;
    cpl_polynomial_set_coeff(poly1, &i, 1);

    cpl_test_zero( cpl_polynomial_solve_1d(poly1, 10*x1, &x, 1) );
    z = cpl_polynomial_eval_1d(poly1, x, &y);
    cpl_msg_info("", "(X-1)^3(X-1-eps) (%g) " CPL_LDBL_EPSILON_STR
                 ": %g %g %g", x1,
                 (double)((x-x1)/x1/CPL_LDBL_EPSILON), z/DBL_EPSILON,
                 (double)((y-y_1)/y_1/DBL_EPSILON));
    cpl_test_rel( x, x1, 32768.0 * CPL_LDBL_EPSILON );

    /* Need factor 3 on P-III Coppermine */
    cpl_test_abs( z, 0.0, 3.0*fabs(x1)*DBL_EPSILON );
    cpl_test_abs( y, y_1, 70e3*CPL_LDBL_EPSILON );

    i = 3;
    /* Try to solve for the triple-root - cannot get a good solution */
    cpl_test_zero( cpl_polynomial_solve_1d(poly1, -10*x2, &x, i) );
    z = cpl_polynomial_eval_1d(poly1, x, &y);
    cpl_msg_info("", "(X-1)^3(X-1-eps) (%g) [DBL_EPSILON]: %g %g %g", x2,
                 (x-x2)/x2/DBL_EPSILON, z/DBL_EPSILON,
                 y/DBL_EPSILON);
    /*
       Huge variance in accuracy:
       [DBL_EPSILON]:
       i686:      -3.33964e+06  0.00195312 -0.000488281
       sun4u:     105           0          -3.67206e-12
       9000/785:  105           0          -3.67206e-12
       alpha:     3.93538e+09  -1          -5161

       [LDBL_EPSILON]:
       i686:      -6.83958e+09  4          -1
       sun4u:      1.21057e+20  0          -4.2336e+06
       9000/785:   1.21057e+20  0          -4.2336e+06
       alpha:      3.93538e+09 -1          -5161
     */

    cpl_test_abs( x2, x, fabs(x2)*5e9*DBL_EPSILON );
    cpl_test_abs( z, 0.0, 1.5*fabs(x2)*DBL_EPSILON );
    cpl_test_abs( y, 0.0, 6e3*DBL_EPSILON );

    /* Try to decrease eps... */
    eps = 0.1;
    x1 = 1 + eps;
    x2 = 1;
    i = 0;
    cpl_polynomial_set_coeff(poly1, &i, 1 + eps);
    i++;
    cpl_polynomial_set_coeff(poly1, &i, -4 - 3 * eps);
    i++;
    cpl_polynomial_set_coeff(poly1, &i, 6 + 3 * eps);
    i++;
    cpl_polynomial_set_coeff(poly1, &i, -4 - eps);
    i++;
    cpl_polynomial_set_coeff(poly1, &i, 1);

    /* Cannot get a good solution */
    cpl_test_abs(cpl_polynomial_eval_1d(poly1, x1, &y_1), 0.0,
                            2.45 * DBL_EPSILON);
    cpl_test_zero( cpl_polynomial_solve_1d(poly1, 10*x1, &x, 1) );
    z = cpl_polynomial_eval_1d(poly1, x, &y);
    cpl_msg_info("", "(X-1)^3(X-1-eps) (%g) [DBL_EPSILON]: %g %g %g", x1,
                 (x-x1)/x1/DBL_EPSILON, z/DBL_EPSILON, (y-y_1)/y_1/DBL_EPSILON);
    cpl_test_abs( x, x1, fabs(x1)*2220 * DBL_EPSILON );
    cpl_test_abs( z, 0.0, fabs(x1) * 1e-5 * DBL_EPSILON );
    cpl_test_abs( y, y_1, 147e3*fabs(y_1)*DBL_EPSILON );


    /* Try to solve for the triple-root - cannot get a good solution */
    i = 3;
    cpl_test_zero( cpl_polynomial_solve_1d(poly1, -10*x2, &x, 1) );
    z = cpl_polynomial_eval_1d(poly1, x, &y);
    cpl_msg_info("", "(X-1)^3(X-1-eps) (%g) [DBL_EPSILON]: %g %g %g", x2,
                 (x-x2)/x2/DBL_EPSILON, z/DBL_EPSILON, y/DBL_EPSILON);

    /*
       Small variance in accuracy:
       [DBL_EPSILON]:
       i686:       -5.72688e+10   2.92578   -218506
       sun4u:      -5.72658e+10   2.92597   -218483
       9000/785:   -5.72658e+10   2.92597   -218483
       alpha:      -5.18581e+10   4         -179161
     */

    cpl_test_abs( x2, x, fabs(x2)*58e9*DBL_EPSILON );
    cpl_test_abs( z, 0.0, fabs(x2)*4*DBL_EPSILON );
    cpl_test_abs( y, 0.0, 22e4*DBL_EPSILON );


    /* An approximate dispersion relation with central wavelength at
       2.26463 micron (ISAAC.2002-11-13T05:30:53.242) */
    i = 0;
    cpl_polynomial_set_coeff(poly1, &i, 22016 - 22646.3);
    i++;
    cpl_polynomial_set_coeff(poly1, &i, 1.20695);
    i++;
    cpl_polynomial_set_coeff(poly1, &i, 3.72002e-05);
    i++;
    cpl_polynomial_set_coeff(poly1, &i, -3.08727e-08);
    i++;
    cpl_polynomial_set_coeff(poly1, &i, 0);

    cpl_test_eq( cpl_polynomial_get_degree(poly1), 3);

    x1 = 512.5; /* Central wavelength should be here */

    i = 1;
    cpl_polynomial_eval_1d(poly1, x1, &y_1);
    cpl_test_zero( cpl_polynomial_solve_1d(poly1, 0, &x, i) );
    z1 = cpl_polynomial_eval_1d(poly1, x, &y);
    cpl_msg_info("", "ISAAC (%g) [FLT_EPSILON]: %g %g %g", x1,
                 (x-x1)/x1/FLT_EPSILON, z1/DBL_EPSILON, (y-y_1)/y_1/FLT_EPSILON);

    cpl_test_abs( z1, 0.0, fabs(x1)*0.6*DBL_EPSILON );
    cpl_test_abs( y, y_1, 1e3*fabs(y_1)*FLT_EPSILON );

    /* Shift the polynomial 12.5 places */
    eps = 12.5;
    x2 = x1 - eps;
    x1 = x;
    cpl_test_zero( cpl_polynomial_shift_1d(poly1, 0, eps) );

    cpl_polynomial_eval_1d(poly1, x2, &y2);
    cpl_test_abs( y_1, y2, y_1*DBL_EPSILON);
    cpl_test_zero( cpl_polynomial_solve_1d(poly1, 0, &x, i) );

    y_1 = y;
    z2 = cpl_polynomial_eval_1d(poly1, x, &y);
    cpl_msg_info("", "Shift (%g) [DBL_EPSILON]: %g %g %g %g", eps,
                 (x-x2)/x2/DBL_EPSILON, z2/DBL_EPSILON,
                 (x1-x-eps)/x2/DBL_EPSILON, (y_1-y)/y_1/DBL_EPSILON);

    /* Root and derivative should be preserved to machine accuracy */
    cpl_test_abs( x1-x, eps, fabs(x2)*1.024*DBL_EPSILON);
    cpl_test_abs( y_1, y, 2*fabs(y_1)*DBL_EPSILON);

    x1 = 1e3;
    x2 = -1e2;

    xmax = cpl_polynomial_eval_1d(poly1, x2, NULL)
         - cpl_polynomial_eval_1d(poly1, x1, NULL)
         - cpl_polynomial_eval_1d_diff(poly1, x2, x1, &y);

    cpl_msg_info("", "Diff (%g:%g) [DBL_EPSILON]: %g ", x2,x1,xmax/DBL_EPSILON);

    cpl_test_abs(xmax, 0.0, 1.05 * fabs(x1)*DBL_EPSILON);
    cpl_test_abs(cpl_polynomial_eval_1d(poly1, x2, NULL), y, DBL_EPSILON);

    cpl_polynomial_delete(poly1);

    /* A small performance test - twice to test DFS02166 */
    for (k=0; k < 1 ; k++) {

        vec = cpl_vector_new(POLY_DIM);
        cpl_test_zero( cpl_vector_fill(vec, 1));

        poly2 = cpl_polynomial_new(POLY_DIM);
        for (i = 0; i < POLY_SIZE; i++) {
            for (j = 0; j < POLY_DIM; j++) expo[j] = j + i;
            cpl_test_zero( cpl_polynomial_set_coeff(poly2, expo, 1));
        }

        for (i = 0; i < POLY_SIZE*VECTOR_SIZE; i++) {
            eps = cpl_polynomial_eval(poly2, vec);
        }
        cpl_polynomial_delete(poly2);
        cpl_vector_delete(vec);

    }
   
    for (k=0; k < 1 ; k++) {

        vec = cpl_vector_new(POLY_DIM);
        cpl_test_zero( cpl_vector_fill(vec, 1));

        poly2 = cpl_polynomial_new(POLY_DIM);
        for (i = 0; i < POLY_SIZE; i++) {
            for (j = 0; j < POLY_DIM; j++) expo[j] = j + i;
            cpl_test_zero( cpl_polynomial_set_coeff(poly2, expo, 1));
        }

        for (i = 0; i < POLY_SIZE*VECTOR_SIZE; i++) {
            eps = cpl_polynomial_eval(poly2, vec);
        }
        cpl_polynomial_delete(poly2);
        cpl_vector_delete(vec);

    }

    poly1 = cpl_polynomial_fit_test_2d(stream, CPL_TRUE);
    cpl_polynomial_delete(poly1);
    poly1 = cpl_polynomial_fit_test_2d(stream, CPL_FALSE);

    poly3 = cpl_polynomial_new(1);

    cpl_test_null( cpl_polynomial_extract(poly1, -1, poly3) );

    cpl_test_error(CPL_ERROR_ILLEGAL_INPUT);

    cpl_test_null( cpl_polynomial_extract(poly1, 2, poly3) );

    cpl_test_error(CPL_ERROR_ACCESS_OUT_OF_RANGE);

    cpl_polynomial_dump(poly1, stream);

    poly2 = cpl_polynomial_extract(poly1, 0, poly3);

    cpl_test_eq( cpl_polynomial_get_dimension(poly2),
                 cpl_polynomial_get_dimension(poly1) - 1);

    /* poly1 comes from cpl_polynomial_fit_2d_create() */
    cpl_test_eq( cpl_polynomial_get_degree(poly2),
                 cpl_polynomial_get_degree(poly1));

    cpl_polynomial_dump(poly2, stream);

    i = 0;
    eps = cpl_polynomial_get_coeff(poly2, &i);
    cpl_polynomial_delete(poly2);

    poly2 = cpl_polynomial_extract(poly1, 1, poly3);

    cpl_test_eq( cpl_polynomial_get_dimension(poly2),
                 cpl_polynomial_get_dimension(poly1) - 1);

    /* poly1 comes from cpl_polynomial_fit_2d_create() */
    cpl_test_eq( cpl_polynomial_get_degree(poly2),
                 cpl_polynomial_get_degree(poly1));

    cpl_polynomial_dump(poly2, stream);

    i = 0;
    xmax = eps - cpl_polynomial_get_coeff(poly2, &i);
    cpl_polynomial_delete(poly2);

    cpl_test_abs( xmax, 0.0, 0.0);

    /* Use this block to benchmark 2D cpl_polynomial_eval() */
    vec = cpl_vector_new(2);

    cpl_test_zero( cpl_vector_fill(vec, 1));

    for (k=0; k < VECTOR_SIZE*VECTOR_SIZE; k++) 
        eps = cpl_polynomial_eval(poly1, vec);


    cpl_vector_delete(vec);
    cpl_polynomial_delete(poly1);


    /* Create and differentiate a 1d-polynomial with uniform coefficients */
    poly1 = cpl_polynomial_new(1);

    x = CPL_MATH_PI_4;
    for (i=0; i < POLY_SIZE; i++) {

        cpl_test_zero( cpl_polynomial_set_coeff(poly1, &i, x));
    }

    x1 = x;
    for (i=POLY_SIZE-1; i > 0; i--) {

        x1 *= (double)i;
        cpl_test_zero( cpl_polynomial_derivative(poly1, 0) );

    }

    cpl_test_zero( cpl_polynomial_get_degree(poly1) );

     /* Verify the value of the constant term */
    cpl_test_abs( x1, cpl_polynomial_get_coeff(poly1, &i), 0.0);

    cpl_polynomial_delete(poly1);

    /* Create and collapse and differentiate a 2d-polynomial with
       uniform coefficients */
    poly1 = cpl_polynomial_new(2);

    x = CPL_MATH_PI_4;
    y = CPL_MATH_E;
    x1 = 1.0;
    for (i=0; i < POLY_SIZE; i++) {
        for (j=0; j < POLY_SIZE; j++) {
            expo[0] = i;
            expo[1] = j;

            cpl_test_zero( cpl_polynomial_set_coeff(poly1, expo, x));

        }
        if (i > 0) x1 *= (double)i;
    }

    i = 0;
    cpl_test_zero( cpl_polynomial_set_coeff(poly3, &i, y));

    for (j = 0; j < 2; j++) {

        poly2 = cpl_polynomial_extract(poly1, j, poly3);

        z1 = 0.0;
        for (i=0; i < POLY_SIZE; i++) {
            z1 += pow(y, (double)i);
        }

        z1 *= x;

        for (i=0; i < POLY_SIZE; i++) {
            z = cpl_polynomial_get_coeff(poly2, &i);
            cpl_test_abs( z, z1, 2.0*POLY_SIZE*FLT_EPSILON);
        }

        cpl_test_zero( cpl_polynomial_derivative(poly2, 0) );
        i = 0;
        z = cpl_polynomial_get_coeff(poly2, &i);

        for (i=1; i < POLY_SIZE-1; i++) {
            z1 = cpl_polynomial_get_coeff(poly2, &i);
            cpl_test_abs( z * (double)(i+1), z1, 11.0*POLY_SIZE*FLT_EPSILON);
        }

        cpl_polynomial_delete(poly2);
    }

    cpl_polynomial_delete(poly3);

    for (i=POLY_SIZE-1; i > 0; i--) {

        cpl_test_zero( cpl_polynomial_derivative(poly1, 0) );

        cpl_test_zero( cpl_polynomial_derivative(poly1, 1) );

    }

    cpl_test_zero( cpl_polynomial_get_degree(poly1) );

    expo[0] = expo[1] = 0;

    x2 = cpl_polynomial_get_coeff(poly1, expo);
    /* The constant term is huge at this point
       - reduce by a factor eps which is inaccurate */
    eps = x2 * DBL_EPSILON > 1 ? x2 * DBL_EPSILON : 1;
    xmax = x2/eps - x * x1 * x1/eps;
    cpl_test_abs( xmax, 0.0, 1.0);


    cpl_polynomial_delete(poly1);

    /* Test for DFS06121 */
    poly2 = cpl_polynomial_new(2);
    expo[0] = 0;

    expo[1] = 0;
    cpl_test_zero( cpl_polynomial_set_coeff(poly2, expo, -256.0) );

    expo[1] = 1;
    cpl_test_zero( cpl_polynomial_set_coeff(poly2, expo, 1.0) );

    expo[1] = 2;
    cpl_test_zero( cpl_polynomial_set_coeff(poly2, expo, 0.001) );

    cpl_test_zero( cpl_polynomial_derivative(poly2, 0) );

    cpl_test_zero( cpl_polynomial_get_degree(poly2) );

    cpl_test_zero( cpl_polynomial_derivative(poly2, 0) );

    cpl_test_zero( cpl_polynomial_get_degree(poly2) );

    cpl_test_zero( cpl_polynomial_derivative(poly2, 1) );

    cpl_test_zero( cpl_polynomial_get_degree(poly2) );

    cpl_polynomial_delete(poly2);


    if (cpl_msg_get_level() <= CPL_MSG_INFO) 
        cpl_polynomial_fit_bench_2d(1, 5 * POLY_SIZE, POLY_SIZE);

    if (mse_secs > 0.0)
        cpl_msg_info("","Speed of cpl_vector_fill_polynomial_fit_residual() "
                     "over %g secs [Mflop/s]: %g", mse_secs,
                     (double)mse_flops/mse_secs/1e6);

    if (stream != stdout) cpl_test_zero( fclose(stream) );

    return cpl_test_end(0);
}

/**@}*/

/*----------------------------------------------------------------------------*/
/**
  @internal
  @brief      Test the CPL function
  @return    void

 */
/*----------------------------------------------------------------------------*/
static void cpl_vector_fill_polynomial_fit_residual_test(void)
{

    cpl_error_code    error;

    /* Test 1: NULL input */
    error = cpl_vector_fill_polynomial_fit_residual(NULL, NULL, NULL, NULL,
                                                    NULL, NULL);

    cpl_test_eq_error(error, CPL_ERROR_NULL_INPUT);

    return;

}


/*----------------------------------------------------------------------------*/
/**
  @internal
  @brief    Test cpl_polynomial_fit() in 2D
  @param    stream  A stream to dump to
  @param    dimdeg  Passed to cpl_polynomial_fit()
  @return   void

 */
/*----------------------------------------------------------------------------*/
static cpl_polynomial * cpl_polynomial_fit_test_2d(FILE * stream,
                                                   cpl_boolean dimdeg)
{

    cpl_matrix     * xy_pos;
    cpl_vector     * xpoint;
    const double     xy_offset = 100;
    cpl_size         degree;
    cpl_polynomial * poly1 = cpl_polynomial_new(2);
    cpl_polynomial * poly2 = cpl_polynomial_new(2);
    cpl_vector     * vec;
    double           xmax = 0.0; /* Maximum rounding error on x */
    double           ymax = 0.0; /* Maximum rounding error on y */
    double           eps;
    double           x, y;
    const cpl_size   zerodeg2[] = {0, 0};
    cpl_size         i, j, k;
    cpl_error_code   error;


    /* Try to fit increasing degrees to f(x,y) = sqrt(x)*log(1+y)
       - with exactly as many points as there are coefficient to determine
         thus the residual should be exactly zero.
    */

    xpoint = cpl_vector_new(2);
    /* f(1/4,sqrt(e)-1) = 1/4 */
    cpl_vector_set(xpoint, 0, 0.25+xy_offset);
    cpl_vector_set(xpoint, 1, sqrt(CPL_MATH_E)-1+xy_offset);

    for (degree = 0; degree < POLY_SIZE; degree++) {
        const cpl_size maxdeg[2] = {degree, degree};
        const cpl_size nc = dimdeg ? (degree+1)*(degree+1)
            : (degree+1)*(degree+2)/2;

        k = 0;

        vec = cpl_vector_new(nc);
        xy_pos = cpl_matrix_new(2, nc);

        for (i=0; i < 1 + degree; i++) {
            for (j=0; j < 1 + (dimdeg ? degree : i); j++, k++) {

                cpl_test_zero( cpl_matrix_set(xy_pos, 0, k,
                                              (double)i+xy_offset) );
                cpl_test_zero( cpl_matrix_set(xy_pos, 1, k,
                                              (double)j+xy_offset) );
                cpl_test_zero( cpl_vector_set(vec,   k,
                                              sqrt((double)i)*
                                              log((double)(j+1))));
            }
        }

        cpl_test_eq( k, nc );

        error = cpl_polynomial_fit_cmp(poly2, xy_pos, NULL, vec, NULL,
                                       dimdeg, zerodeg2, maxdeg);
        cpl_test_eq_error(error, CPL_ERROR_NONE);

        cpl_test_eq( cpl_polynomial_get_dimension(poly2), 2 );
        cpl_test_eq( cpl_polynomial_get_degree(poly2),
                     dimdeg ? 2 * degree : degree );

        cpl_polynomial_dump(poly2, stream);

        eps = cpl_vector_get_mse(vec, poly2, xy_pos, NULL);

        /* The increase in mse must be bound */
        if (xmax > 0) cpl_test_leq( eps/xmax, 5.39953e+09);

        cpl_vector_delete(vec);
        cpl_matrix_delete(xy_pos);

        cpl_msg_info(cpl_func,"2D-Polynomial with degree %" CPL_SIZE_FORMAT
                     " (dimdeg=%d) fit of a %" CPL_SIZE_FORMAT "-point "
                     "dataset has a mean square error ratio to a %"
                     CPL_SIZE_FORMAT "-degree fit: %g (%g > %g)", degree,
                     dimdeg, nc, degree-1, xmax > 0.0 ? eps/xmax : 0.0,
                     eps, xmax);
        xmax = eps;

        eps = cpl_polynomial_eval(poly2, xpoint);

        if (nc < (dimdeg ? 25 : 40))
            cpl_test_abs( eps, 0.25, fabs(0.25 - ymax));

        if (fabs(0.25-eps) >= fabs(0.25 - ymax) && degree > 0) {
            /* Should be able to fit at least a 5-degree polynomial
               with increased accuracy - and without error-margin */
            cpl_msg_info(cpl_func,"2D-Polynomial with degree %" CPL_SIZE_FORMAT
                         " fit of a %" CPL_SIZE_FORMAT "-point dataset has a "
                         "greater residual than a %" CPL_SIZE_FORMAT "-degree "
                         "fit to a %" CPL_SIZE_FORMAT "-point dataset: "
                         "fabs(%g) > fabs(%g)", degree, nc, degree-1,
                         degree*(degree+1)/2, eps-0.25, ymax-0.25);
            break;
        }

        ymax = eps;
        

    }

    /* Try to fit increasing degrees to f(x,y) = sqrt(x)*log(1+y)
       - with a constant, overdetermined number of points
       - The residual should decrease with increasing degree until the system
       becomes near singular */

    /* f(1/4,sqrt(e)-1) = 1/4 */
    cpl_vector_set(xpoint, 0, 0.25+xy_offset);
    cpl_vector_set(xpoint, 1, sqrt(CPL_MATH_E)-1-xy_offset);

    k = 0;

    vec    = cpl_vector_new(   2 * POLY_SIZE * 2 * POLY_SIZE);
    xy_pos = cpl_matrix_new(2, 2 * POLY_SIZE * 2 * POLY_SIZE);

    for (i=0, k = 0; i < 2 * POLY_SIZE; i++) {
        for (j=0; j < 2 * POLY_SIZE; j++, k++) {

            x = (double) i * 0.5;
            y = (double) j * 2.0;

            cpl_test_zero( cpl_matrix_set(xy_pos, 0, k, (double)i+xy_offset) );
            cpl_test_zero( cpl_matrix_set(xy_pos, 1, k, (double)j-xy_offset) );

            cpl_test_zero( cpl_vector_set(vec,   k, sqrt(x)*log(y+1.0)));
        }
    }

    cpl_test_eq( 2 * POLY_SIZE * 2 * POLY_SIZE, k );

    ymax = 0;
    for (degree = 0; degree < POLY_SIZE; degree++) {
        const cpl_size maxdeg[2] = {degree, degree};
        const cpl_size nc = 2 * POLY_SIZE * 2 * POLY_SIZE;
        double mse;

        error = cpl_polynomial_fit_cmp(poly2, xy_pos, NULL, vec, NULL,
                                   dimdeg, zerodeg2, maxdeg);
        cpl_test_eq_error(error, CPL_ERROR_NONE);

        cpl_test_eq( cpl_polynomial_get_dimension(poly2), 2 );
        cpl_test_eq( cpl_polynomial_get_degree(poly2),
                     dimdeg ? 2 * degree : degree );

        cpl_polynomial_dump(poly2, stream);

        mse = cpl_vector_get_mse(vec, poly2, xy_pos, NULL);

        eps = cpl_polynomial_eval(poly2, xpoint);

        cpl_msg_info(cpl_func, "2D-Polynomial with degree %" CPL_SIZE_FORMAT
                     " fit of a %" CPL_SIZE_FORMAT "-point dataset "
                     "has a mean square error: %g (P0-error=%g)",
                     degree, nc, mse, 0.25-eps);

        /* mse must decrease */
        if (degree > 0) {
            if (fabs(eps-0.25) > fabs(ymax - 0.25))
                cpl_msg_info(cpl_func, "2D-Polynomial with degree %"
                             CPL_SIZE_FORMAT " fit of a %" CPL_SIZE_FORMAT
                             "-point dataset has a larger error than a %"
                             CPL_SIZE_FORMAT "-degree fit: fabs(%g-0.25) > "
                             "fabs(%g-0.25)",
                             degree, nc, degree - 1, eps, ymax);
            if (mse > xmax) {
                cpl_msg_info(cpl_func, "2D-Polynomial with degree %"
                             CPL_SIZE_FORMAT " fit of a %" CPL_SIZE_FORMAT
                             "-point dataset has a larger mean square error "
                             "than a %" CPL_SIZE_FORMAT "-degree fit: %g > %g",
                             degree, nc, degree - 1, mse, xmax);
                break;
            }
        }

        xmax = mse;
        ymax = eps;

        cpl_test_zero(cpl_polynomial_copy(poly1, poly2));

    }

    cpl_vector_delete(vec);
    cpl_matrix_delete(xy_pos);

    cpl_vector_delete(xpoint);
    cpl_polynomial_delete(poly2);

    return poly1;

}


/*----------------------------------------------------------------------------*/
/**
  @internal
  @brief    Test cpl_polynomial_fit() in 1D
  @param    stream  A stream to dump to
  @return   void

 */
/*----------------------------------------------------------------------------*/
static void cpl_polynomial_fit_test_1d(FILE * stream)
{

    const double      dvec6[] = {1, 3, 5, 2, 4, 6};
    const double      dtay6[] = {0, 2*4, 2*20, 2*1, 2*10, 2*35};
    const double      dvec4[] = {1, 3, 4, 6};
    const double      dtay4[] = {0, 2*4, 2*10, 2*35};
    const double      dsq6[]  = {1, 9, 25, 4, 16, 36};
    cpl_matrix      * samppos1 = cpl_matrix_wrap(1, 6, (double*)dvec6);
    cpl_vector      * taylor = cpl_vector_wrap(6, (double*)dtay6);
    cpl_matrix      * samppos14 = cpl_matrix_wrap(1, 4, (double*)dvec4);
    cpl_vector      * taylor4 = cpl_vector_wrap(4, (double*)dtay4);
    cpl_matrix      * samppos;
    cpl_vector      * fitvals;
    cpl_polynomial  * poly1a = cpl_polynomial_new(1);
    cpl_polynomial  * poly1b = cpl_polynomial_new(1);
    cpl_polynomial  * poly2 = cpl_polynomial_new(2);
    const cpl_boolean symsamp = CPL_TRUE;
    const cpl_size    zerodeg = 0;
    const cpl_size    sqdeg  = 2;
    const cpl_size    maxdeg = 3;
    cpl_size          mindeg, errdeg;
    double            eps, rechisq;
    double            xmax; /* Maximum rounding error on x */
    double            ymax; /* Maximum rounding error on y */
    double            zmax;
    cpl_error_code    error;
    cpl_size          i, j;


    /* Test 1: NULL input */
    error = cpl_polynomial_fit_cmp(NULL, NULL, NULL, NULL, NULL, CPL_FALSE,
                               NULL, NULL);
    cpl_test_eq_error(error, CPL_ERROR_NULL_INPUT);

    error = cpl_polynomial_fit_cmp(NULL, samppos1, NULL, taylor, NULL,
                               CPL_FALSE, NULL, &maxdeg);
    cpl_test_eq_error(error, CPL_ERROR_NULL_INPUT);

    error = cpl_polynomial_fit_cmp(poly1a, NULL, NULL, taylor, NULL,
                               CPL_FALSE, NULL, &maxdeg);
    cpl_test_eq_error(error, CPL_ERROR_NULL_INPUT);

    error = cpl_polynomial_fit_cmp(poly1a, samppos1, NULL, NULL, NULL,
                               CPL_FALSE, NULL, &maxdeg);
    cpl_test_eq_error(error, CPL_ERROR_NULL_INPUT);

    error = cpl_polynomial_fit_cmp(poly1a, samppos1, NULL, taylor, NULL,
                               CPL_FALSE, NULL, NULL);
    cpl_test_eq_error(error, CPL_ERROR_NULL_INPUT);

    /* Test 1a: negative maxdeg */
    errdeg = -1;
    error = cpl_polynomial_fit_cmp(poly1a, samppos1, NULL, taylor, NULL,
                               CPL_FALSE, NULL, &errdeg);
    cpl_test_eq_error(error, CPL_ERROR_ILLEGAL_INPUT);

    /* Test 1b: maxdeg less than mindeg*/
    errdeg = 0;
    error = cpl_polynomial_fit_cmp(poly1a, samppos1, NULL, taylor, NULL,
                               CPL_FALSE, &maxdeg, &errdeg);
    cpl_test_eq_error(error, CPL_ERROR_ILLEGAL_INPUT);

    /* Test 1c: Wrong dimension of poly */

    error = cpl_polynomial_fit_cmp(poly2, samppos1, NULL, taylor, NULL,
                               CPL_FALSE, NULL, &maxdeg);
    cpl_test_eq_error(error, CPL_ERROR_INCOMPATIBLE_INPUT);

    /* Test 2: Unordered insertion */

    error = cpl_polynomial_fit_cmp(poly1a, samppos1, NULL, taylor, NULL,
                               CPL_FALSE, &zerodeg, &maxdeg);

    cpl_test_eq_error(error, CPL_ERROR_NONE);

    error = cpl_polynomial_fit_cmp(poly1a, samppos1, NULL, taylor, NULL,
                               CPL_FALSE, &zerodeg, &maxdeg);

    cpl_test_eq_error(error, CPL_ERROR_NONE);

    eps = cpl_vector_get_mse(taylor, poly1a, samppos1, &rechisq);

    cpl_test_leq(0.0, rechisq);

    cpl_test_abs( eps, 0.0, 4359*DBL_EPSILON*DBL_EPSILON); /* alphaev56 */

    /* Test 3: Symmetric 1D sampling (also test dimdeg and reset of preset
               1D-coeffs) */
    errdeg = maxdeg + 1;
    cpl_test_zero( cpl_polynomial_set_coeff(poly1b, &errdeg, 1.0) );
    error = cpl_polynomial_fit_cmp(poly1b, samppos1, &symsamp, taylor, NULL,
                               CPL_TRUE, &zerodeg, &maxdeg);

    cpl_test_eq_error(error, CPL_ERROR_NONE);

    cpl_test_polynomial_abs(poly1a, poly1b, DBL_EPSILON);

    cpl_test_abs( cpl_vector_get_mse(taylor, poly1b, samppos1, NULL), 0.0,
                  fabs(eps));

    /* Test 3A: Same, except mindeg set to 1 */
    mindeg = 1;
    cpl_test_zero( cpl_polynomial_set_coeff(poly1b, &errdeg, 1.0) );
    error = cpl_polynomial_fit_cmp(poly1b, samppos1, &symsamp, taylor, NULL,
                               CPL_TRUE, &mindeg, &maxdeg);

    cpl_test_eq_error(error, CPL_ERROR_NONE);

    cpl_test_eq(cpl_polynomial_get_degree(poly1b), maxdeg);
    cpl_test_abs(cpl_polynomial_get_coeff(poly1b, &zerodeg), 0.0, 0.0);

    cpl_test_zero( cpl_polynomial_set_coeff(poly1a, &zerodeg, 0.0) );

    cpl_test_polynomial_abs(poly1a, poly1b, 700.0 * DBL_EPSILON);

    cpl_test_abs( cpl_vector_get_mse(taylor, poly1b, samppos1, NULL), 0.0,
                  70.0 * fabs(eps));

    /* Test 3B: Symmetric, non-equidistant 1D sampling (also test dimdeg and
                reset of preset 1D-coeffs) */
    error = cpl_polynomial_fit_cmp(poly1a, samppos14, NULL, taylor4, NULL,
                               CPL_TRUE, &zerodeg, &maxdeg);

    cpl_test_eq_error(error, CPL_ERROR_NONE);

    error = cpl_polynomial_fit_cmp(poly1b, samppos14, &symsamp, taylor4, NULL,
                               CPL_TRUE, &zerodeg, &maxdeg);

    cpl_test_eq_error(error, CPL_ERROR_NONE);

    cpl_test_polynomial_abs(poly1a, poly1b, DBL_EPSILON);

    /* Test 3C: Same except mindeg set to 1 */
    error = cpl_polynomial_fit_cmp(poly1b, samppos14, &symsamp, taylor4, NULL,
                               CPL_TRUE, &mindeg, &maxdeg);

    cpl_test_eq_error(error, CPL_ERROR_NONE);

    cpl_test_eq(cpl_polynomial_get_degree(poly1b), maxdeg);
    cpl_test_abs(cpl_polynomial_get_coeff(poly1b, &zerodeg), 0.0, 0.0);

    cpl_test_zero( cpl_polynomial_set_coeff(poly1a, &zerodeg, 0.0) );

    cpl_test_polynomial_abs(poly1a, poly1b, 600.0 * DBL_EPSILON);

    /* Not overdetermined, so must pass NULL for rechisq */
    eps = cpl_vector_get_mse(taylor4, poly1a, samppos14, NULL);

    cpl_test_error(CPL_ERROR_NONE);

    cpl_test_abs( cpl_vector_get_mse(taylor4, poly1b, samppos14, NULL), 0.0,
                  283.0 * fabs(eps));

    /* Test 4: Only one distinct sampling point */
    samppos = cpl_matrix_new(1, 6);
    cpl_test_zero(cpl_matrix_fill(samppos, 1.0));

    /*  - should not be able to fit with only one distinct x-value */    
    error = cpl_polynomial_fit_cmp(poly1a, samppos, &symsamp, taylor, NULL,
                               CPL_TRUE, &zerodeg, &maxdeg);
    cpl_test_eq_error(error, CPL_ERROR_SINGULAR_MATRIX);

    /* Test 4B: - unless the degree is 0 */
    error = cpl_polynomial_fit_cmp(poly1a, samppos, &symsamp, taylor, NULL,
                               CPL_TRUE, &zerodeg, &zerodeg);
    cpl_test_eq_error(error, CPL_ERROR_NONE);

    cpl_test_zero(cpl_polynomial_get_degree(poly1a));

    cpl_test_abs( cpl_polynomial_get_coeff(poly1a, &zerodeg),
                  cpl_vector_get_mean(taylor), DBL_EPSILON);


    /* Test 4B: - or unless mindeg equals the degree */
    for (i = 1; i < POLY_SIZE; i++) {
        error = cpl_polynomial_fit_cmp(poly1a, samppos, &symsamp, taylor, NULL,
                                   CPL_TRUE, &i, &i);
        cpl_test_eq_error(error, CPL_ERROR_NONE);

        cpl_test_eq(cpl_polynomial_get_degree(poly1a), i);

        /* FIXME: Weak test, sampling positions are all at 1.0 */
        /* Another test should be added */
        cpl_test_abs( cpl_polynomial_get_coeff(poly1a, &i),
                      cpl_vector_get_mean(taylor), 16.0 * DBL_EPSILON);

        error = cpl_polynomial_set_coeff(poly1a, &i, 0.0);
        cpl_test_eq_error(error, CPL_ERROR_NONE);

        cpl_test_zero(cpl_polynomial_get_degree(poly1a));

    }

    /* Test 5: Try to fit j-coefficients to a sqrt() sampled j
                times at 0,1,...,j-1
       - since sqrt(0) == 0, the constant coefficient of the fit
         should be zero */

    fitvals = cpl_vector_new(1);

    xmax = 0.0;
    ymax = -DBL_EPSILON;
    zmax = 0.0;
    for (j=1; j <= POLY_SIZE; j++) {
        const cpl_size degree = j - 1;
        cpl_test_zero( cpl_matrix_set_size(samppos, 1, j) );
        cpl_test_zero( cpl_vector_set_size(fitvals,    j) );
        for (i=0; i < j; i++) {
            cpl_test_zero( cpl_matrix_set(samppos, 0, i, (double)i) );
            cpl_test_zero( cpl_vector_set(fitvals, i, sqrt((double)i)) );
        }

        error = cpl_polynomial_fit_cmp(poly1a, samppos, &symsamp, fitvals, NULL,
                                   CPL_TRUE, &zerodeg, &degree);

        if (error == CPL_ERROR_SINGULAR_MATRIX) {

            cpl_msg_info("FIXME","1D-Polynomial fit of a %" CPL_SIZE_FORMAT
                         "-point dataset with degree %" CPL_SIZE_FORMAT
                         " leads to a (near) singular system of equations",
                         j, degree);
            cpl_test_eq_error(error, CPL_ERROR_SINGULAR_MATRIX);
            break;
        }

        cpl_test_eq_error( error, CPL_ERROR_NONE );

        cpl_test_eq( cpl_polynomial_get_dimension(poly1a), 1 );
        cpl_test_eq( cpl_polynomial_get_degree(poly1a), degree );

        cpl_polynomial_dump(poly1a, stream);

        eps = cpl_vector_get_mse(fitvals, poly1a, samppos, NULL);

        /* The increase in mse must be bound 
           (i686 can manage with a bound of 957 instead of 9016) */
        if (xmax > 0) cpl_test_leq( eps/xmax, 9016);
        xmax = eps;

        eps = cpl_polynomial_get_coeff(poly1a, &zerodeg);
        if (fabs(eps) > fabs(zmax)) {
            cpl_msg_info(cpl_func,"1D-Polynomial with degree %" CPL_SIZE_FORMAT
                         " fit of a %" CPL_SIZE_FORMAT "-point dataset has a "
                         "greater error on P0 than a %" CPL_SIZE_FORMAT
                         "-degree fit to a %" CPL_SIZE_FORMAT "-point dataset: "
                         "fabs(%g) > fabs(%g)",
                         degree, j, degree-1, j-1, eps, zmax);
        }
        /* Should loose at most one decimal digit per degree
           fabs(eps) < DBL_EPSILON * 10 ** (degree-2)  */
        cpl_test_abs( eps, 0.0, DBL_EPSILON * pow(10, (double)(j-3)) );
        zmax = eps;

        /* Compute approximation to sqrt(0.25)
           - this will systematically be too low,
             but less and less so with increasing degree
           - until the approximation goes bad */
        eps = cpl_polynomial_eval_1d(poly1a, 0.25, NULL);

        if (j < 18) cpl_test_abs( eps, 0.5, fabs(0.5 - ymax));
        if (eps <= ymax) {
            /* Should be able to fit at least an 18-degree polynomial
               with increased accuracy - and without error-margin */
            cpl_msg_info(cpl_func, "1D-Polynomial with degree %"
                         CPL_SIZE_FORMAT " fit of a %" CPL_SIZE_FORMAT "-point "
                         "dataset has a greater residual than a %"
                         CPL_SIZE_FORMAT "-degree fit to a %" CPL_SIZE_FORMAT
                         "-point dataset: %g > %g",
                         degree, j, degree-1, j-1, 0.5-eps, 0.5-ymax);
            break;
        }

        ymax = eps;

    }

    /* And the mse itself must be bound */
    cpl_test_leq( eps, 0.411456 * cpl_error_margin);

    /* Test 6: Try to fit increasing degrees to a sqrt() */    

    cpl_test_zero( cpl_matrix_set_size(samppos, 1, VECTOR_SIZE) );
    cpl_test_zero( cpl_vector_set_size(fitvals, VECTOR_SIZE) );
    for (i=0; i < VECTOR_SIZE; i++) {
        cpl_test_zero( cpl_matrix_set(samppos, 0, i, (double)i + 1e4) );
        cpl_test_zero( cpl_vector_set(fitvals, i, sqrt((double)i)) );
    }

    eps = FLT_MAX;
    for (i = 0; i < VECTOR_SIZE-1; i++) {

        xmax = eps;
        error = cpl_polynomial_fit_cmp(poly1a, samppos, &symsamp, fitvals, NULL,
                                   CPL_TRUE, &zerodeg, &i);
        cpl_test_eq_error(error, CPL_ERROR_NONE);

        cpl_test_eq( cpl_polynomial_get_dimension(poly1a), 1 );
        cpl_test_eq( cpl_polynomial_get_degree(poly1a), i );

        eps = cpl_vector_get_mse(fitvals, poly1a, samppos, NULL);

        if (eps > xmax) {
            /* Should be able to fit at least a 8-degree polynomial
               with no degradation - and without error-margin
               (i686 can manage one degree more) */
            if (i < 9) cpl_test_leq( eps, xmax);
            cpl_msg_info(cpl_func, "1D-Polynomial with degree %"
                         CPL_SIZE_FORMAT " fit of a %d-point "
                         "dataset has a higher mean square error than a %"
                         CPL_SIZE_FORMAT "-degree fit: %g > %g",
                         i, VECTOR_SIZE, i-1, eps, xmax);
            break;
        }

    }

    /* Test 7A: Fit a 2nd degree polynomial to a parabola, using mindeg */

    cpl_vector_delete(fitvals);
    fitvals = cpl_vector_wrap(6, (double*)dsq6); /* Not modified */

    mindeg = 1;

    error = cpl_polynomial_fit_cmp(poly1a, samppos1, NULL, fitvals, NULL,
                               CPL_TRUE, &mindeg, &sqdeg);
    cpl_test_eq_error(error, CPL_ERROR_NONE);

    cpl_test_eq( cpl_polynomial_get_dimension(poly1a), 1 );
    cpl_test_eq( cpl_polynomial_get_degree(poly1a), sqdeg );

    cpl_test_abs( cpl_polynomial_get_coeff(poly1a, &zerodeg), 0.0, DBL_EPSILON);
    cpl_test_abs( cpl_polynomial_get_coeff(poly1a, &mindeg),  0.0,
                  6.0 * DBL_EPSILON);
    cpl_test_abs( cpl_polynomial_get_coeff(poly1a,  &sqdeg),  1.0, DBL_EPSILON);

    /* Test 7A: Fit a 2nd degree polynomial to a parabola, using mindeg */

    error = cpl_polynomial_fit_cmp(poly1a, samppos1, NULL, fitvals, NULL,
                               CPL_TRUE, &sqdeg, &sqdeg);
    cpl_test_eq_error(error, CPL_ERROR_NONE);

    cpl_test_eq( cpl_polynomial_get_dimension(poly1a), 1 );
    cpl_test_eq( cpl_polynomial_get_degree(poly1a), sqdeg );

    cpl_test_abs( cpl_polynomial_get_coeff(poly1a, &zerodeg), 0.0, DBL_EPSILON);
    cpl_test_abs( cpl_polynomial_get_coeff(poly1a, &mindeg),  0.0, DBL_EPSILON);
    cpl_test_abs( cpl_polynomial_get_coeff(poly1a,  &sqdeg),  1.0, DBL_EPSILON);


    cpl_polynomial_delete(poly1a);
    cpl_polynomial_delete(poly1b);
    cpl_polynomial_delete(poly2);
    (void)cpl_matrix_unwrap(samppos1);
    (void)cpl_vector_unwrap(taylor);
    (void)cpl_matrix_unwrap(samppos14);
    (void)cpl_vector_unwrap(taylor4);
    cpl_matrix_delete(samppos);
    cpl_vector_unwrap(fitvals);

    return;
}


/*----------------------------------------------------------------------------*/
/**
  @internal
  @brief      Benchmark the CPL function
  @param nr   The number of repeats
  @param nc   The number of coefficients
  @param nd   The max degree
  @return    void

 */
/*----------------------------------------------------------------------------*/
static void cpl_polynomial_fit_bench_2d(cpl_size nr, cpl_size nc, cpl_size nd)
{


    cpl_polynomial * poly2  = cpl_polynomial_new(2);
    cpl_vector     * vec    = cpl_vector_new(nc * nc);
    cpl_matrix     * xy_pos = cpl_matrix_new(2, nc * nc);
    cpl_flops        flops  = 0;
    double           secs   = 0.0;
    double           xmax   = DBL_MAX; /* This valuie should not be read */
    cpl_size         i, j, k, degree = 0;
    cpl_error_code   error = CPL_ERROR_NONE;



    for (i=0, k = 0; i < nc; i++) {
        for (j=0; j < nc; j++, k++) {
            const double x = (double) i * 0.5;
            const double y = (double) j * 2.0;

            error = cpl_matrix_set(xy_pos, 0, k, (double)i);
            cpl_test_eq_error(error, CPL_ERROR_NONE);

            error = cpl_matrix_set(xy_pos, 1, k, (double)j);
            cpl_test_eq_error(error, CPL_ERROR_NONE);

            error = cpl_vector_set(vec,   k, sqrt(x)*log(y+1.0));
            cpl_test_eq_error(error, CPL_ERROR_NONE);
        }
    }

    cpl_test_eq( k, nc * nc );

    for (i = 0; i < nr; i++) {
        for (degree = 0; degree < nd; degree++) {
            double mse;
            const cpl_flops flops0 = cpl_tools_get_flops();
            const double    secs0  = cpl_test_get_cputime();

            error = cpl_polynomial_fit_cmp(poly2, xy_pos, NULL, vec, NULL,
                                       CPL_FALSE, NULL, &degree);

            secs += cpl_test_get_cputime() - secs0;
            flops += cpl_tools_get_flops() - flops0;

            cpl_test_eq_error(error, CPL_ERROR_NONE);

            if (error) break;

            mse = cpl_vector_get_mse(vec, poly2, xy_pos, NULL);

            if (degree > 0 && mse > xmax) break;

            xmax = mse;

        }
        if (error) break;
    }

    cpl_vector_delete(vec);
    cpl_matrix_delete(xy_pos);
    cpl_polynomial_delete(poly2);

    cpl_msg_info("","Speed while fitting %" CPL_SIZE_FORMAT " 2D-points with "
                 "up to degree %" CPL_SIZE_FORMAT " in %g secs [Mflop/s]: %g",
                 nc*nc, degree, secs, (double)flops/secs/1e6);

}


/*----------------------------------------------------------------------------*/
/**
  @internal
  @brief      Get the mean squared error from a vector of residuals
  @param fitvals   The fitted values
  @param fit       The fitted polynomial
  @param samppos   The sampling points
  @param prechisq  If non-NULL, the reduced chi square
  @return  the mse, or negative on NULL input

 */
/*----------------------------------------------------------------------------*/
static double cpl_vector_get_mse(const cpl_vector     * fitvals,
                                 const cpl_polynomial * fit,
                                 const cpl_matrix     * samppos,
                                 double               * prechisq)
{

    const cpl_size np = cpl_vector_get_size(fitvals);
    cpl_vector   * residual;
    double         mse;
    cpl_flops      flops  = 0;
    double         secs   = 0.0;

    cpl_ensure(fitvals != NULL, CPL_ERROR_NULL_INPUT, -1.0);
    cpl_ensure(fit     != NULL, CPL_ERROR_NULL_INPUT, -2.0);
    cpl_ensure(samppos != NULL, CPL_ERROR_NULL_INPUT, -3.0);

    residual = cpl_vector_new(1+np/2); /* Just to test the resizing... */

    flops = cpl_tools_get_flops();
    secs  = cpl_test_get_cputime();

    cpl_vector_fill_polynomial_fit_residual(residual, fitvals, NULL, fit,
                                            samppos, prechisq);

    mse_secs += cpl_test_get_cputime() - secs;
    mse_flops += cpl_tools_get_flops() - flops;

    mse = cpl_vector_product(residual, residual) / (double)np;
    cpl_vector_delete(residual);

    return mse;
}


/*----------------------------------------------------------------------------*/
/**
  @internal
  @brief   Fit and compare to a higher dimension fit with a zero-degree
  @param  self    Polynomial of dimension d to hold the coefficients
  @param  samppos Matrix of p sample positions, with d rows and p columns
  @param  sampsym NULL, or d booleans, true iff the sampling is symmetric
  @param  fitvals Vector of the p values to fit
  @param  fitsigm Uncertainties of the sampled values, or NULL for all ones
  @param  dimdeg  True iff there is a fitting degree per dimension
  @param  mindeg  Pointer to 1 or d minimum fitting degree(s), or NULL
  @param  maxdeg  Pointer to 1 or d maximum fitting degree(s), at least mindeg
  @return CPL_ERROR_NONE on success, else the relevant #_cpl_error_code_
  @see cpl_polynomial_fit()

 */
/*----------------------------------------------------------------------------*/
cpl_error_code cpl_polynomial_fit_cmp(cpl_polynomial    * self,
                                      const cpl_matrix  * samppos,
                                      const cpl_boolean * sampsym,
                                      const cpl_vector  * fitvals,
                                      const cpl_vector  * fitsigm,
                                      cpl_boolean         dimdeg,
                                      const cpl_size    * mindeg,
                                      const cpl_size    * maxdeg)
{

    const cpl_error_code error = cpl_polynomial_fit(self, samppos, sampsym,
                                                    fitvals, fitsigm, dimdeg,
                                                    mindeg, maxdeg);

    cpl_test_error(error);

    if (self != NULL) {
        const cpl_size mdim = cpl_polynomial_get_dimension(self);
        const cpl_size ndim = 1 + mdim;
        cpl_polynomial * self1p = cpl_polynomial_new(ndim);
        cpl_size idim;

        /* This is quaranteed to fail */
        const cpl_error_code error1e = cpl_polynomial_fit(self1p, samppos,
                                                         sampsym, fitvals,
                                                         fitsigm, dimdeg,
                                                         mindeg, maxdeg);
        cpl_test_error(error1e);
        cpl_test(error1e == error || error1e == CPL_ERROR_INCOMPATIBLE_INPUT);

        if (samppos != NULL && (mdim == 1 || dimdeg == CPL_TRUE)) {
            cpl_polynomial * zeropol = cpl_polynomial_new(mdim);
            const cpl_size np = cpl_matrix_get_ncol(samppos);
            const cpl_size nc = cpl_matrix_get_nrow(samppos);
            cpl_matrix * samppos1p = cpl_matrix_new(1 + nc, np);

            cpl_boolean * sampsym1p = sampsym
                ? (cpl_boolean *)cpl_malloc((size_t)ndim * sizeof(*sampsym1p))
                : NULL;
            cpl_size * mindeg1p = mindeg
                ? (cpl_size*)cpl_malloc((size_t)ndim * sizeof(*mindeg1p))
                : NULL;
            cpl_size * maxdeg1p = maxdeg
                ? (cpl_size*)cpl_malloc((size_t)ndim * sizeof(*maxdeg1p))
                : NULL;
            cpl_error_code error1p;

            for (idim = 0; idim < ndim; idim++) {
                cpl_size i, j;

                /* First copy all rows to new matrix, leaving one with zeros */
                for (j = 0; j < idim; j++) {
                    if (j <= nc) {
                        for (i = 0; i < np; i++) {
                            cpl_matrix_set(samppos1p, j, i,
                                           cpl_matrix_get(samppos, j, i));
                        }
                    }
                    if (sampsym1p != NULL) sampsym1p[j] = sampsym[j];
                    if (mindeg1p  != NULL) mindeg1p [j] = mindeg [j];
                    if (maxdeg1p  != NULL) maxdeg1p [j] = maxdeg [j];
                }
                if (j <= nc) {
                    for (i = 0; i < np; i++) {
                        cpl_matrix_set(samppos1p, j, i, 0.0);
                    }
                }
                if (sampsym1p != NULL) sampsym1p[j] = CPL_TRUE;
                if (mindeg1p  != NULL) mindeg1p [j] = 0;
                if (maxdeg1p  != NULL) maxdeg1p [j] = 0;

                for (j++; j < ndim; j++) {
                    if (j <= nc) {
                        for (i = 0; i < np; i++) {
                            cpl_matrix_set(samppos1p, j, i,
                                           cpl_matrix_get(samppos, j-1, i));
                        }
                    }
                    if (sampsym1p != NULL) sampsym1p[j] = sampsym[j-1];
                    if (mindeg1p  != NULL) mindeg1p [j] = mindeg [j-1];
                    if (maxdeg1p  != NULL) maxdeg1p [j] = maxdeg [j-1];
                }

                error1p = cpl_polynomial_fit(self1p, samppos1p,
                                             sampsym1p, fitvals,
                                             fitsigm, CPL_TRUE,
                                             mindeg1p, maxdeg1p);
                if (ndim != 2) {
                    cpl_test_eq_error(error1p, CPL_ERROR_UNSUPPORTED_MODE);
                } else if (mindeg != NULL && mindeg[0] > 0) {
                    cpl_test_eq_error(error1p, CPL_ERROR_UNSUPPORTED_MODE);
                } else {
                    const cpl_size degree = cpl_polynomial_get_degree(self);
                    if (!error && error1p == CPL_ERROR_SINGULAR_MATRIX &&
                        1 + degree == np && np >= 20) {

                        cpl_test_eq_error(error1p, CPL_ERROR_SINGULAR_MATRIX);

                        /* In the non-overdetermined case, the multi-variate fit
                           is not as accurate as the uni-variate,
                           for degree 20 the difference is a failure */

                        cpl_msg_debug(cpl_func, "2D-fail(%d:%" CPL_SIZE_FORMAT
                                      ":%" CPL_SIZE_FORMAT ":%" CPL_SIZE_FORMAT
                                      "): S: %" CPL_SIZE_FORMAT "x %"
                                      CPL_SIZE_FORMAT, error, idim, maxdeg1p[0],
                                      maxdeg1p[1], 1 + nc, np);
                        if (cpl_msg_get_level() <= CPL_MSG_DEBUG)
                            cpl_matrix_dump(samppos1p, stdout);
                    } else if (!error1p && !error) {
                        const cpl_size i0 = 0;
                        const double k0 = cpl_polynomial_get_coeff(self, &i0);
                        cpl_polynomial * self0p
                            = cpl_polynomial_extract(self1p, idim, zeropol);

                        /* FIXME: Need relative polynomial comparison */
                        cpl_test_polynomial_abs(self, self0p, CX_MAX(fabs(k0),1)
                                                * pow(10.0, (double)degree)
                                                * DBL_EPSILON);

                        cpl_polynomial_delete(self0p);
                    } else {
                        cpl_test_eq_error(error1p, error);
                    }
                }

            }

            cpl_matrix_delete(samppos1p);
            cpl_polynomial_delete(zeropol);
            cpl_free(sampsym1p);
            cpl_free(mindeg1p);
            cpl_free(maxdeg1p);
        }

        cpl_polynomial_delete(self1p);
    }

    return cpl_error_set_(error); /* Tested and reset by caller */
}
