/* $Id: cpl_apertures_img-test.c,v 1.22 2010/11/11 09:23:18 llundin Exp $
 *
 * This file is part of the ESO Common Pipeline Library
 * Copyright (C) 2001-2008 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/*
 * $Author: llundin $
 * $Date: 2010/11/11 09:23:18 $
 * $Revision: 1.22 $
 * $Name: cpl-5_3_0-BRANCH $
 */

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/

#include <math.h>

#include <cpl_image_gen.h>
#include <cpl_image_io.h>
#include <cpl_stats.h>
#include <cpl_msg.h>
#include <cpl_mask.h>
#include <cpl_test.h>
#include <cpl_tools.h>
#include <cpl_memory.h>

#include "cpl_apertures_img.h"

/*-----------------------------------------------------------------------------
                                   Define
 -----------------------------------------------------------------------------*/

#ifndef IMAGESZ
#define IMAGESZ     512
#endif

/*-----------------------------------------------------------------------------
                                  Main
 -----------------------------------------------------------------------------*/
int main(void)
{
    cpl_image       *   imd ;
    cpl_image       *   small;
    double              median,
                        med_dist ;
    cpl_mask        *   bin ;
    cpl_image       *   labels ;
    double              threshold ;
    int                 nb_aperts ;
    cpl_apertures   *   aperts ;
    cpl_bivector    *   fwhms ;
    cpl_errorstate      prevstate;
    cpl_error_code      error;
    FILE              * stream;


    cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_WARNING);

    /* Insert tests below */

    stream = cpl_msg_get_level() > CPL_MSG_INFO
        ? fopen("/dev/null", "a") : stdout;

    cpl_test_nonnull( stream );

    /* Create test image */
    cpl_msg_info("", "Create a %dx%d test image", IMAGESZ, IMAGESZ) ;
    imd = cpl_image_fill_test_create(IMAGESZ, IMAGESZ) ;
    cpl_test_nonnull( imd );

    /* Threshold and labelize the image */
    cpl_msg_info("", "Threshold and labelize the test image") ;
    median = cpl_image_get_median_dev(imd, &med_dist) ;
    threshold = median + 2.0 * med_dist ; 
    bin = cpl_mask_threshold_image_create(imd, threshold, DBL_MAX) ;
    cpl_test_nonnull( bin );
    
    labels = cpl_image_labelise_mask_create(bin, &nb_aperts) ;
    cpl_test_nonnull( labels );
    cpl_mask_delete(bin) ;

    /* Compute statistics on the apertures */
    cpl_msg_info("", "Compute statistics on detected apertures") ;
    aperts = cpl_apertures_new_from_image(imd, labels);
    cpl_test_nonnull( aperts );
    cpl_image_delete(labels) ;

    cpl_apertures_dump(aperts, stream);

    /* Test the handling of a wrong image */
    small = cpl_image_new(1, 1, CPL_TYPE_FLOAT);
    cpl_test_nonnull( small );

    fwhms = cpl_apertures_get_fwhm(small, aperts);
    cpl_test_error(CPL_ERROR_ACCESS_OUT_OF_RANGE);
    cpl_test_null( fwhms );

    cpl_image_delete(small);

    /* The FWHM will fail on negative objects */
    error = cpl_image_threshold(imd, 1, DBL_MAX, 1, DBL_MAX);
    cpl_test_eq_error(error, CPL_ERROR_NONE);

    /* Test cpl_apertures_get_fwhm() */
    /* - including its ability to work with an existing error code */
    cpl_error_set(cpl_func, CPL_ERROR_UNSPECIFIED);
    prevstate = cpl_errorstate_get();
    fwhms = cpl_apertures_get_fwhm(imd, aperts);
    cpl_test(cpl_errorstate_is_equal(prevstate));
    cpl_test_error(CPL_ERROR_UNSPECIFIED);

    cpl_test_nonnull( fwhms );

    cpl_bivector_dump(fwhms, stream);
    cpl_bivector_delete(fwhms);

    /* Test the handling of an image with fewer FWHMs */
    error = cpl_image_subtract_scalar(imd, cpl_image_get_max(imd)/10.0);
    cpl_test_eq_error(error, CPL_ERROR_NONE);

    /* Otherwise the subsequent test will fail */
    cpl_test_lt(0.0, cpl_image_get_max(imd));

    cpl_error_set(cpl_func, CPL_ERROR_UNSPECIFIED);
    prevstate = cpl_errorstate_get();
    fwhms = cpl_apertures_get_fwhm(imd, aperts);
    cpl_test(cpl_errorstate_is_equal(prevstate));
    cpl_test_error(CPL_ERROR_UNSPECIFIED);

    cpl_test_nonnull( fwhms );

    cpl_bivector_dump(fwhms, stream);
    cpl_bivector_delete(fwhms);

    /* Test the handling of an image with no FWHMs */

    error = cpl_image_subtract_scalar(imd, 1.0 + cpl_image_get_max(imd));
    cpl_test_eq_error(error, CPL_ERROR_NONE);

    fwhms = cpl_apertures_get_fwhm(imd, aperts);
    cpl_test_error(CPL_ERROR_DATA_NOT_FOUND);

    cpl_test_null( fwhms );

    /* Test handling of NULL input */
    fwhms = cpl_apertures_get_fwhm(imd, NULL);
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    cpl_test_null( fwhms );

    /* Test handling of NULL input */
    fwhms = cpl_apertures_get_fwhm(NULL, aperts);
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    cpl_test_null( fwhms );
    
    cpl_apertures_delete(aperts) ;
    cpl_image_delete(imd) ;

    if (stream != stdout) cpl_test_zero( fclose(stream) );

    return cpl_test_end(0);
}
