/* $Id: cpl_apertures-test.c,v 1.32 2010/11/15 09:54:43 llundin Exp $
 *
 * This file is part of the ESO Common Pipeline Library
 * Copyright (C) 2001-2008 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/*
 * $Author: llundin $
 * $Date: 2010/11/15 09:54:43 $
 * $Revision: 1.32 $
 * $Name: cpl-5_3_0-BRANCH $
 */

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/

#include <math.h>

#include <cpl_image_gen.h>
#include <cpl_image_io.h>
#include <cpl_image_bpm.h>
#include <cpl_stats.h>
#include <cpl_msg.h>
#include <cpl_memory.h>
#include <cpl_mask.h>
#include <cpl_test.h>

#include "cpl_apertures.h"

/*-----------------------------------------------------------------------------
                                   Define
 -----------------------------------------------------------------------------*/

#define IMAGESZ     10
#define IMAGESZ2    512

#define CPL_APERTS_CMP_IMAGE(OP)                                               \
    do {                                                                       \
        cpl_test_abs( CPL_CONCAT(cpl_apertures_get, OP)(aperts, 1),            \
                      CPL_CONCAT2X(cpl_image_get, CPL_CONCAT(OP,window))       \
                      (imb, 1, 1, IMAGESZ, IMAGESZ),                           \
                       4.0 * DBL_EPSILON);                                     \
    } while (0)    



/*-----------------------------------------------------------------------------
                                  Main
 -----------------------------------------------------------------------------*/
int main(void)
{
    const cpl_type img_types[] = {CPL_TYPE_DOUBLE, CPL_TYPE_FLOAT, CPL_TYPE_INT};
    cpl_image     * imd;
    cpl_mask      * bpm;
    cpl_apertures * aperts;
    cpl_apertures * nullapert;
    double        * val;
    const double    psigmas[] = {5, 2, 1, 0.5};
    const double    maxval = 10.0;
    cpl_vector    * sigmas;
    unsigned        itype;
    FILE          * stream;
    cpl_error_code  error;
    int             pos;

    cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_WARNING);

    /* Insert tests below */

    stream = cpl_msg_get_level() > CPL_MSG_INFO
        ? fopen("/dev/null", "a") : stdout;

    /* Test 1: NULL pointers */
    cpl_apertures_delete(NULL); /* Just verify the return */
    cpl_apertures_dump(NULL, NULL); /* Just verify the return */
    cpl_apertures_dump(NULL, stream); /* Just verify the return */

    /* Statistics computation */
    nullapert = cpl_apertures_new_from_image(NULL, NULL);
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    cpl_test_null(nullapert);

    /* Accessor functions */
    cpl_test_eq(cpl_apertures_get_size(NULL), -1);
    cpl_test_error(CPL_ERROR_NULL_INPUT);

    cpl_test_zero(cpl_apertures_get_max_x(NULL, 1));
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    cpl_test_zero(cpl_apertures_get_max_y(NULL, 1));
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    cpl_test_zero(cpl_apertures_get_centroid_x(NULL, 1));
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    cpl_test_zero(cpl_apertures_get_centroid_y(NULL, 1));
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    cpl_test_eq(cpl_apertures_get_npix(NULL, 1), -1.0);
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    cpl_test_eq(cpl_apertures_get_left(NULL, 1), -1.0);
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    cpl_test_eq(cpl_apertures_get_left_y(NULL, 1), -1.0);
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    cpl_test_eq(cpl_apertures_get_right(NULL, 1), -1.0);
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    cpl_test_eq(cpl_apertures_get_right_y(NULL, 1), -1.0);
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    cpl_test_eq(cpl_apertures_get_top_x(NULL, 1), -1.0);
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    cpl_test_eq(cpl_apertures_get_top(NULL, 1), -1.0);
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    cpl_test_eq(cpl_apertures_get_bottom_x(NULL, 1), -1.0);
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    cpl_test_eq(cpl_apertures_get_bottom(NULL, 1), -1.0);
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    cpl_test_zero(cpl_apertures_get_max(NULL, 1));
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    cpl_test_zero(cpl_apertures_get_min(NULL, 1));
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    cpl_test_zero(cpl_apertures_get_mean(NULL, 1));
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    cpl_test_zero(cpl_apertures_get_median(NULL, 1));
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    cpl_test_zero(cpl_apertures_get_stdev(NULL, 1));
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    cpl_test_zero(cpl_apertures_get_flux(NULL, 1));
    cpl_test_error(CPL_ERROR_NULL_INPUT);

    /* Sorting functions */
    error = cpl_apertures_sort_by_npix(NULL);
    cpl_test_eq_error(error, CPL_ERROR_NULL_INPUT);

    error = cpl_apertures_sort_by_max(NULL);
    cpl_test_eq_error(error, CPL_ERROR_NULL_INPUT);

    error = cpl_apertures_sort_by_flux(NULL);
    cpl_test_eq_error(error, CPL_ERROR_NULL_INPUT);

    /* Detection functions */
    cpl_test_null(cpl_apertures_extract(NULL, NULL, NULL));
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    cpl_test_null(cpl_apertures_extract_window(NULL, NULL, 1, 1, 2, 2, NULL));
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    cpl_test_null(cpl_apertures_extract_sigma(NULL, 1.0));
    cpl_test_error(CPL_ERROR_NULL_INPUT);


    imd = cpl_image_new(IMAGESZ, IMAGESZ, CPL_TYPE_DOUBLE);

    val = cpl_image_get_data_double(imd);
    val[33] = val[34] = val[35] = val[36] = 1.0;
    val[43] = val[44] = val[45] = val[46] = 1.0;
    val[53] = val[54] = val[56] = 1.0;
    val[55] = maxval;
    val[63] = val[64] = val[65] = val[66] = 1.0;
    val[88] = 2.0 * maxval;

    bpm = cpl_mask_threshold_image_create(imd, 0.5, 0.5 + maxval);
    cpl_test_zero(cpl_mask_not(bpm));

    for (itype = 0; itype < sizeof(img_types)/sizeof(img_types[0]); itype++) {

        cpl_image * labels = cpl_image_new(IMAGESZ, IMAGESZ, CPL_TYPE_INT);
        cpl_image * img = cpl_image_cast(imd, img_types[itype]);
        cpl_image * imb = cpl_image_duplicate(img);
        cpl_mask  * bin;
        int         xpos, ypos;

        cpl_test_zero(cpl_image_reject_from_mask(imb, bpm));

        /* Create test image and the labels */
        cpl_msg_info(cpl_func, "Testing with a %d X %d-image (type %u)",
                     IMAGESZ, IMAGESZ, (unsigned)img_types[itype]);

        /* Test 3: with a 'bad' label image */
        cpl_test_nonnull(labels);

        aperts = cpl_apertures_new_from_image(img, labels);
        cpl_test_error(CPL_ERROR_ILLEGAL_INPUT);
        cpl_test_null(aperts);

        cpl_image_set(labels, 1, 1, 1.0);
        cpl_test_error(CPL_ERROR_NONE);

        aperts = cpl_apertures_new_from_image(img, labels);
        cpl_test_error(CPL_ERROR_NONE);
        cpl_test_nonnull(aperts);

        cpl_apertures_dump(aperts, stream);
        cpl_apertures_delete(aperts);
        cpl_image_set(labels, 1, 1, 2.0);
        cpl_test_error(CPL_ERROR_NONE);

        aperts = cpl_apertures_new_from_image(img, labels);
        cpl_test_error(CPL_ERROR_DATA_NOT_FOUND);
        cpl_test_null(aperts);

        cpl_image_delete(labels);

        /* Test 4: Thresholding */
        bin = cpl_mask_threshold_image_create(img, 0.5, DBL_MAX);
        cpl_test_nonnull(bin);
        labels = cpl_image_labelise_mask_create(bin, NULL);
        cpl_test_nonnull(labels);
        cpl_mask_delete(bin);

        /* Compute statistics on the apertures */
        cpl_msg_info("","Compute statistics on detected apertures");
        aperts = cpl_apertures_new_from_image(img, labels);
        cpl_test_error(CPL_ERROR_NONE);
        cpl_test_nonnull(aperts);

        cpl_test_zero(cpl_apertures_sort_by_npix(aperts));

        cpl_apertures_dump(aperts, stream);

        cpl_msg_info("","Compare statistics with those from same image with %d "
                     "rejected pixels", cpl_image_count_rejected(imb));

        cpl_test_eq(cpl_apertures_get_npix(aperts, 1),
                    IMAGESZ * IMAGESZ - cpl_image_count_rejected(imb));

        CPL_APERTS_CMP_IMAGE(min);
        CPL_APERTS_CMP_IMAGE(max);
        CPL_APERTS_CMP_IMAGE(mean);
        CPL_APERTS_CMP_IMAGE(median);
        CPL_APERTS_CMP_IMAGE(stdev);
        CPL_APERTS_CMP_IMAGE(flux);
        CPL_APERTS_CMP_IMAGE(centroid_x);
        CPL_APERTS_CMP_IMAGE(centroid_y);

        cpl_test_zero(cpl_image_get_maxpos(imb, &xpos, &ypos));

        /* FIXME: Update required after change of val[] */
        cpl_test_eq(cpl_apertures_get_left(aperts, 1), 4);
        cpl_test_eq(cpl_apertures_get_right(aperts, 1), 7);
        cpl_test_eq(cpl_apertures_get_left_y(aperts, 1), 4);
        cpl_test_eq(cpl_apertures_get_right_y(aperts, 1), 4);

        cpl_test_eq(cpl_apertures_get_bottom(aperts, 1), 4);
        cpl_test_eq(cpl_apertures_get_top(aperts, 1), 7);
        cpl_test_eq(cpl_apertures_get_bottom_x(aperts, 1), 4);
        cpl_test_eq(cpl_apertures_get_top_x(aperts, 1), 4);

        cpl_test_leq(cpl_apertures_get_left(aperts, 1),
                     cpl_apertures_get_max_x(aperts, 1));
        cpl_test_leq(cpl_apertures_get_max_x(aperts, 1),
                     cpl_apertures_get_right(aperts, 1));

        cpl_test_leq(cpl_apertures_get_bottom(aperts, 1),
                     cpl_apertures_get_max_y(aperts, 1));
        cpl_test_leq(cpl_apertures_get_max_y(aperts, 1),
                     cpl_apertures_get_top(aperts, 1));

        cpl_test_leq(cpl_apertures_get_left(aperts, 1), xpos);
        cpl_test_leq(xpos, cpl_apertures_get_right(aperts, 1));

        cpl_test_leq(cpl_apertures_get_bottom(aperts, 1), ypos);
        cpl_test_leq(ypos, cpl_apertures_get_top(aperts, 1));

        cpl_apertures_delete(aperts);
    
        /* Set a bad pixel and recompute the apertures */
        cpl_image_reject(img, 6, 6);
        cpl_test_error(CPL_ERROR_NONE);

        aperts = cpl_apertures_new_from_image(img, labels);
        cpl_test_error(CPL_ERROR_NONE);
        cpl_test_nonnull(aperts);

        cpl_apertures_dump(aperts, stream);
        cpl_image_delete(labels);

        /* Get the biggest object */
        cpl_msg_info("","Get the biggest aperture:");
        cpl_test_eq(cpl_apertures_sort_by_npix(aperts), CPL_ERROR_NONE);
        cpl_test_error(CPL_ERROR_NONE);
        cpl_msg_info("","number of pixels: %d",
                     cpl_apertures_get_npix(aperts, 1));

        /* Get the brightest object */
        cpl_msg_info("","Get the brightest aperture:");
        cpl_test_eq(cpl_apertures_sort_by_flux(aperts), CPL_ERROR_NONE);
        cpl_test_error(CPL_ERROR_NONE);
        cpl_msg_info("","flux: %g", cpl_apertures_get_flux(aperts, 1));

        /* Get the aperture with the highest peak */
        cpl_msg_info("","Get the aperture with the highest peak:");
        cpl_test_eq(cpl_apertures_sort_by_max(aperts), CPL_ERROR_NONE);
        cpl_test_error(CPL_ERROR_NONE);
        cpl_msg_info("","maximum value: %g", cpl_apertures_get_max(aperts, 1));
        cpl_apertures_dump(aperts, stream);
        cpl_apertures_delete(aperts);

        cpl_image_delete(imb);
        cpl_image_delete(img);

    }

    cpl_image_delete(imd);
 
    /* Create a new test image */
    imd = cpl_image_fill_test_create(IMAGESZ2, IMAGESZ2);
    
    sigmas = cpl_vector_wrap(4, (double*)psigmas);

    /* Test cpl_apertures_extract() */
    cpl_msg_info("","Test the apertures detection with a thresholding method");
    aperts = cpl_apertures_extract(imd, sigmas, &pos);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_nonnull(aperts);

    cpl_apertures_dump(aperts, stream);
    cpl_apertures_delete(aperts);

    /* Test cpl_apertures_extract_window() */
    cpl_msg_info("","Test the apertures detection (threshold) on a zone");
    aperts = cpl_apertures_extract_window(imd, sigmas, IMAGESZ2/4, IMAGESZ2/4,
                                          3*IMAGESZ2/4, 3*IMAGESZ2/4, &pos);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_nonnull(aperts);

    cpl_apertures_dump(aperts, stream);

    cpl_vector_unwrap(sigmas);
    cpl_apertures_delete(aperts);
    cpl_image_delete(imd);
    cpl_mask_delete(bpm);

    if (stream != stdout) cpl_test_zero( fclose(stream) );

    return cpl_test_end(0);
}
