/* -*- Mode: C; tab-width: 2; indent-tabs-mode: nil; c-basic-offset: 2 -*- */
/* vim:set sw=2 sts=2 et cin: */
/*
 * This file is part of the MUSE Instrument Pipeline
 * Copyright (C) 2007-2014 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */

#define _BSD_SOURCE /* get setenv() from stdlib.h */
#include <stdlib.h> /* setenv() */
#include <string.h>

#include <muse.h>

/* accuracy limits for xsc, ysc, xang, yang recovered from INM test data */
const double klimitsINM[][4] = {
  { 5.3e-8, 2.3e-8, 0.012, 0.020 }, /* for Gaussian */
  { 4.1e-8, 2.9e-8, 0.002, 0.009 }, /* for Moffat */
  { 4.8e-8, 2.4e-8, 0.001, 0.020 }  /* for Box */
};
const double kLimitPos = 0.05; /* [pix] object position detection accuracy */
/* accuracy limits for coordinate transformation tests */
const double kLimitDeg = DBL_EPSILON * 115; /* ~10 nano-arcsec for trafo to deg */
const double kLimitDegF = FLT_EPSILON * 13.51; /* ~5.8 milli-arcsec for trafo to  *
                                                * deg, with value stored in float */
const double kLimitPix = FLT_EPSILON * 205; /* ~1/40000th pixel for trafo to pix */
const double kLimitRot = FLT_EPSILON * 10; /* ~4.3 milli-arcsec for rotations */
const double kLimitSca = DBL_EPSILON * 7461; /* pixel scales to ~5.5 nano-arcsec */
const double kLimitPPl = FLT_EPSILON * 2.88; /* ~1.24 milli-arcsec in proj. plane */

/* test coordinates in the pixel plane */
const int kPx[][4] = {
  {   1,   1 }, /* increasing x position */
  {  50,   1 },
  { 100,   1 },
  { 150,   1 },
  { 200,   1 },
  { 250,   1 },
  { 300,   1 },
  {   1,  50 }, /* increasing y position */
  {   1, 100 },
  {   1, 150 },
  {   1, 200 },
  {   1, 250 },
  {   1, 300 },
  {  50,  50 }, /* diagonal */
  { 100, 100 },
  { 150, 150 },
  { 200, 200 },
  { 250, 250 },
  { 300, 300 },
  {  -1,  -1 }
};


/* check that detected objects are close to the expected position, *
 * assume that the objects are relatively isolated (no others      *
 * within the surrounding +/- 5 pix)                               */
void
muse_test_wcs_check_object_detection(cpl_table *aDetected, double aX, double aY)
{
  int i, nrow = cpl_table_get_nrow(aDetected);
  for (i = 0; i < nrow; i++) {
    double x = cpl_table_get_double(aDetected, "XPOS", i, NULL),
           y = cpl_table_get_double(aDetected, "YPOS", i, NULL),
           f = cpl_table_get_double(aDetected, "FLUX", i, NULL);
    if (fabs(aX - x) < 5 && fabs(aY - y) < 5) { /* right object */
      cpl_msg_debug(__func__, "expected at %f,%f, detected at %f,%f, deltas "
                    "are %e,%e, flux = %e", aX, aY, x, y, x - aX, y - aY, f);
      /* adapt check accuracy based on flux */
      double limit = kLimitPos;
      if (f < 500.) {
        limit *= 5.;
      } else if (f < 1000.) {
        limit *= 2.5;
      } else if (f < 10000.) {
        limit *= 2.4;
      } else if (f < 15000.) {
        limit *= 2.3;
      }
      cpl_test(fabs(aX - x) < limit && fabs(aY - y) < limit);
      return;
    }
  } /* for i (table rows) */
  /* this object was not found, fail the test! */
  cpl_test(CPL_ERROR_NONE == CPL_ERROR_DATA_NOT_FOUND);
} /* muse_test_wcs_check_object_detection() */

/* create typical MUSE header with TAN projection, *
 * similar to muse_wcs_create_default()            */
cpl_propertylist *
muse_test_wcs_create_typical_tan_header(void)
{
  cpl_propertylist *p = cpl_propertylist_new();
  cpl_propertylist_append_double(p, "CRPIX1", 156.);
  cpl_propertylist_append_double(p, "CRVAL1", 10.);
  cpl_propertylist_append_double(p, "RA", 10.);
  cpl_propertylist_append_double(p, "CD1_1", -0.2 / 3600.);
  cpl_propertylist_append_string(p, "CTYPE1", "RA---TAN");
  cpl_propertylist_append_string(p, "CUNIT1", "deg");
  cpl_propertylist_append_double(p, "CRPIX2", 151.);
  cpl_propertylist_append_double(p, "CRVAL2", -30.);
  cpl_propertylist_append_double(p, "DEC", -30.);
  cpl_propertylist_append_double(p, "CD2_2", 0.2 / 3600);
  cpl_propertylist_append_string(p, "CTYPE2", "DEC--TAN");
  cpl_propertylist_append_string(p, "CUNIT2", "deg");
  cpl_propertylist_append_double(p, "CD1_2", 0.);
  cpl_propertylist_append_double(p, "CD2_1", 0.);
  return p;
} /* muse_test_wcs_create_typical_tan_header() */

/* create typical MUSE header with linear scaling, *
 * similar to muse_resampling_cube                 */
cpl_propertylist *
muse_test_wcs_create_typical_lin_header(void)
{
  cpl_propertylist *p = cpl_propertylist_new();
  cpl_propertylist_append_double(p, "CRPIX1", 1.);
  cpl_propertylist_append_double(p, "CRVAL1", -150.);
  cpl_propertylist_append_string(p, "CTYPE1", "PIXEL");
  cpl_propertylist_append_string(p, "CUNIT1", "pixel");
  cpl_propertylist_append_double(p, "CD1_1", 1.);
  cpl_propertylist_append_double(p, "CRPIX2", 1.);
  cpl_propertylist_append_double(p, "CRVAL2", -150.);
  cpl_propertylist_append_string(p, "CTYPE2", "PIXEL");
  cpl_propertylist_append_string(p, "CUNIT2", "pixel");
  cpl_propertylist_append_double(p, "CD2_2", 1.);
  cpl_propertylist_append_double(p, "CD1_2", 0.);
  cpl_propertylist_append_double(p, "CD2_1", 0.);
  /* RA,DEC needed to not trigger error states */
  cpl_propertylist_append_double(p, "RA", 10.);
  cpl_propertylist_append_double(p, "DEC", -30.);
  return p;
} /* muse_test_wcs_create_typical_lin_header() */

/* create header for part of example 1 (Table 5) of   *
 * Greisen & Calabretta 2002 A&A 395, 1077 (Paper II) */
cpl_propertylist *
muse_test_wcs_create_example_1_header(void)
{
  cpl_propertylist *p = cpl_propertylist_new();
  /* leave out the velocity and stokes axes */
  cpl_propertylist_append_int(p, "NAXIS", 2);
  cpl_propertylist_append_int(p, "NAXIS1", 512);
  cpl_propertylist_append_int(p, "NAXIS2", 512);
  cpl_propertylist_append_double(p, "CRPIX1", 256.);
  /* use the CDi_j matrix instead of CDELT */
  cpl_propertylist_append_double(p, "CD1_1", -0.003);
  cpl_propertylist_append_string(p, "CTYPE1", "RA---TAN");
  cpl_propertylist_append_double(p, "CRVAL1", 45.83);
  cpl_propertylist_append_string(p, "CUNIT1", "deg");
  cpl_propertylist_append_double(p, "CRPIX2", 257.);
  cpl_propertylist_append_double(p, "CD2_2", 0.003);
  cpl_propertylist_append_string(p, "CTYPE2", "DEC--TAN");
  cpl_propertylist_append_double(p, "CRVAL2", 63.57);
  cpl_propertylist_append_string(p, "CUNIT2", "deg");
  /* no cross terms */
  cpl_propertylist_append_double(p, "CD1_2", 0.);
  cpl_propertylist_append_double(p, "CD2_1", 0.);
  /* XXX RA,DEC identical to the CRVALi for our functions */
  cpl_propertylist_append_double(p, "RA", 45.83);
  cpl_propertylist_append_double(p, "DEC", 63.57);
  return p;
} /* muse_test_wcs_create_example_1_header() */

/* check muse_wcs transformations against reference values for the same *
 * header using the CPL wrapper around wcslib; aFudge is an extra fudge *
 * factor so that one can be more generous for some tests               */
void
muse_test_wcs_check_celestial_transformation(cpl_propertylist *aWCS,
                                             const char *aComment,
                                             const double aFudge)
{
  cpl_wcs *cwcs = cpl_wcs_new_from_propertylist(aWCS);
  int i = -1;
  while (kPx[++i][0] > 0) {
    /* compute reference data using the CPL wrapper around wcslib */
    cpl_matrix *pxcoords = cpl_matrix_new(1, 2);
    cpl_matrix_set(pxcoords, 0, 0, kPx[i][0]);
    cpl_matrix_set(pxcoords, 0, 1, kPx[i][1]);
    cpl_matrix *skycoords = NULL;
    cpl_array *status = NULL;
    cpl_wcs_convert(cwcs, pxcoords, &skycoords, &status, CPL_WCS_PHYS2WORLD);
    cpl_array_delete(status);
    cpl_matrix_delete(pxcoords);
    double raref = cpl_matrix_get(skycoords, 0, 0),
           decref = cpl_matrix_get(skycoords, 0, 1);
    cpl_matrix_delete(skycoords);

    /* projection onto the sky */
    double ra, dec;
    cpl_test(muse_wcs_celestial_from_pixel(aWCS, kPx[i][0], kPx[i][1], &ra, &dec)
             == CPL_ERROR_NONE);
    cpl_test(fabs(ra - raref) < aFudge * kLimitDeg);
    cpl_test(fabs(dec - decref) < aFudge * kLimitDeg);
    cpl_msg_debug(__func__, "%s: ra/dec = %f,%f (%f,%f, %e,%e <? %e)", aComment,
                  ra, dec, raref, decref, ra - raref, dec - decref,
                  aFudge * kLimitDeg);

    /* projection into the pixel plane */
    double x, y;
    cpl_test(muse_wcs_pixel_from_celestial(aWCS, ra, dec, &x, &y)
             == CPL_ERROR_NONE);
    cpl_test(fabs(x - kPx[i][0]) < kLimitPix);
    cpl_test(fabs(y - kPx[i][1]) < kLimitPix);
    cpl_msg_debug(__func__, "%s: x,y = %f,%f (%d,%d, %e,%e <? %e)", aComment,
                  x, y, kPx[i][0], kPx[i][1], x - kPx[i][0], y - kPx[i][1],
                  kLimitPix);
  } /* while */
  cpl_wcs_delete(cwcs);
} /* muse_test_wcs_check_celestial_transformation() */

/*----------------------------------------------------------------------------*/
/**
  @brief    Test program to check that all the functions of the muse_wcs module
            work correctly.

  This program explicitely tests
    muse_wcs_object_new
    muse_wcs_object_delete
    muse_wcs_locate_sources
    muse_wcs_solve
    muse_wcs_create_default
    muse_wcs_project_tan
    muse_wcs_position_celestial
    muse_wcs_apply_cd
    muse_wcs_celestial_from_pixel
    muse_wcs_pixel_from_celestial
    muse_wcs_projplane_from_celestial
    muse_wcs_projplane_from_pixel
    muse_wcs_pixel_from_projplane
    muse_wcs_get_angles
    muse_wcs_get_scales
    muse_wcs_new
  It implicitely also tests
    muse_wcs_pixel_from_celestial_fast
    muse_wcs_celestial_from_pixel_fast
    muse_wcs_projplane_from_pixel_fast
    muse_wcs_pixel_from_projplane_fast
 */
/*----------------------------------------------------------------------------*/
int main(int argc, char **argv)
{
  UNUSED_ARGUMENTS(argc, argv);
  cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_DEBUG);

  /*********************************************
   * check WCS creation from input pixel table *
   *********************************************/
  muse_pixtable *ptin = muse_pixtable_load(BASEFILENAME"_pt_in.fits");
  cpl_table *wcsref = cpl_table_load(BASEFILENAME"_scene.fits", 1, 1);
  cpl_test(ptin && wcsref);
  cpl_error_code rc;
  /* loop through the possible WCS centroid types */
  muse_wcs_centroid_type type;
  const char *typestring[] = {
    "GAUSSIAN",
    "MOFFAT",
    "BOX"
  };
  muse_wcs_object *wcsobj = NULL;
  for (type = MUSE_WCS_CENTROID_GAUSSIAN; type <= MUSE_WCS_CENTROID_BOX; type++) {
    cpl_msg_debug(__func__, "\n____ centroid type %s _____", typestring[type]);
    wcsobj = muse_wcs_object_new();
    cpl_test_nonnull(wcsobj);
    /* for the last one, replace the normal DAR *
     * header with the one from dar_check       */
    if (type == MUSE_WCS_CENTROID_BOX) {
      cpl_propertylist_erase_regexp(ptin->header, MUSE_HDR_PT_DAR_NAME, 0);
      cpl_propertylist_append_float(ptin->header, MUSE_HDR_PT_DAR_CORR, 0.001);
    }
    rc = muse_wcs_locate_sources(ptin, 3., type, wcsobj);
    cpl_test(rc == CPL_ERROR_NONE);
    cpl_test_nonnull(wcsobj->detected);
#if 0
    char *fn = cpl_sprintf("muse_test_wcs_detected_%s.fits", typestring[type]);
    cpl_table_save(wcsobj->detected, NULL, NULL, fn, CPL_IO_CREATE);
    cpl_free(fn);
#endif
    cpl_test(cpl_table_get_nrow(wcsobj->detected) == 63);
    /* check position of selected isolated objects, reference *
     * positions measured using IRAF imexamine 'a'            */
    muse_test_wcs_check_object_detection(wcsobj->detected,  26.47,  41.08);
    muse_test_wcs_check_object_detection(wcsobj->detected,  62.98, 121.62);
    muse_test_wcs_check_object_detection(wcsobj->detected, 169.56,  33.6);
    muse_test_wcs_check_object_detection(wcsobj->detected, 117.26,  26.92);
    muse_test_wcs_check_object_detection(wcsobj->detected, 233.57,  41.53);
    muse_test_wcs_check_object_detection(wcsobj->detected, 211.7,  151.3);
    muse_test_wcs_check_object_detection(wcsobj->detected, 163.29, 152.43);
    muse_test_wcs_check_object_detection(wcsobj->detected,  90.38, 168.80);
    muse_test_wcs_check_object_detection(wcsobj->detected,  30.85, 253.93);
    muse_test_wcs_check_object_detection(wcsobj->detected, 187.42, 233.65);
    muse_test_wcs_check_object_detection(wcsobj->detected, 237.70, 270.73);
    muse_test_wcs_check_object_detection(wcsobj->detected, 226.48, 231.46);
    rc = muse_wcs_solve(wcsobj, wcsref, 5., 5., 2, 3.);
    cpl_msg_debug(__func__, "Solving WCS returned rc=%d: %s", rc,
                  rc != CPL_ERROR_NONE ? cpl_error_get_message() : "");
    cpl_test(rc == CPL_ERROR_NONE);
    cpl_test_nonnull(wcsobj->wcs);
    /* get output fit properties (using functions tested below...) */
    double xang, yang, xsc, ysc;
    muse_wcs_get_angles(wcsobj->wcs, &xang, &yang);
    muse_wcs_get_scales(wcsobj->wcs, &xsc, &ysc);
    cpl_msg_debug(__func__, "angles: %e,%e, scales: %e,%e", xang, yang, xsc, ysc);
    cpl_test_lt(fabs(xang), klimitsINM[type][2]); /* almost no rotation */
    cpl_test_lt(fabs(yang), klimitsINM[type][3]);
    /* should both be about 0.2''/px */
    cpl_test_abs(xsc, 0.2/3600., klimitsINM[type][0]);
    cpl_test_abs(ysc, 0.2/3600., klimitsINM[type][1]);
    if (type != MUSE_WCS_CENTROID_BOX) {
       muse_wcs_object_delete(wcsobj);
    }
  } /* for type */
  /* test some failure cases of muse_wcs_locate_sources() */
  cpl_errorstate state = cpl_errorstate_get();
  rc = muse_wcs_locate_sources(NULL, 3., MUSE_WCS_CENTROID_BOX, wcsobj);
  cpl_test(!cpl_errorstate_is_equal(state) && rc == CPL_ERROR_NULL_INPUT);
  cpl_errorstate_set(state);
  rc = muse_wcs_locate_sources(ptin, -5., MUSE_WCS_CENTROID_BOX, wcsobj);
  cpl_test(!cpl_errorstate_is_equal(state) && rc == CPL_ERROR_ILLEGAL_INPUT);
  cpl_errorstate_set(state);
  rc = muse_wcs_locate_sources(ptin, 0., MUSE_WCS_CENTROID_BOX, wcsobj);
  cpl_test(!cpl_errorstate_is_equal(state) && rc == CPL_ERROR_ILLEGAL_INPUT);
  cpl_errorstate_set(state);
  rc = muse_wcs_locate_sources(ptin, 3., -1, wcsobj);
  cpl_test(!cpl_errorstate_is_equal(state) && rc == CPL_ERROR_ILLEGAL_INPUT);
  cpl_errorstate_set(state);
  rc = muse_wcs_locate_sources(ptin, 3., MUSE_WCS_CENTROID_BOX + 1, wcsobj);
  cpl_test(!cpl_errorstate_is_equal(state) && rc == CPL_ERROR_ILLEGAL_INPUT);
  cpl_errorstate_set(state);
  rc = muse_wcs_locate_sources(ptin, 3., MUSE_WCS_CENTROID_BOX, NULL);
  cpl_test(!cpl_errorstate_is_equal(state) && rc == CPL_ERROR_NULL_INPUT);
  cpl_errorstate_set(state);
  muse_pixtable *pt2 = cpl_calloc(1, sizeof(muse_pixtable));
  pt2->table = cpl_table_duplicate(ptin->table);
  rc = muse_wcs_locate_sources(pt2, 3., MUSE_WCS_CENTROID_BOX, wcsobj); /* without header */
  cpl_test(!cpl_errorstate_is_equal(state) && rc == CPL_ERROR_NULL_INPUT);
  cpl_errorstate_set(state);
  pt2->header = cpl_propertylist_duplicate(ptin->header);
  /* with pixel table without apparent DAR correction headers */
  muse_datacube_delete(wcsobj->cube);
  cpl_table_delete(wcsobj->detected);
  cpl_propertylist_erase_regexp(ptin->header, "ESO DRS MUSE PIXTABLE DAR ", 0);
  rc = muse_wcs_locate_sources(ptin, 3., MUSE_WCS_CENTROID_MOFFAT, wcsobj);
  cpl_test(!cpl_errorstate_is_equal(state) &&
           cpl_error_get_code() == CPL_ERROR_UNSUPPORTED_MODE);
  cpl_errorstate_set(state);
  /* create pixel table with flat data, should fail object detection */
  cpl_table_fill_column_window_float(pt2->table, MUSE_PIXTABLE_DATA,
                                     0, cpl_table_get_nrow(pt2->table), 100);
  muse_datacube_delete(wcsobj->cube); /* don't want to leak the previous cube */
  rc = muse_wcs_locate_sources(pt2, 3., MUSE_WCS_CENTROID_BOX, wcsobj);
  cpl_test(!cpl_errorstate_is_equal(state) && rc == CPL_ERROR_DATA_NOT_FOUND);
  cpl_errorstate_set(state);
  /* test some failure cases of muse_wcs_solve() */
  rc = muse_wcs_solve(NULL, wcsref, 5., 5., 2, 3.);
  cpl_test(!cpl_errorstate_is_equal(state) && rc == CPL_ERROR_NULL_INPUT);
  cpl_errorstate_set(state);
  rc = muse_wcs_solve(wcsobj, NULL, 5., 5., 2, 3.);
  cpl_test(!cpl_errorstate_is_equal(state) && rc == CPL_ERROR_ILLEGAL_INPUT);
  cpl_errorstate_set(state);
  rc = muse_wcs_solve(wcsobj, wcsref, 0., 5., 2, 3.);
  cpl_test(!cpl_errorstate_is_equal(state) && rc == CPL_ERROR_ILLEGAL_INPUT);
  cpl_errorstate_set(state);
  rc = muse_wcs_solve(wcsobj, wcsref, 5., 0., 2, 3.);
  cpl_test(!cpl_errorstate_is_equal(state) && rc == CPL_ERROR_ILLEGAL_INPUT);
  cpl_errorstate_set(state);
  rc = muse_wcs_solve(wcsobj, wcsref, -2., 5., 2, 3.);
  cpl_test(!cpl_errorstate_is_equal(state) && rc == CPL_ERROR_ILLEGAL_INPUT);
  cpl_errorstate_set(state);
  rc = muse_wcs_solve(wcsobj, wcsref, 5., -2., 2, 3.);
  cpl_test(!cpl_errorstate_is_equal(state) && rc == CPL_ERROR_ILLEGAL_INPUT);
  cpl_errorstate_set(state);
  /* fake bad entries in the reference table to mess up object identification */
  cpl_table_fill_column_window(wcsref, "RA", 0, 100000, 10.);
  cpl_table_fill_column_window(wcsref, "DEC", 0, 100000, -10.);
  rc = muse_wcs_solve(wcsobj, wcsref, 5., 5., 2, 3.);
  cpl_test(!cpl_errorstate_is_equal(state) && rc == CPL_ERROR_DATA_NOT_FOUND);
  cpl_errorstate_set(state);
  cpl_table_erase_column(wcsref, "RA");
  rc = muse_wcs_solve(wcsobj, wcsref, 5., 5., 2, 3.);
  cpl_test(!cpl_errorstate_is_equal(state) && rc == CPL_ERROR_BAD_FILE_FORMAT);
  cpl_errorstate_set(state);
  /* don't know how to simulate failure of cpl_wcs_platesol()... */
  /* clean up this part */
  muse_pixtable_delete(pt2);
  muse_pixtable_delete(ptin);
  cpl_table_delete(wcsref);
  state = cpl_errorstate_get();
  muse_wcs_object_delete(wcsobj);
  cpl_test(cpl_errorstate_is_equal(state));
  muse_wcs_object_delete(NULL); /* should work with NULL, too */
  cpl_test(cpl_errorstate_is_equal(state));

  /*************************************************
   * create and delete a header with a default WCS *
   *************************************************/
  state = cpl_errorstate_get();
  cpl_propertylist *pwcs = muse_wcs_create_default();
  cpl_test(cpl_errorstate_is_equal(state));
  cpl_test(!strncmp("RA---TAN", cpl_propertylist_get_string(pwcs, "CTYPE1"), 9) &&
           !strncmp("DEC--TAN", cpl_propertylist_get_string(pwcs, "CTYPE2"), 9));
  cpl_propertylist_append_double(pwcs, "RA", 0.); /* muse_wcs_new() needs RA and DEC */
  cpl_propertylist_append_double(pwcs, "DEC", 0.);
  state = cpl_errorstate_get();
  muse_wcs *wcs = muse_wcs_new(pwcs);
  cpl_test(cpl_errorstate_is_equal(state));
  cpl_test(wcs->cd11 < 0. && wcs->cd22 > 0. && wcs->cddet != 0.);
  cpl_test(wcs->cd12 == 0. && wcs->cd21 == 0.);
  cpl_free(wcs);
  cpl_propertylist_delete(pwcs);

  /****************************************
   * test transformations of pixel tables *
   ****************************************/
  /* use values from WCS Paper II, example 1 as references */
  cpl_propertylist *p = muse_test_wcs_create_example_1_header();
  /* second WCS header, with wrong coordinate type */
  cpl_propertylist *p2 = cpl_propertylist_duplicate(p);
  cpl_propertylist_update_string(p2, "CTYPE1", "BLABLA");
  cpl_propertylist_update_string(p2, "CTYPE2", "BLABLA");
  /* create a pixel table with all the necessary headers/contents needed here */
  muse_pixtable *pt = cpl_calloc(1, sizeof(muse_pixtable));
  pt->table = muse_cpltable_new(muse_pixtable_def, 3);
  pt->header = cpl_propertylist_new();
  /* add some keywords to fake a position angle of zero */
  cpl_propertylist_append_double(pt->header, "ESO TEL PARANG START", 0.);
  cpl_propertylist_append_double(pt->header, "ESO TEL ALT", 75.);
  cpl_propertylist_append_double(pt->header, "ESO INS DROT START", 37.5);
  /* also add the position angle directly */
  cpl_propertylist_append_double(pt->header, "ESO INS DROT POSANG", 0.);
  cpl_propertylist_append_string(pt->header, "ESO INS DROT MODE", "SKY");
  /* set pixel positions from WCS Paper II, example 1 (Table 6) */
  cpl_table_set(pt->table, MUSE_PIXTABLE_XPOS, 0,   1); /* SE corner */
  cpl_table_set(pt->table, MUSE_PIXTABLE_YPOS, 0,   2);
  cpl_table_set(pt->table, MUSE_PIXTABLE_XPOS, 1,   1); /* NE corner */
  cpl_table_set(pt->table, MUSE_PIXTABLE_YPOS, 1, 512);
  cpl_table_set(pt->table, MUSE_PIXTABLE_XPOS, 2, 511); /* NW corner */
  cpl_table_set(pt->table, MUSE_PIXTABLE_YPOS, 2, 512);
  /* override with values, so that CRPIXi in our function *
   * ends up like the one expected by the example         */
  cpl_propertylist_append_float(pt->header, MUSE_HDR_PT_XLO, 0);
  cpl_propertylist_append_float(pt->header, MUSE_HDR_PT_XHI, 0);
  cpl_propertylist_append_float(pt->header, MUSE_HDR_PT_YLO, 0);
  cpl_propertylist_append_float(pt->header, MUSE_HDR_PT_YHI, 0);

  state = cpl_errorstate_get();
  cpl_test(muse_wcs_project_tan(pt, NULL) == CPL_ERROR_NULL_INPUT);
  cpl_test(muse_wcs_project_tan(NULL, p) == CPL_ERROR_NULL_INPUT);
  cpl_errorstate_set(state);
  state = cpl_errorstate_get();
  cpl_test(muse_wcs_project_tan(pt, p2) == CPL_ERROR_UNSUPPORTED_MODE);
  cpl_errorstate_set(state);

  cpl_test(muse_wcs_project_tan(pt, p) == CPL_ERROR_NONE);
  /* output units in radians */
  cpl_test(cpl_table_get_column_unit(pt->table, MUSE_PIXTABLE_XPOS) &&
           !strncmp(cpl_table_get_column_unit(pt->table, MUSE_PIXTABLE_XPOS), "rad", 4));
  cpl_test(cpl_table_get_column_unit(pt->table, MUSE_PIXTABLE_YPOS) &&
           !strncmp(cpl_table_get_column_unit(pt->table, MUSE_PIXTABLE_YPOS), "rad", 4));
  /* (xpos,ypos) should now be (phi,theta - pi/2) */
  float *xpos = cpl_table_get_data_float(pt->table, MUSE_PIXTABLE_XPOS),
        *ypos = cpl_table_get_data_float(pt->table, MUSE_PIXTABLE_YPOS);
  cpl_test_abs(xpos[0], 45.0 / CPL_MATH_DEG_RAD, FLT_EPSILON);
  cpl_test_abs(ypos[0] + CPL_MATH_PI_2, 88.918255 / CPL_MATH_DEG_RAD, FLT_EPSILON);
  cpl_test_abs(xpos[1], 135.0 / CPL_MATH_DEG_RAD, FLT_EPSILON);
  cpl_test_abs(ypos[1] + CPL_MATH_PI_2, 88.918255 / CPL_MATH_DEG_RAD, FLT_EPSILON);
  cpl_test_abs(xpos[2], 225.0 / CPL_MATH_DEG_RAD, FLT_EPSILON);
  cpl_test_abs(ypos[2] + CPL_MATH_PI_2, 88.918255 / CPL_MATH_DEG_RAD, FLT_EPSILON);
  cpl_msg_debug(__func__, "after project_tan: (0: %e,%e, 1: %e,%e, 2: %e,%e) <? %e",
                xpos[0] - 45.0 / CPL_MATH_DEG_RAD,
                ypos[0] + CPL_MATH_PI_2 - 88.918255 / CPL_MATH_DEG_RAD,
                xpos[1] - 135.0 / CPL_MATH_DEG_RAD,
                ypos[1] + CPL_MATH_PI_2 - 88.918255 / CPL_MATH_DEG_RAD,
                xpos[2] - 225.0 / CPL_MATH_DEG_RAD,
                ypos[2] + CPL_MATH_PI_2 - 88.918255 / CPL_MATH_DEG_RAD, FLT_EPSILON);
  /* pixel table now in radians, should fail! */
  state = cpl_errorstate_get();
  cpl_test(muse_wcs_project_tan(pt, p) == CPL_ERROR_INCOMPATIBLE_INPUT);
  cpl_errorstate_set(state);

  state = cpl_errorstate_get();
  cpl_test(muse_wcs_position_celestial(NULL, 45.83, 63.57) == CPL_ERROR_NULL_INPUT);
  cpl_errorstate_set(state);
  cpl_propertylist *pt_header_copy = cpl_propertylist_duplicate(pt->header);
  cpl_propertylist_update_string(pt->header, "CTYPE1", "BLABLA");
  cpl_propertylist_update_string(pt->header, "CTYPE2", "BLABLA");
  state = cpl_errorstate_get();
  cpl_test(muse_wcs_position_celestial(pt, 45.83, 63.57) == CPL_ERROR_UNSUPPORTED_MODE);
  cpl_errorstate_set(state);
  cpl_propertylist_delete(pt->header);
  pt->header = NULL;
  state = cpl_errorstate_get();
  cpl_test(muse_wcs_position_celestial(NULL, 45.83, 63.57) == CPL_ERROR_NULL_INPUT);
  cpl_errorstate_set(state);
  pt->header = pt_header_copy;
  cpl_test(muse_wcs_position_celestial(pt, 45.83, 63.57) == CPL_ERROR_NONE);
  /* output units in degrees */
  cpl_test(cpl_table_get_column_unit(pt->table, MUSE_PIXTABLE_XPOS) &&
           !strncmp(cpl_table_get_column_unit(pt->table, MUSE_PIXTABLE_XPOS), "deg", 4));
  cpl_test(cpl_table_get_column_unit(pt->table, MUSE_PIXTABLE_YPOS) &&
           !strncmp(cpl_table_get_column_unit(pt->table, MUSE_PIXTABLE_YPOS), "deg", 4));
  /* (xpos,ypos) should now be (ra,dec) */
  xpos = cpl_table_get_data_float(pt->table, MUSE_PIXTABLE_XPOS),
  ypos = cpl_table_get_data_float(pt->table, MUSE_PIXTABLE_YPOS);
  cpl_test_abs(xpos[0], 47.503264 - 45.83, kLimitDegF);
  cpl_test_abs(ypos[0], 62.795111 - 63.57, kLimitDegF);
  cpl_test_abs(xpos[1], 47.595581 - 45.83, kLimitDegF);
  cpl_test_abs(ypos[1], 64.324332 - 63.57, kLimitDegF);
  cpl_test_abs(xpos[2], 44.064419 - 45.83, kLimitDegF);
  cpl_test_abs(ypos[2], 64.324332 - 63.57, kLimitDegF);
  cpl_msg_debug(__func__, "after position_celestial: (0: %e,%e, 1: %e,%e, 2: %e,%e) <? %e",
                xpos[0] - (47.503264 - 45.83), ypos[0] - (62.795111 - 63.57),
                xpos[1] - (47.595581 - 45.83), ypos[1] - (64.324332 - 63.57),
                xpos[2] - (44.064419 - 45.83), ypos[2] - (64.324332 - 63.57), kLimitDegF);
  /* pixel table now in degrees, should fail! */
  state = cpl_errorstate_get();
  cpl_test(muse_wcs_position_celestial(pt, 45.83, 63.57) == CPL_ERROR_INCOMPATIBLE_INPUT);
  cpl_errorstate_set(state);

  muse_pixtable_delete(pt);
  cpl_propertylist_delete(p);
  cpl_propertylist_delete(p2);

  /***************************************************
   * test rotation of exposure by zeropoint rotation *
   ***************************************************/
  cpl_propertylist *inexp = cpl_propertylist_new(),
                   *incal = cpl_propertylist_new();
  /* start with no rotation */
  cpl_propertylist_append_string(inexp, "ESO INS DROT MODE", "SKY");
  cpl_propertylist_append_double(inexp, "ESO INS DROT POSANG", 0.);
  cpl_propertylist_append_double(incal, "CD1_1", -0.2/3600.);
  cpl_propertylist_append_double(incal, "CD1_2", 0.);
  cpl_propertylist_append_double(incal, "CD2_1", 0.);
  cpl_propertylist_append_double(incal, "CD2_2", 0.2/3600.);
  cpl_propertylist *cdout = muse_wcs_apply_cd(inexp, incal);
  double xang, yang, xsc, ysc;
  muse_wcs_get_scales(cdout, &xsc, &ysc);
  muse_wcs_get_angles(cdout, &xang, &yang);
  cpl_test_abs(xsc, 0.2/3600., DBL_EPSILON);
  cpl_test_abs(ysc, 0.2/3600., DBL_EPSILON);
  cpl_test_abs(xang, 0., DBL_EPSILON);
  cpl_test_abs(yang, 0., DBL_EPSILON);
  cpl_propertylist_delete(cdout);
  /* standard rotation tested as working fine during Comm1:                       *
   * WCS solution: scales 0.19866269991584470378 / 0.19892623242553914009 arcsec, *
   *   angles 0.57924428478310829860 / -0.27738138025024355882 deg                *
   * Updated CD matrix (-59.99999999999999289457 deg field rotation):             *
   *   -2.736310e-05 4.757814e-05 4.792937e-05 2.811352e-05                       *
   *   (scales 0.19758783554940947957 / 0.20003805495272852788 arcsec,            *
   *   angles -60.09596472295560687371 / -59.60576088395551863641 deg)            */
  cpl_propertylist_append_string(inexp, "DATE-OBS", "2014-02-22T03:57:01.454");
  cpl_propertylist_update_double(inexp, "ESO INS DROT POSANG", 60.);
  cpl_propertylist_update_double(incal, "CD1_1", -5.51812632497E-05);
  cpl_propertylist_update_double(incal, "CD1_2", -5.57886124193E-07);
  cpl_propertylist_update_double(incal, "CD2_1", 2.67511546844E-07);
  cpl_propertylist_update_double(incal, "CD2_2", 5.52566392427E-05);
  cdout = muse_wcs_apply_cd(inexp, incal);
  muse_wcs_get_scales(cdout, &xsc, &ysc);
  muse_wcs_get_angles(cdout, &xang, &yang);
  cpl_test_abs(xsc, 0.19758783554940947957 / 3600., DBL_EPSILON);
  cpl_test_abs(ysc, 0.20003805495272852788 / 3600., DBL_EPSILON);
  cpl_test_abs(xang, -60.09596472295560687371, DBL_EPSILON);
  cpl_test_abs(yang, -59.60576088395551863641, DBL_EPSILON);
  cpl_propertylist_delete(cdout);
  cpl_propertylist_erase(inexp, "DATE-OBS");
  /* 30 deg field rotation, no calibration rotation */
  cpl_propertylist_update_double(inexp, "ESO INS DROT POSANG", -30.);
  cpl_propertylist_update_double(incal, "CD1_1", -0.2/3600.);
  cpl_propertylist_update_double(incal, "CD1_2", 0.);
  cpl_propertylist_update_double(incal, "CD2_1", 0.);
  cpl_propertylist_update_double(incal, "CD2_2", 0.2/3600.);
  cdout = muse_wcs_apply_cd(inexp, incal);
  muse_wcs_get_scales(cdout, &xsc, &ysc);
  muse_wcs_get_angles(cdout, &xang, &yang);
  cpl_test_abs(xsc, 0.2/3600., DBL_EPSILON);
  cpl_test_abs(ysc, 0.2/3600., DBL_EPSILON);
  cpl_test_abs(xang, 30., 20. * DBL_EPSILON);
  cpl_test_abs(yang, 30., 20. * DBL_EPSILON);
  cpl_propertylist_delete(cdout);
  /* 35 deg field rotation, 5. deg calibration rotation */
  cpl_propertylist_update_double(inexp, "ESO INS DROT POSANG", -35.);
  cpl_propertylist_update_double(incal, "CD1_1", 0.2/3600. * cos(5. * CPL_MATH_RAD_DEG));
  cpl_propertylist_update_double(incal, "CD1_2", 0.2/3600. * sin(5. * CPL_MATH_RAD_DEG));
  cpl_propertylist_update_double(incal, "CD2_1", -0.2/3600. * sin(5. * CPL_MATH_RAD_DEG));
  cpl_propertylist_update_double(incal, "CD2_2", 0.2/3600. * cos(5. * CPL_MATH_RAD_DEG));
  cdout = muse_wcs_apply_cd(inexp, incal);
  muse_wcs_get_scales(cdout, &xsc, &ysc);
  muse_wcs_get_angles(cdout, &xang, &yang);
  cpl_test_abs(xsc, 0.2/3600., DBL_EPSILON);
  cpl_test_abs(ysc, 0.2/3600., DBL_EPSILON);
  cpl_test_abs(xang, -40., DBL_EPSILON);
  cpl_test_abs(yang, -40., DBL_EPSILON);
  cpl_propertylist_delete(cdout);
  /* -10. deg field rotation, -3./2. deg calibration rotation */
  cpl_propertylist_update_double(inexp, "ESO INS DROT POSANG", -10.);
  cpl_propertylist_update_double(incal, "CD1_1", -0.2/3600. * cos(-3. * CPL_MATH_RAD_DEG));
  cpl_propertylist_update_double(incal, "CD1_2", -0.2/3600. * sin(-3. * CPL_MATH_RAD_DEG));
  cpl_propertylist_update_double(incal, "CD2_1", 0.2/3600. * sin(-2. * CPL_MATH_RAD_DEG));
  cpl_propertylist_update_double(incal, "CD2_2", 0.2/3600. * cos(-2. * CPL_MATH_RAD_DEG));
  cdout = muse_wcs_apply_cd(inexp, incal);
  muse_wcs_get_scales(cdout, &xsc, &ysc);
  muse_wcs_get_angles(cdout, &xang, &yang);
  cpl_test_abs(xsc, 0.19789511778677520981 / 3600., DBL_EPSILON);
  cpl_test_abs(ysc, 0.20359239586930183430 / 3600., DBL_EPSILON);
  cpl_test_abs(xang, 7.10199686161382981453, 10 * DBL_EPSILON);
  cpl_test_abs(yang, 11.83086044790917945591, DBL_EPSILON);
  cpl_propertylist_delete(cdout);
  /* failures */
  state = cpl_errorstate_get();
  cdout = muse_wcs_apply_cd(NULL, incal);
  cpl_test(!cpl_errorstate_is_equal(state) &&
           cpl_error_get_code() == CPL_ERROR_NULL_INPUT);
  cpl_errorstate_set(state);
  cpl_test_null(cdout);
  cdout = muse_wcs_apply_cd(inexp, NULL);
  cpl_test(!cpl_errorstate_is_equal(state) &&
           cpl_error_get_code() == CPL_ERROR_NULL_INPUT);
  cpl_errorstate_set(state);
  cpl_test_null(cdout);
  cpl_propertylist_delete(inexp);
  cpl_propertylist_delete(incal);

  /***************************************************************************
   * test transformations between pixels and spherical celestial coordinates *
   ***************************************************************************/
  /* first for a non-rotated WCS with standard MUSE scaling */
  p = muse_test_wcs_create_typical_tan_header();
  cpl_propertylist_append_int(p, "NAXIS", 2); /* needed for cpl_wcs_*() */
  cpl_propertylist_append_int(p, "NAXIS1", 300);
  cpl_propertylist_append_int(p, "NAXIS2", 300);
  muse_test_wcs_check_celestial_transformation(p, "standard MUSE", 1.);

  /* now for a 25 deg rotated WCS with unequal scaling (0.25'' in x, 0.15'' in y) */
  cpl_propertylist_update_double(p, "CD1_1", -6.2938038673665E-5);
  cpl_propertylist_update_double(p, "CD2_2", 3.77628232041998E-5);
  cpl_propertylist_update_double(p, "CD1_2", -2.9348490966691E-5);
  cpl_propertylist_update_double(p, "CD2_1", -1.7609094580015E-5);
  muse_test_wcs_check_celestial_transformation(p, "25 deg rotated", 1.);

  /* same thing again, now closer to the south pole */
  cpl_propertylist_update_double(p, "CRVAL1", 300.);
  cpl_propertylist_update_double(p, "CRVAL2", -87.5);
  cpl_propertylist_update_double(p, "RA", 300.);
  cpl_propertylist_update_double(p, "DEC", -87.5);
  /* more generous comparison in this case */
  muse_test_wcs_check_celestial_transformation(p, "near pole", 11.2);

  /* different region of the sky */
  cpl_propertylist_update_double(p, "CRVAL1", 111.);
  cpl_propertylist_update_double(p, "CRVAL2", +25.5);
  cpl_propertylist_update_double(p, "RA", 111.);
  cpl_propertylist_update_double(p, "DEC", +25.5);
  /* more generous comparison in this case */
  muse_test_wcs_check_celestial_transformation(p, "northern", 1.);

  /* check failure cases */
  double v1, v2;
  state = cpl_errorstate_get();
  cpl_test(muse_wcs_celestial_from_pixel(NULL, 1., 1., &v1, &v2)
           == CPL_ERROR_NULL_INPUT);
  cpl_test(muse_wcs_celestial_from_pixel(p, 1., 1., NULL, &v2)
           == CPL_ERROR_NULL_INPUT);
  cpl_test(muse_wcs_celestial_from_pixel(p, 1., 1., &v1, NULL)
           == CPL_ERROR_NULL_INPUT);
  cpl_errorstate_set(state);
  cpl_test(muse_wcs_pixel_from_celestial(NULL, 10., 10., &v1, &v2)
           == CPL_ERROR_NULL_INPUT);
  cpl_test(muse_wcs_pixel_from_celestial(p, 10., 10., NULL, &v2)
           == CPL_ERROR_NULL_INPUT);
  cpl_test(muse_wcs_pixel_from_celestial(p, 10., 10., &v1, NULL)
           == CPL_ERROR_NULL_INPUT);
  cpl_errorstate_set(state);
  cpl_propertylist_update_string(p, "CTYPE1", "RA_not_TAN");
  cpl_propertylist_update_string(p, "CTYPE2", "DEC_not_TAN");
  cpl_test(muse_wcs_celestial_from_pixel(p, 1., 1., &v1, &v2)
           == CPL_ERROR_UNSUPPORTED_MODE);
  cpl_test(muse_wcs_pixel_from_celestial(p, 10., 10., &v1, &v2)
           == CPL_ERROR_UNSUPPORTED_MODE);
  cpl_errorstate_set(state);
  cpl_propertylist_delete(p);

  /***************************************************************************
   * test transformations between projection plane and celestial coordinates *
   ***************************************************************************/
  /* use values from WCS Paper II, example 1 as references */
  p = muse_test_wcs_create_example_1_header();
  double x,y;
  cpl_test(muse_wcs_projplane_from_celestial(p, 47.503264, 62.795111, &x, &y)
           == CPL_ERROR_NONE);
  cpl_test(fabs(x - 0.765000) < kLimitPPl && fabs(y - (-0.765000)) < kLimitPPl);
  cpl_msg_debug(__func__, "SE corner: %f,%f (%e,%e <? %e", x, y, x - 0.765000,
                y - (-0.765000), kLimitPPl);
  cpl_test(muse_wcs_projplane_from_celestial(p, 47.595581, 64.324332, &x, &y)
           == CPL_ERROR_NONE);
  cpl_test(fabs(x - 0.765000) < kLimitPPl && fabs(y - 0.765000) < kLimitPPl);
  cpl_msg_debug(__func__, "NE corner: %f,%f (%e,%e <? %e", x, y, x - 0.765000,
                y - 0.765000, kLimitPPl);
  cpl_test(muse_wcs_projplane_from_celestial(p, 44.064419, 64.324332, &x, &y)
           == CPL_ERROR_NONE);
  cpl_test(fabs(x - (-0.765000)) < kLimitPPl && fabs(y - 0.765000) < kLimitPPl);
  cpl_msg_debug(__func__, "NW corner: %f,%f (%e,%e <? %e", x, y, x - (-0.765000),
                y - 0.765000, kLimitPPl);
  state = cpl_errorstate_get();
  cpl_test(muse_wcs_projplane_from_celestial(NULL, 1., 1., &v1, &v2)
           == CPL_ERROR_NULL_INPUT);
  cpl_test(muse_wcs_projplane_from_celestial(p, 1., 1., NULL, &v2)
           == CPL_ERROR_NULL_INPUT);
  cpl_test(muse_wcs_projplane_from_celestial(p, 1., 1., &v1, NULL)
           == CPL_ERROR_NULL_INPUT);
  cpl_errorstate_set(state);
  cpl_propertylist_update_string(p, "CTYPE1", "RA_not_TAN");
  cpl_propertylist_update_string(p, "CTYPE2", "DEC_not_TAN");
  cpl_test(muse_wcs_projplane_from_celestial(p, 1., 1., &v1, &v2)
           == CPL_ERROR_UNSUPPORTED_MODE);
  cpl_errorstate_set(state);
  cpl_propertylist_delete(p);

  /************************************************************************
   * test transformations between pixels and projection plane coordinates *
   ************************************************************************/
  /* test a non-rotated WCS with standard MUSE scaling */
  p = muse_test_wcs_create_typical_lin_header();
  int i = -1;
  while (kPx[++i][0] > 0) {
    double xpx, ypx,
           xexpect = (kPx[i][0] - 1.) * 1. - 150.,
           yexpect = (kPx[i][1] - 1.) * 1. - 150.;

    /* to projection plane */
    cpl_test(muse_wcs_projplane_from_pixel(p, kPx[i][0], kPx[i][1], &x, &y)
             == CPL_ERROR_NONE);
    cpl_test(fabs(x - xexpect) < kLimitPix);
    cpl_test(fabs(y - yexpect) < kLimitPix);

    /* back to pixel plane */
    cpl_test(muse_wcs_pixel_from_projplane(p, x, y, &xpx, &ypx)
             == CPL_ERROR_NONE);
    cpl_test(fabs(xpx - kPx[i][0]) < kLimitPix);
    cpl_test(fabs(ypx - kPx[i][1]) < kLimitPix);
    cpl_msg_debug(__func__, "%s: x,y = %f,%f (%d,%d, %e,%e <? %e)", "standard MUSE",
                  x, y, kPx[i][0], kPx[i][1], x - kPx[i][0], y - kPx[i][1],
                  kLimitPix);
  } /* while */
  state = cpl_errorstate_get();
  cpl_test(muse_wcs_projplane_from_pixel(NULL, 1., 1., &v1, &v2)
           == CPL_ERROR_NULL_INPUT);
  cpl_test(muse_wcs_projplane_from_pixel(p, 1., 1., NULL, &v2)
           == CPL_ERROR_NULL_INPUT);
  cpl_test(muse_wcs_projplane_from_pixel(p, 1., 1., &v1, NULL)
           == CPL_ERROR_NULL_INPUT);
  cpl_errorstate_set(state);
  cpl_test(muse_wcs_pixel_from_projplane(NULL, 150., 150., &v1, &v2)
           == CPL_ERROR_NULL_INPUT);
  cpl_test(muse_wcs_pixel_from_projplane(p, 150., 150., NULL, &v2)
           == CPL_ERROR_NULL_INPUT);
  cpl_test(muse_wcs_pixel_from_projplane(p, 150., 150., &v1, NULL)
           == CPL_ERROR_NULL_INPUT);
  cpl_errorstate_set(state);
  cpl_propertylist_delete(p);

  /*************************
   * test muse_wcs getters *
   *************************/
  /* first for a non-rotated WCS with standard MUSE scaling */
  p = muse_test_wcs_create_typical_tan_header();
  cpl_test(muse_wcs_get_angles(p, &xang, &yang) == CPL_ERROR_NONE);
  cpl_test(xang == 0. && yang == 0.);
  cpl_test(muse_wcs_get_scales(p, &xsc, &ysc) == CPL_ERROR_NONE);
  cpl_test(xsc == 0.2 / 3600. && ysc == 0.2 / 3600.);

  /* now for a 25 deg rotated WCS with unequal scaling (0.25'' in x, 0.15'' in y) */
  cpl_propertylist_update_double(p, "CD1_1", -6.2938038673665E-5);
  cpl_propertylist_update_double(p, "CD2_2", 3.77628232041998E-5);
  cpl_propertylist_update_double(p, "CD1_2", -2.9348490966691E-5);
  cpl_propertylist_update_double(p, "CD2_1", -1.7609094580015E-5);
  cpl_test(muse_wcs_get_angles(p, &xang, &yang) == CPL_ERROR_NONE);
  cpl_test(fabs(xang - 25.) < kLimitRot && fabs(yang - 25.) < kLimitRot);
  cpl_test(muse_wcs_get_scales(p, &xsc, &ysc) == CPL_ERROR_NONE);
  cpl_test(fabs(xsc - 0.25 / 3600.) < kLimitSca &&
           fabs(ysc - 0.15 / 3600.) < kLimitSca);
  cpl_msg_debug(__func__, "angles: %e,%e (%e,%e <? %e), scales: %e,%e (%e,%e <? %e)",
                xang, yang, xang - 25., yang - 25., kLimitRot,
                xsc, ysc, xsc - 0.25 / 3600., ysc - 0.15 / 3600., kLimitSca);

  state = cpl_errorstate_get();
  cpl_test(muse_wcs_get_angles(NULL, &xang, &yang) == CPL_ERROR_NULL_INPUT);
  cpl_test(muse_wcs_get_scales(NULL, &xsc, &ysc) == CPL_ERROR_NULL_INPUT);
  cpl_test(muse_wcs_get_angles(p, NULL, &yang) == CPL_ERROR_NULL_INPUT);
  cpl_test(muse_wcs_get_scales(p, NULL, &ysc) == CPL_ERROR_NULL_INPUT);
  cpl_test(muse_wcs_get_angles(p, &xang, NULL) == CPL_ERROR_NULL_INPUT);
  cpl_test(muse_wcs_get_scales(p, &xsc, NULL) == CPL_ERROR_NULL_INPUT);
  cpl_errorstate_set(state);
  cpl_propertylist_erase_regexp(p, "CD1_?", 0);
  cpl_test(muse_wcs_get_angles(p, &xang, &yang) == CPL_ERROR_DATA_NOT_FOUND);
  cpl_test(muse_wcs_get_scales(p, &xsc, &ysc) == CPL_ERROR_DATA_NOT_FOUND);
  cpl_errorstate_set(state);
  cpl_propertylist_delete(p);

  /**************************
   * test muse_wcs creation *
   **************************/
  /* create and delete a wcs without a header */
  state = cpl_errorstate_get();
  wcs = muse_wcs_new(NULL);
  cpl_test(cpl_errorstate_is_equal(state));
  cpl_test(wcs->cd11 == 1. && wcs->cd22 == 1. && wcs->cddet == 1.);
  cpl_test(wcs->cd12 == 0. && wcs->cd21 == 0. && wcs->crpix1 == 0.
           && wcs->crpix2 == 0. && wcs->crval1 == 0. && wcs->crval2 == 0.);
  cpl_test(wcs->iscelsph == CPL_FALSE);
  cpl_free(wcs);

  /* create and delete a wcs without a header */
  p = muse_test_wcs_create_typical_tan_header();
  state = cpl_errorstate_get();
  wcs = muse_wcs_new(p);
  cpl_test(cpl_errorstate_is_equal(state));
  cpl_test(wcs->cd11 == -0.2 / 3600. && wcs->cd22 == 0.2 / 3600. &&
           wcs->cddet == -pow(0.2 / 3600., 2));
  cpl_test(wcs->cd12 == 0. && wcs->cd21 == 0.);
  cpl_test(wcs->crval1 == 10. && wcs->crpix1 == 156.);
  cpl_test(wcs->crval2 == -30. && wcs->crpix2 == 151.);
  cpl_test(wcs->iscelsph == CPL_FALSE);
  cpl_free(wcs);
  /* should work with debugging on, too */
  cpl_test(setenv("MUSE_DEBUG_WCS", "1", 1) == 0);
  state = cpl_errorstate_get();
  wcs = muse_wcs_new(p);
  cpl_free(wcs);
  cpl_test(cpl_errorstate_is_equal(state));
  cpl_test(unsetenv("MUSE_DEBUG_WCS") == 0);
  cpl_propertylist_delete(p);

  /* create and delete a wcs with a bad header */
  p = muse_test_wcs_create_typical_tan_header();
  cpl_propertylist_update_double(p, "CD1_1", 0.);
  cpl_propertylist_update_double(p, "CD2_2", 0.);
  state = cpl_errorstate_get();
  wcs = muse_wcs_new(p);
  cpl_test(!cpl_errorstate_is_equal(state) &&
           cpl_error_get_code() == CPL_ERROR_ILLEGAL_INPUT);
  cpl_errorstate_set(state);
  cpl_free(wcs);
  cpl_propertylist_delete(p);

  return cpl_test_end(0);
}
