/*
 * cpcieject.c
 *
 * System shutdown driver for DTI cPCI special.
 *
 * Author: MontaVista Software, Inc.
 *         jpeters@mvista.com
 *         source@mvista.com
 *
 *  Copyright (C) 2001 MontaVista Software Inc.
 *  Copyright (C) 2002 Hewlett-Packard Co. (Matthew Wilcox, Dann Frazier)
 *
 *  Changelog:
 *    - 2002-03-14, changed name from dti_sdd to cpcieject
 *    - 2002-02-25, now builds as a module
 *
 *  This program is free software; you can redistribute it and/or modify it
 *  under the terms of the GNU General Public License as published by the
 *  Free Software Foundation; either version 2 of the License, or (at your
 *  option) any later version.
 *
 *  THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 *  INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY 
 *  AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL 
 *  THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, 
 *  EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
 *  PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR 
 *  PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF 
 *  LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING 
 *  NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS 
 *  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/errno.h>
#include <linux/fs.h>
#include <linux/types.h>
#include <linux/version.h>
#include <linux/init.h>
#include <linux/slab.h>
#include <linux/ioctl.h>
#include <linux/wait.h>
#include <linux/pci.h>
#include <linux/reboot.h>
#include <asm/io.h>
#include <asm/uaccess.h>

int		cpciejectInit(void);
static void	cpciejectIntr(int, void *, struct pt_regs *);
static int	cpciejectOpen(struct inode *, struct file *);
static int	cpciejectClose(struct inode *, struct file *);
static int	cpciejectFasync(int, struct file *, int);
static int      cpciejectBlueLight(struct notifier_block *self, unsigned long, void *);

static struct notifier_block nb = {
	cpciejectBlueLight,
	NULL,
        INT_MIN,
};

static struct file_operations cpciejectFops = {
	open:		cpciejectOpen,
	release:	cpciejectClose,
	fasync:		cpciejectFasync,
};

#define GPSTS	0xd
#define GPEN	0xf
#define PMCNTRL	0x4
#define GPIREG	0x32
#define GPOREG	0x34

MODULE_DESCRIPTION("Compact PCI Eject Detection Driver");
MODULE_LICENSE("GPL");

static int UseCount;
static unsigned int PmBase;
static int AsyncPid;
static struct fasync_struct *FasyncPtr;

int
cpciejectInit(void)
{
	unsigned char Bus, DevFn;

	if (pcibios_find_device(PCI_VENDOR_ID_INTEL,
				PCI_DEVICE_ID_INTEL_82371AB_3,
				0, &Bus, &DevFn)) {
		printk("Intel PIIX4 ACPI bridge not found\n");
		return -ENODEV;
	}

	pcibios_read_config_dword(Bus, DevFn, 0x40, &PmBase);

	PmBase -= 1;	/* Strip off bit 0 */

	/* O.K. Now do the Linux stuff */
	if (register_chrdev(0, "cpcieject", &cpciejectFops) < 0) {
		printk("Cound not register the DTI system shutdown driver\n");
		return -EIO;
	}

	if (request_irq(9, cpciejectIntr, SA_SHIRQ, "cpcieject", (void *)&cpciejectFops)) {
		printk("DTI cpcieject: Interrupt installation failed\n");
		return -EIO;
	}

	UseCount = 0;

	register_reboot_notifier(&nb);
	
	printk("System shutdown driver: initialized\n");
	printk("Copyright MontaVista Software 2001\n");
	return 0;
}

static void
cpciejectExit(void)
{
	unregister_reboot_notifier(&nb);
}

static int
cpciejectOpen(struct inode *ip, struct file *fp)
{
	if (UseCount) {
		return -EPERM;
	}

	UseCount++;
	MOD_INC_USE_COUNT;
	
	return 0;
}

static int
cpciejectClose(struct inode *ip, struct file *fp)
{
	UseCount--;
	MOD_DEC_USE_COUNT;
	
	return 0;
}

static int
cpciejectFasync(int fd, struct file *fp, int on)
{
	if (on) {
		AsyncPid = current->pid;
		fasync_helper(fd, fp, on, &FasyncPtr);

		/* Initialize the interrupt. */
		outb(inb(PmBase + GPSTS), PmBase + GPSTS);	/* Clear GPI Status */
		outb(inb(PmBase + GPEN) | 0x2, PmBase + GPEN);	/* Enable GPI Event */
		outb(inb(PmBase + PMCNTRL)|1, PmBase + PMCNTRL);/* Enable SCI on GPI */
	} else {
		AsyncPid = current->pid;
		fasync_helper(fd, fp, on, &FasyncPtr);

		/* Initialize the interrupt. */
		outb(inb(PmBase + GPSTS), PmBase + GPSTS);	/* Clear GPI Status */
		outb(inb(PmBase + GPEN) & ~0x2, PmBase + GPEN);	/* Enable GPI Event */
		outb(inb(PmBase + PMCNTRL) & ~0x1, PmBase + PMCNTRL);/* Enable SCI on GPI */
	}

	return 0;
}

static void
cpciejectIntr(int irq, void *dev_id, struct pt_regs *regs)
{
	int status;

	if (!(status = inb(PmBase + GPSTS)) & 2) {
		printk("Bad status returned for int 9: 0x%x\n", status);
		return;
	}

	outb(status, PmBase + GPSTS);	/* Clear GPI Status */
	
	status = inb(PmBase + GPIREG);

	if (status & 0x1) {	/* GPI 16 clear? */
		return;
	}

	/* Disable the interrupt */
	outb(inb(PmBase + GPEN) & ~0x2, PmBase + GPEN);
	outb(inb(PmBase + PMCNTRL) & ~0x1, PmBase + PMCNTRL);

	kill_fasync(&FasyncPtr, SIGIO, POLL_IN);
	return;
}

static int
cpciejectBlueLight(struct notifier_block *self, unsigned long _, void *__)
{
	outb(inb(PmBase + GPOREG) & ~1, PmBase + GPOREG);	/* Clear GPI Status */
	
	return NOTIFY_OK;
}

module_init(cpciejectInit);
module_exit(cpciejectExit);
