/*
 * cowbell
 * Copyright (c) 2005 Brad Taylor
 *
 * This file is part of cowbell.
 *
 * cowbell is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * cowbell is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with cowbell; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

using System;
using Cowbell.Base;
using Cowbell.Gui;
using Gtk;

namespace Cowbell
{
	public class Runtime
	{
		public static MainWindow MainWindow {
			get { return mainwindow; }
		}
		
		public static SongDatabase Database {
			get { return database; }
		}

		public static Preferences Preferences {
			get { return preferences; }
		}

		public static bool IsWindows {
			get { return (int)Environment.OSVersion.Platform <= 3; }
		}

		public static bool Dirty;
		public static bool DebugEnabled;

		static CliParser parser;

		public static void Main (string[] args)
		{
			Catalog.Init ();

			parser = new CliParser (Globals.AppName,
						Catalog.GetString ("An elegant music organizer."),
						Catalog.GetString ("[OPTIONS]... [FILE]..."));

			parser.AddOption (new string[] {"h", "help"}, Catalog.GetString ("display this help and exit"), false);
			parser.AddOption (new string[] {"v", "version"}, Catalog.GetString ("display the current version and exit"), false);
			parser.AddOption (new string[] {"debug"}, Catalog.GetString ("enables debugging output"), false);
			parser.AddOption (new string[] {"batch"}, Catalog.GetString ("enables batch tagging mode"), false);
			
			try {
				parser.Parse (args);
			} catch (Exception e) {
				Console.WriteLine (e.Message);
				Console.WriteLine (parser.GetHelp ());
				Environment.Exit (1);
			}

			if (parser["help"] != null || parser["h"] != null) {
				Console.WriteLine (parser.GetHelp ());
				return;
			}

			if (parser["version"] != null || parser["v"] != null) {
				Console.WriteLine (Catalog.GetString ("{0}, version {1}"),
				                   Globals.AppName, Globals.AppVersion);
				return;
			}

			DebugEnabled = false;
			if (parser["debug"] != null) {
				Console.WriteLine ("Debug mode enabled.");
				DebugEnabled = true;
			}

			if (parser["batch"] != null) {
				new Runtime ();
				new Batch (parser.Parameters);
				return;
			}

			Runtime r = new Runtime ();
			Application.Init ();
			r.Show ();
		}

		public Runtime ()
		{
			Dirty = false;

			preferences = new Preferences ();
			database = new SongDatabase ();
		}

		public void Show ()
		{
			mainwindow = new MainWindow ();
			Application.Run ();
		}

		public static void Debug (int num)
		{
			if (DebugEnabled)
				Console.WriteLine (num);
		}
		
		public static void Debug (string str)
		{
			if (DebugEnabled)
				Console.WriteLine (str);
		}

		public static void Debug (object o)
		{
			if (DebugEnabled)
				Console.WriteLine (o.ToString ());
		}

		public static void Debug (string format, object arg)
		{
			if (DebugEnabled)
				Console.WriteLine (String.Format (format, arg));
		}

		public static void Debug (string format, params object[] args)
		{
			if (DebugEnabled)
				Console.WriteLine (String.Format (format, args));
		}

		/* private fields */
		private static Preferences preferences;
		private static SongDatabase database;
		private static MainWindow mainwindow;
	}
}
