/*
 * cowbell
 * Copyright (c) 2005 Brad Taylor
 *
 * This file is part of cowbell.
 *
 * cowbell is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * cowbell is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with cowbell; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

using System;
using System.IO;
using System.Collections;

namespace Cowbell.Base
{
	public class Filesystem
	{
		/* public methods */
		public static bool Import (string filename)
		{
			if (filename == "") {
				return false;
			}	

			Import (new string[] { filename });

			return true;
		}

		public static bool Import (string[] uris)
		{
			if (uris.Length < 1) {
				return false;
			}
			
			Runtime.Database.Clear ();

			foreach (string uri in uris)
			{
				string filename = GetLocalPathFromUri (uri);
				Song s = GetSongFromFile (filename);

				if (s == null) {
					continue;
				}

				Runtime.Database.Add (s);
			}

			Song globaldata = new Song ();
			if (Runtime.Database.Count > 0) {
				globaldata = (Song)Runtime.Database[0];
			}

			Runtime.Database.GlobalData = globaldata;
		
			// Is this album by multiple artists?
			bool finished = false;
			foreach (Song a in Runtime.Database)
			{
				foreach (Song b in Runtime.Database)
				{
					int dist = Utils.LevenshteinDistance (a.Artist.ToLower (), b.Artist.ToLower ());
					if (dist > 3) {
						Runtime.Database.MultipleArtists = true;
						finished = true;
						break;
					}
				}
			}

			if (!finished) {
				Runtime.Database.MultipleArtists = false;
			}

			return true;
		}

		public static bool Save ()
		{
			if (Runtime.Database.Count < 1) {
				return false;
			}

			foreach (Song s in Runtime.Database)
			{
				if (!Runtime.Database.MultipleArtists) {
					s.Artist = Runtime.Database.GlobalData.Artist;
				}

				if (!Runtime.Database.MultipleDiscs) {
					s.Album = Runtime.Database.GlobalData.Album;
				}

				s.Year = Runtime.Database.GlobalData.Year;
				s.Genre = Runtime.Database.GlobalData.Genre;

				s.Save ();
			}

			string path = GetBasedir ();
			if (Runtime.Preferences.CacheCoverImage) {
				Runtime.MainWindow.SaveCoverImage (Path.Combine (path, Catalog.GetString ("cover.jpg")));
			}

			if (Runtime.Preferences.RenameFiles) {
				Runtime.Database.RenameAll (Runtime.Preferences.RenameFilesPattern);
			}

			if (Runtime.Preferences.ExportPlaylist) {
				ExportPlaylist (path);
			}

			return true;
		}

		public static string GetBasedir ()
		{
			if (Runtime.Database.Count < 1) {
				return String.Empty;
			}

			Hashtable basedirs = new Hashtable ();
			foreach (Song s in Runtime.Database)
			{
				string basedir = Path.GetDirectoryName (s.Filename);
				if (basedirs.ContainsKey (basedir)) {
					int rank = (int)basedirs[basedir];
					basedirs[basedir] = rank++;
				} else {
					basedirs[basedir] = 1;
				}
			}

			string path = String.Empty;
			int count = 0;
			foreach (DictionaryEntry entry in basedirs)
			{
				if ((int)entry.Value > count) {
					path = (string)entry.Key;
				}
			}
			return path;
		}

		public static Song GetSongFromFile (string filename)
		{
			Song s;

			try {
				// Feed it right into the song class
				s = new Song (filename);

				if (s.Title == "") {
					s.PathToTitle (filename);
				}
			} catch (Exception e) {
				s = null;
			}

			return s;
		}

		public static Song[] GetSongsFromFilesystem (string filename)
		{
			Song[] songs = null;

			if (File.GetAttributes (filename) == FileAttributes.Directory) {
				string[] files = Directory.GetFiles (filename);
				string[] dirs = Directory.GetDirectories (filename);

				foreach (string file in files)
				{
					songs = ArrayJoin (songs, GetSongsFromFilesystem (file));
				}

				foreach (string dir in dirs)
				{
					songs = ArrayJoin (songs, GetSongsFromFilesystem (dir));
				}
			} else {
				try {
					// Feed it right into the song class
					songs = new Song[1];
					songs[0] = new Song (filename);

					if (songs[0].Title == "")
						songs[0].PathToTitle (filename);
				} catch (Exception e) {
					songs = null;
				}
			}

			return songs;
		}

		public static string GetLocalPathFromUri (string path)
		{
			Uri uri = new Uri (path);

			if (uri.Scheme == "file") {
				return uri.LocalPath;
			}

			return null;
		}

		public static void ExportPlaylist (string path)
		{
			if (Runtime.Preferences.PlaylistType == "pls") {
				ExportPls (Path.Combine (path, Catalog.GetString ("playlist.pls")));
			} else {
				ExportM3u (Path.Combine (path, Catalog.GetString ("playlist.m3u")));
			}
		}

		/* private methods */
		private static void ExportM3u (string filename)
		{
			ArrayList sorted = Runtime.Database;
			System.IO.StreamWriter stream = new StreamWriter (filename);

			sorted.Sort ();
			stream.Write ("#EXTM3U\n");

			foreach (Song s in sorted)
			{
				stream.Write ("#EXTINF:" + s.Length + "," + s.Artist + " - " + s.Title + "\n");
				stream.Write (s.Filename + "\n");
			}

			stream.Flush ();
			stream.Close ();
		}

		private static void ExportPls (string filename)
		{
			ArrayList sorted = Runtime.Database;
			System.IO.StreamWriter stream = new StreamWriter (filename);
	
			sorted.Sort ();	
			stream.Write ("[playlist]\n");
			stream.Write ("NumberOfEntries=" + Runtime.Database.Count + "\n");

			int i = 0;
			foreach (Song s in sorted)
			{
				stream.Write ("File" + i + "=" + s.Filename + "\n");
				i++;
			}

			stream.Flush ();
			stream.Close ();			
		}

		private static Song[] ArrayJoin (Song[] a, Song[] b)
		{
			if (a == null) {
				return b;
			} else if (b == null) {
				return a;
			}

			int i, k;

			Song[] songs = new Song[a.Length + b.Length];
			
			for (i = 0; i < a.Length; i++)
				songs[i] = a[i];
			
			for (k = 0; k < b.Length; i++, k++)
				songs[i] = b[k];

			return songs;
		}
	}
}
