# Copyright (c) 2005 Fredrik Kuivinen <freku045@student.liu.se>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License version 2 as
# published by the Free Software Foundation.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

import sys, os, re

from ctcore import *

def initialize():
    if not os.environ.has_key('GIT_DIR'):
        os.environ['GIT_DIR'] = '.git'

    if not os.environ.has_key('GIT_OBJECT_DIRECTORY'):
        os.environ['GIT_OBJECT_DIRECTORY'] = os.environ['GIT_DIR'] + '/objects'

    if not (os.path.exists(os.environ['GIT_DIR']) and
            os.path.exists(os.environ['GIT_DIR'] + '/refs') and
            os.path.exists(os.environ['GIT_OBJECT_DIRECTORY'])):
        print "Git archive not found."
        print "Make sure that the current working directory contains a '.git' directory, or\nthat GIT_DIR is set appropriately."
        sys.exit(1)

    files = runProgram(['git-diff-files', '--name-only', '-z']).split('\0')
    files.pop()
    runXargsStyle(['git-update-index', '--remove', '--'], files)
    
parseDiffRE = re.compile(':([0-9]+) ([0-9]+) ([0-9a-f]{40}) ([0-9a-f]{40}) ([MCRNADUT])([0-9]*)')
def parseDiff(prog):
    inp = runProgram(prog)
    ret = []
    try:
        recs = inp.split("\0")
        recs.pop() # remove last entry (which is '')
        it = recs.__iter__()
        while True:
            rec = it.next()
            m = parseDiffRE.match(rec)
            
            if not m:
                print "Unknown output from " + str(prog) + "!: " + rec + "\n"
                continue

            f = File()
            f.srcMode = m.group(1)
            f.dstMode = m.group(2)
            f.srcSHA = m.group(3)
            f.dstSHA = m.group(4)
            if m.group(5) == 'N':
                f.change = 'A'
            else:
                f.change = m.group(5)
            f.score = m.group(6)
            f.srcName = f.dstName = it.next()

            if f.change == 'C' or f.change == 'R':
                f.dstName = it.next()
                f.patch = getPatch(f.srcName, f.dstName)
            else:
                f.patch = getPatch(f.srcName)

            ret.append(f)
    except StopIteration:
        pass
    return ret

def runXargsStyle(origProg, args):
    steps = range(10, len(args), 10)
    prog = origProg[:]
    prev = 0
    for i in steps:
        prog.extend(args[prev:i])
        runProgram(prog)
        prog = origProg[:]
        prev = i

    prog.extend(args[prev:])
    runProgram(prog)

def getUnknownFiles():
     args = []

     if settings().gitExcludeFile():
         if os.path.exists(settings().gitExcludeFile()):
             args.append('--exclude-from=' + settings().gitExcludeFile())
     if settings().gitExcludeDir():
         args.append('--exclude-per-directory=' + settings().gitExcludeDir())

     inp = runProgram(['git-ls-files', '-z', '--others'] + args)
     files = inp.split("\0")
     files.pop() # remove last entry (which is '')
     
     fileObjects = []

     runXargsStyle(['git-update-index', '--add', '--'], files)
     for fileName in files:
         f = File()
         f.srcName = f.dstName = fileName
         f.change = '?'
         f.patch = runProgram(['git-diff-cache', '-p', '--cached', 'HEAD', '--', fileName])
         fileObjects.append(f)
         f.text = 'New file: ' + fileName

     runXargsStyle(['git-update-index', '--force-remove', '--'], files)
     return fileObjects

# HEAD is src in the returned File objects. That is, srcName is the
# name in HEAD and dstName is the name in the cache.
def getFiles():
    files = parseDiff('git-diff-files -z')
    for f in files:
        doUpdateCache(f.srcName)

    files = parseDiff('git-diff-cache -z -M --cached HEAD')
    for f in files:
        c = f.change
        if   c == 'C':
            f.text = 'Copy from ' + f.srcName + ' to ' + f.dstName
        elif c == 'R':
            f.text = 'Rename from ' + f.srcName + ' to ' + f.dstName
        elif c == 'A':
            f.text = 'New file: ' + f.srcName
        elif c == 'D':
            f.text = 'Deleted file: ' + f.srcName
        elif c == 'T':
            f.text = 'Type change: ' + f.srcName
        else:
            f.text = f.srcName

    if settings().showUnknown:
        files.extend(getUnknownFiles())

    return files

def getPatch(file, otherFile = None):
    if otherFile:
        f = [file, otherFile]
    else:
        f = [file]
    return runProgram(['git-diff-cache', '-p', '-M', '--cached', 'HEAD'] + f)

def doUpdateCache(filename):
    runProgram(['git-update-index', '--remove', '--add', '--replace', '--', filename])

def doCommit(filesToKeep, filesToCommit, msg):
    for file in filesToKeep:
        # If we have a new file in the cache which we do not want to
        # commit we have to remove it from the cache. We will add this
        # cache entry back in to the cache at the end of this
        # function.        
        if file.change == 'A':
            runProgram(['git-update-index', '--force-remove',
                        '--', file.srcName])
        elif file.change == 'R':
            runProgram(['git-update-index', '--force-remove',
                        '--', file.dstName])
            runProgram(['git-update-index', '--add', '--replace',
                        '--cacheinfo', file.srcMode, file.srcSHA, file.srcName])
        elif file.change == '?':
            pass
        else:
            runProgram(['git-update-index', '--add', '--replace',
                        '--cacheinfo', file.srcMode, file.srcSHA, file.srcName])

    for file in filesToCommit:
        if file.change == '?':
            runProgram(['git-update-index', '--add', '--', file.dstName])

    tree = runProgram(['git-write-tree'])
    tree = tree.rstrip()

    if commitIsMerge():
        merge = ['-p', 'MERGE_HEAD']
    else:
        merge = []
    commit = runProgram(['git-commit-tree', tree, '-p', 'HEAD'] + merge, msg).rstrip()
    
    runProgram(['git-update-ref', 'HEAD', commit])

    try:
        os.unlink(os.environ['GIT_DIR'] + '/MERGE_HEAD')
    except OSError:
        pass

    for file in filesToKeep:
        # Don't add files that are going to be deleted back to the cache
        if file.change != 'D' and file.change != '?':
            runProgram(['git-update-index', '--add', '--replace', '--cacheinfo',
                        file.dstMode, file.dstSHA, file.dstName])

        if file.change == 'R':
            runProgram(['git-update-index', '--remove', '--', file.srcName])

def discardFile(file):
    runProgram(['git-read-tree', 'HEAD'])
    c = file.change
    if c == 'M' or c == 'T':
        runProgram(['git-checkout-cache', '-f', '-q', '--', file.dstName])
    elif c == 'A' or c == 'C':
        # The file won't be tracked by git now. We could unlink it
        # from the working directory, but that seems a little bit
        # too dangerous.
        pass 
    elif c == 'D':
        runProgram(['git-checkout-cache', '-f', '-q', '--', file.dstName])
    elif c == 'R':
        # Same comment applies here as to the 'A' or 'C' case.
        runProgram(['git-checkout-cache', '-f', '-q', '--', file.srcName])

def ignoreFile(file):
    ignoreExpr = re.sub(r'([][*?!\\])', r'\\\1', file.dstName)
    
    excludefile = settings().gitExcludeFile()
    excludefiledir = os.path.dirname(excludefile)
    if not os.path.exists(excludefiledir):
        os.mkdir(excludefiledir)
    if not os.path.isdir(excludefiledir):
        return
    exclude = open(excludefile, 'a')
    print >> exclude, ignoreExpr
    exclude.close()

    pass

def commitIsMerge():
    try:
        os.stat(os.environ['GIT_DIR'] + '/MERGE_HEAD')
        return True
    except OSError:
        return False

def mergeMessage():
    return '''This is a merge commit if you do not want to commit a ''' + \
           '''merge remove the file $GIT_DIR/MERGE_HEAD.'''
