# Cohoba - a GNOME client for Telepathy
#
# Copyright (C) 2006 Collabora Limited
#
# This package is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License as
# published by the Free Software Foundation; either
# version 2 of the License, or (at your option) any later version.
#
# This package is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with this package; if not, write to the Free Software
# Foundation, 51 Franklin Street, Fifth Floor, Boston, MA, 02110-1301 USA.

import gobject, dbus

from telepathy.constants import CONNECTION_HANDLE_TYPE_CONTACT
from telepathy.interfaces import (CONN_INTERFACE, CONN_INTERFACE_PRESENCE,
	CONN_INTERFACE_AVATARS, CONN_INTERFACE_ALIASING, CHANNEL_TYPE_TEXT)

from cohoba.common.ui.PresenceIcons import get_presence_icons
from cohoba.common.Utils import global_error_handler, presence_status_to_string, presence_status_message, SignalConnection
from cohoba.common.TelepathyConstants import (MISSION_CONTROL_INTERFACE,
	MISSION_CONTROL_OBJECT_PATH)
from cohoba.common.TelepathyGroupMember import TelepathyGroupMember
from cohoba.common.DBusProxyProvider import get_proxy_provider
from cohoba.common.Store import ContactStore
from cohoba.contacts.AbstractContact import AbstractContact


class Contact(TelepathyGroupMember, AbstractContact):
	def __init__(self, conn_path, handle, serialize=True):
		TelepathyGroupMember.__init__(self, handle)
		AbstractContact.__init__(self)
		
		# Serialized data
		self.avatar = (None, None)
		self.groups = []
		self.favorite = False
		
		# Static data
		self.handle = handle
		self.conn_path = conn_path
		self.name = None
		self.alias = None
		self.status = ('offline', None)
		
		# Never Changes
		self.is_group = False
		self.do_serialize = serialize
				
		self.connect_to_connection()

	def get_icon(self):
		return get_presence_icons(self.status[0], self.avatar[1])[1]

	def action(self):
		mission_control_iface = get_proxy_provider().get_iface(
			MISSION_CONTROL_OBJECT_PATH,
			MISSION_CONTROL_INTERFACE)
		mission_control_iface.RequestChannel(
			get_proxy_provider().get_service(self.conn_path),
			self.conn_path,
			CHANNEL_TYPE_TEXT,
			CONNECTION_HANDLE_TYPE_CONTACT,
			self.handle,
			reply_handler=lambda: None,
			error_handler=global_error_handler)

	def set_avatar(self, token, avatar):
		self.avatar = (token, avatar)
		self.save()
				
	def save(self):
		if self.name != None and self.do_serialize:
			ContactStore().serialize(self.conn_path, self.name, self)
		self.emit("updated")
		
	def serialize(self):
		return (self.avatar, self.groups, self.favorite)
	
	def unserialize(self, data):
		self.avatar, self.groups, self.favorite = data
		print 'Unserialized:', self.avatar, self.groups, self.favorite
		self.emit("updated")
	
	def connect_to_connection(self):
		conn_iface = get_proxy_provider().get_iface(self.conn_path, CONN_INTERFACE)
		conn_iface.InspectHandles(CONNECTION_HANDLE_TYPE_CONTACT, [self.handle], reply_handler=self.on_handle_inspected, error_handler=global_error_handler)

		self.presence_iface = get_proxy_provider().get_iface(self.conn_path, CONN_INTERFACE_PRESENCE)
		self.__PresenceUpdate_conn = SignalConnection(self.presence_iface, 'PresenceUpdate', self.on_got_presence)
		self.presence_iface.RequestPresence([dbus.UInt32(self.handle)], reply_handler=lambda: None, error_handler=global_error_handler)
		
		self.alias_iface = get_proxy_provider().get_iface(self.conn_path, CONN_INTERFACE_ALIASING)
		self.__AliasesChanged_conn = SignalConnection(self.alias_iface, 'AliasesChanged', self.on_got_aliases)
		self.alias_iface.RequestAliases([dbus.UInt32(self.handle)], reply_handler=self.on_got_alias, error_handler=global_error_handler)	

		#self.cinfo_iface = get_proxy_provider().get_iface(self.conn_path, 'org.freedesktop.Telepathy.Connection.Interface.ContactInfo')
		#self.cinfo_iface.connect_to_signal('GotContactInfo', self.on_got_contact_infos)
		#self.cinfo_iface.RequestContactInfo(dbus.UInt32(self.handle))
		
		self.avatar_iface = get_proxy_provider().get_iface(self.conn_path, CONN_INTERFACE_AVATARS)
		self.__AvatarUpdated__con = SignalConnection(self.avatar_iface, 'AvatarUpdated', self.on_avatar_updated)
		# FIXME: Experimental support for avatars, uncomment the following line
		#self.on_got_avatar_token(["FIXME:token"])
		#self.avatar_iface.GetAvatarTokens([dbus.UInt32(self.handle)], reply_handler=self.on_got_avatar_token, error_handler=global_error_handler)
		
	def set_alias(self, alias):
		self.alias = alias
		self.emit('updated')
	
	def set_status(self, code, message):
		self.status = (code, message)
		self.emit('updated')
		
	def on_handle_inspected(self, names):
		assert len(names) == 1
		name = names[0]
		# Read saved data
		ContactStore().unserialize(self.conn_path, name, self)
		print 'Unserialized:', self.name, self.groups

		self.set_name(name)

	def on_got_presence(self, presences):
		if self.handle not in presences:
			return
		
		idle, statuses = presences[self.handle]

		for name, params in statuses.items():
			self.set_status(name, presence_status_message(params))
			break

	def on_got_alias(self, aliases):
		self.set_alias(aliases[0])

	def on_got_aliases(self, aliases):
		for handle, alias in aliases:
			if handle == self.handle:
				self.set_alias(alias)
	
	def on_got_avatar_token(self, tokens):
		if tokens[0] != "":
			self.avatar_iface.RequestAvatar(dbus.UInt32(self.handle), reply_handler=lambda data, mime: self.on_got_avatar(tokens[0], data, mime), error_handler=global_error_handler)
			
	def on_got_avatar(self, token, data, mime):
		self.set_avatar(token, ''.join(chr(c) for c in data))
		
	def on_avatar_updated(self, handle, token):
		if handle != self.handle:
			return
		
		self.on_got_avatar_token([token])
	
	def __repr__(self):
		return 'Contact %d: %s/%s (%s): %s' % (self.handle, self.name, self.alias, self.status, self.flag)

